/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service.print;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.net.URI;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JRPrintElement;
import net.sf.jasperreports.engine.JRPrintPage;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperPrintManager;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.fill.JRTemplatePrintFrame;
import net.sf.jasperreports.engine.query.JRXPathQueryExecuterFactory;
import net.sf.jasperreports.engine.util.JRLoader;
import net.sf.jasperreports.engine.util.JRXmlUtils;
import no.g9.exception.G9BaseException;
import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageDispatcher;
import no.g9.message.MessageSystem;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Print and export using JasperReports.
 */
public class JasperReportsPrint implements G9Print {

    /** Default Locale for xml parsing. */
    protected static final Locale DEFAULT_XML_LOCALE = Locale.getDefault();

    /** Default Locale for reports. */
    protected static final Locale DEFAULT_REPORT_LOCALE = Locale.getDefault();

    /** Type for the compiled jasper files. */
    protected static final String JASPER_SUFFIX = ".jasper";

    @Override
    public boolean printReport(Document data, Map<String, Object> params,
            String dialogName, String printerName) {
        JasperPrint print = fillReport(data, params, dialogName);
        try {
            if (printerName == null) {
                return JasperPrintManager.printReport(print, true);
            }
            JasperExplicitPrinterPrintManager.print(print, printerName);
        } catch (Exception e) {
            Message message = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_PRINTING_REPORT_FAILED);
            getMessageDispatcher().dispatch(message);
            throw new G9ServiceException(e, message);
        }
        return true;
    }

    @Override
    public boolean printReport(URI data, Map<String, Object> params,
            String dialogName, String printerName) {
        return printReport(getDocumentFromURI(data), params, dialogName,
                printerName);
    }

    @Override
    public boolean printReport(String data, Map<String, Object> params,
            String dialogName, String printerName) {
        return printReport(getDocumentFromXML(data), params, dialogName,
                printerName);
    }

    /**
     * Fills the report.
     *
     * @param data
     *            Data as document
     * @param params
     *            Input parameters
     * @param dialogName
     *            Dialog name
     * @return JasperPrint
     */
    protected JasperPrint fillReport(Document data, Map<String, Object> params,
            String dialogName) {
        if (params == null) {
            params = new HashMap<String, Object>();
        }
        /**
         * TODO: Jasper currently (ver. 2.0.4) fails if locale is not explicitly
         * set. The following two ifs, along with their static finals, should be
         * removed if/when Jasper handles locale properly.
         */
        if (!params.containsKey(JRXPathQueryExecuterFactory.XML_LOCALE)) {
            params.put(JRXPathQueryExecuterFactory.XML_LOCALE,
                    DEFAULT_XML_LOCALE);
        }
        if (!params.containsKey(JRParameter.REPORT_LOCALE)) {
            params.put(JRParameter.REPORT_LOCALE, DEFAULT_REPORT_LOCALE);
        }
        params.put(JRXPathQueryExecuterFactory.PARAMETER_XML_DATA_DOCUMENT,
                data);
        try {
            long s = System.currentTimeMillis();
            JasperPrint print = JasperFillManager.fillReport(
                    getReport(dialogName), params);
            removeBlankPage(print);
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_FILLED_TIME,
                    new Long(System.currentTimeMillis() - s));
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            return print;
        } catch (Exception e) {
            getMessageDispatcher().dispatch(
                    CRuntimeMsg.PM_FILLING_REPORT_FAILED,e);
            throw new G9ServiceException(e);
        }
    }

    /**
     * Retrieves the print template for the dialog name.
     *
     * @param dialogName
     *            Name of the dialog
     * @return JasperReport
     */
    protected JasperReport getReport(String dialogName) {
        ClassLoader cl = JasperReportsPrint.class.getClassLoader();
        InputStream in = cl.getResourceAsStream(dialogName + JASPER_SUFFIX);
        if (in == null) { // .jasper file not in class path
            throw new G9BaseException(
                    "Unable to load compiled report template file \""
                            + dialogName + JASPER_SUFFIX
                            + "\" (add this file to class path)");
        }
        JasperReport jasperReport = null;
        try {
            jasperReport = (JasperReport) JRLoader.loadObject(in);
        } catch (JRException e) {
            throw new G9BaseException(
                    "Error loading report template class from file \""
                            + dialogName + JASPER_SUFFIX + "\"", e);
        }
        return jasperReport;
    }

    @Override
    public void exportPDF(Document data, String dialogName, File file,
            OutputStream os, Map<String, Object> params) {
        JasperPrint print = fillReport(data, params, dialogName);
        if (os == null && file == null) {
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_MISSING_FILE_OR_STREAM);
            throw new G9ServiceException(msg);
        }
        if (os == null) {
            try {
                /*
                 * This call should be replaced with
                 * JasperExprtManager.exportReportToPdfFile(JasperPrint, File)
                 * when this is supported.
                 */
                G9ExportManager.exportReportToPdfFile(print, file);
            } catch (Exception e) {
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.PM_PDF_EXPORT_FAILED);
                msg.setException(e);
                throw new G9ServiceException(e, msg);
            }
        } else {
            try {
                JasperExportManager.exportReportToPdfStream(print, os);
            } catch (Exception e) {
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.PM_EXPORT_STREAM_FAILED);
                msg.setException(e);
                throw new G9ServiceException(e, msg);
            }
        }
    }

    /**
     * Retrieves a document based on a URI.
     *
     * @param data
     *            URI reference
     * @return Document
     */
    protected Document getDocumentFromURI(URI data) {
        try {
            return JRXmlUtils.parse(data.getRawPath());
        } catch (Exception e) {
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_CONVERSION_FAILED);
            msg.setException(e);
            throw new G9ServiceException(msg);
        }
    }

    /**
     * Retrieves a document based on an input string.
     *
     * @param data
     *            Data formed as an XML
     * @return Document formed from the XML
     */
    protected Document getDocumentFromXML(String data) {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory
                    .newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Reader reader = new StringReader(data);
            return builder.parse(new org.xml.sax.InputSource(reader));
        } catch (FactoryConfigurationError e) {
            throw new G9ServiceException(e);
        } catch (ParserConfigurationException e) {
            throw new G9ServiceException(e);
        } catch (SAXException e) {
            throw new G9ServiceException(e);
        } catch (IOException e) {
            throw new G9ServiceException(e);
        }
    }

    /**
     * @return MessageDispatcher
     */
    protected MessageDispatcher getMessageDispatcher() {
        return MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION);
    }

    private void removeBlankPage(JasperPrint print) {
        List<JRPrintPage> pages = print.getPages();
        for (Iterator<JRPrintPage> i=pages.iterator(); i.hasNext();) {
            JRPrintPage page = i.next();
            List<JRPrintElement> elements = page.getElements();
            boolean has=false;
            for (JRPrintElement el : elements) {
				if(el instanceof JRTemplatePrintFrame && el.getHeight()>0){
					has=true;
					break;
				}
			}
			if (!has)
                i.remove();
        }
    }
    
}
