/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.service.print;

import java.io.File;
import java.io.OutputStream;
import java.util.Map;

import no.esito.util.ServiceLoader;
import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.MessageSystem;

import org.w3c.dom.Document;

/**
 * The Print Central is responsible for delegating print and export to classes
 * implementing the G9Print interface. Currently, the default implementation
 * uses JasperReports to perform both printing and export to pdf.
 */
public class PrintCentral implements PrintService, ExportService {

    /**
     * The G9Print delegate, performing print and export.
     */
    // Implementation note: When the default Spring config is used, this
    // property is set using Spring's IOC, in the SercvicesConfiguration.xml
    // bean definition file.
    private G9Print g9Print;


    /**
     * Get the g9Print instance that is used to perform print and export.
     *
     * @return the g9Print instance
     */
    public G9Print getG9Print() {
        // Lazy loading - see impl note on the field.
        if (g9Print == null) {
            setG9Print(ServiceLoader.getService(G9Print.class));
        }
        return g9Print;
    }

    /**
     * Set the G9Print instance that will be used for performing print and
     * export.
     *
     * @param g9Print the instance to set
     */
    public void setG9Print(G9Print g9Print) {
        // See field comment
        this.g9Print = g9Print;
    }

    /**
     * Print according to the given PrintContext. The print priority is
     * Document, URI, String.
     *
     * @param pCtx the print context
     */
    @Override
    public Object print(final PrintContext pCtx) {
        if (!pCtx.isValid()) {
            return new G9ServiceException(MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.PM_INVALID_PRINT_CONTEXT));
        }
        if (null != pCtx.getDocument()) {
            getG9Print().printReport(pCtx.getDocument(),
                                         pCtx.getParams(),
                                         pCtx.getDialogName(),
                                         pCtx.getPrinterName());
        }
        else if (null != pCtx.getFile()) {
            getG9Print().printReport(pCtx.getFile(),
                                         pCtx.getParams(),
                                         pCtx.getDialogName(),
                                         pCtx.getPrinterName());
        }
        else if (null != pCtx.getXml()) {
            getG9Print().printReport(pCtx.getXml(),
                                         pCtx.getParams(),
                                         pCtx.getDialogName(),
                                         pCtx.getPrinterName());
        }
        return pCtx;
    }

    @Override
    public void exportToPDF(Document data, String dialogName, File file) {
        getG9Print().exportPDF(data, dialogName, file, null, null);
    }

    @Override
    public void exportToPDF(Document data, String dialogName,
            OutputStream outputStream) {
        getG9Print().exportPDF(data, dialogName, null, outputStream, null);
    }

    @Override
    public void exportToPDF(Document data, String dialogName, File file,
            Map<String, Object> params) {
        getG9Print().exportPDF(data, dialogName, file, null, params);
    }

    @Override
    public void exportToPDF(Document data, String dialogName,
            OutputStream outputStream, Map<String, Object> params) {
        getG9Print().exportPDF(data, dialogName, null, outputStream,
                params);
    }

}
