/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Instances of the class holds data for controlling Find/FindAll actions
 */
public class FindData implements Serializable {

    private static int defaultMaxResults = 0;

    /**
     * Holds the chosen find method.
     */
    private EFindMethod findMethod = EFindMethod.DEFAULT;

    /**
     * Holds a list of criteria data for the CRITERIA findMethod.
     */
    private List<Object> criterions;

    /**
     * Holds a map from association "path" to alias name used in criterions for
     * the CRITERIA findMethod.
     */
    private Map<String, String> criterionAliases = new HashMap<>();

    /**
     * Holds a map from association "path" to join type used in criterions for
     * the CRITERIA findMethod.
     */
    private Map<String, Object> criterionJoinTypes = new HashMap<>();
    
    /**
     * Holds a map from association "path" to withClause used in criterions for
     * the CRITERIA findMethod.
     */
    private Map<String, Object> criterionWithClauses = new HashMap<>();
    
    /**
     * Holds a projection for the CRITERIA findMethod.
     */
    private Object criteriaProjection;

    /**
     * Holds the query string for the QUERY findMethod.
     */
    private String query;

    /**
     * Holds the domain class for EXAMPLE, CRITERIA and DEFAULT findMethods
     */
    private Class<?> domainClass;

    /**
     * Holds the key (id) for the DEFAULT findMethod
     */
    private Serializable key;

    /**
     * Holds the example object for the EXAMPLE findMethod
     */
    private Object example;

    /**
     * Holds the query parameters for the QUERY findMethod.
     */
    private List<Object> queryParameters;

    /**
     * Can be used for data transfer for the <code>CUSTOM</code> find method.
     */
    private Object customData;

    /**
     * Holds the "order by" data.
     */
    private List<FindOrder> order;

    /**
     * Maximum number of rows returned by the FindAll action. Use a value of
     * zero to avoid limiting the number of rows.
     */
    private int maxResults = defaultMaxResults;

    /**
     * First row to be returned by the FindAll action. Use a value of zero to
     * start with the first row.
     */
    private int firstResult;

    /**
     * Find all related instances according to the object selection for the
     * FindAll action. The default is to find only the target node and uprelated
     * instances.
     */
    private boolean deepFindAll = false;

    /**
     * If false, "old" style HQL positional parameters are used. HQL positional
     * parameters are marked with "?" in the query. HQL positional parameters
     * are deprecated from Hibernate 4.
     *
     * JPA positional parameters are marked with "?n" in the query, where n is a number
     * starting from 1.
     */
    private boolean useJpaPositionalParameters = true;

    /**
     * Add a new query parameter. The query parameters are used as "positional"
     * parameters, i.e. the ordering must match the query string.
     *
     * @param parameter (missing javadoc)
     */
    public void addQueryParameter(Object parameter) {
        if (getQueryParameters() == null) {
            setQueryParameters(new LinkedList<Object>());
        }
        getQueryParameters().add(parameter);
    }

    /**
     * Add a new criterion alias for the given association path.
     *
     * @param associationPath the association path to add an alias for
     * @param alias the alias to use
     */
    public void addCriterionAlias(String associationPath, String alias) {
        criterionAliases.put(associationPath, alias);
    }

    /**
     * Add a new criterion alias and join type for the given association path.
     *
     * @param associationPath the association path to add an alias for
     * @param alias the alias to use
     * @param joinType the join type to use
     */
    public void addCriterionAlias(String associationPath, String alias, Object joinType) {
        addCriterionAlias(associationPath, alias);
        criterionJoinTypes.put(associationPath, joinType);
    }

    /**
     * Add a new criterion alias and join type for the given association path.
     *
     * @param associationPath the association path to add an alias for
     * @param alias the alias to use
     * @param joinType the join type to use
     * @param withClause the withClause to use
     */
    public void addCriterionAlias(String associationPath, String alias, Object joinType, Object withClause) {
        addCriterionAlias(associationPath, alias, joinType);
        criterionWithClauses.put(associationPath, withClause);
    }
    
    /**
     * Add a new FindOrder object to the list used to generate "order by" from
     * the database.
     *
     * @param findOrder (missing javadoc)
     */
    public void addOrder(FindOrder findOrder) {
        if (getOrder() == null) {
            setOrder(new LinkedList<FindOrder>());
        }
        getOrder().add(findOrder);
    }

    /**
     * @return the current query.
     */
    public String getQuery() {
        return query;
    }

    /**
     * Set the query string for the QUERY findMethod. The syntax of the query
     * string is persistence layer specific.
     *
     * @param query The query to set.
     */
    public void setQuery(String query) {
        this.query = query;
    }

    /**
     * @return the first criterion.
     */
    public Object getCriterion() {
        if ( getCriterions().isEmpty() ) {
            return null;
        }
        return getCriterions().get(0);
    }

    /**
     * Clears the list and sets the first criterion data for the CRITERIA
     * findMethod. This is a persistence layer specific tree of criterions.
     *
     * @param criterion The criterion to set.
     */
    public void setCriterion(Object criterion) {
        getCriterions().clear();
        getCriterions().add(criterion);
    }

    /**
     * Return the criterions list. A new list is created if it is null.
     *
     * @return the criterions list.
     */
    public List<Object> getCriterions() {
        if ( criterions == null ) {
            setCriterions(new LinkedList<Object>());
        }
        return criterions;
    }

    /**
     * @param criterions the criterions to set
     */
    public void setCriterions(List<Object> criterions) {
        this.criterions = criterions;
    }

    /**
     * Add a new criterion to the criterions list.
     *
     * @param criterion the criterion to add.
     */
    public void addCriterion(Object criterion) {
        getCriterions().add(criterion);
    }

    /**
     * @return the findMethod.
     */
    public EFindMethod getFindMethod() {
        return findMethod;
    }

    /**
     * Controls the method used by the persistence layer for finding instances
     * in the database.
     *
     * Legal values are:
     *   DEFAULT - use key properties from the data object.
     *   QUERY - use a query string with optionally queryParameters
     *   EXAMPLE - use properties from the data object for "query by example"
     *   CRITERIA - use a tree of query criterions
     *   CUSTOM - the application programmer is responsible for the find
     *
     * Default value is DEFAULT.
     *
     * @param findMethod The findMethod to set.
     */
    public void setFindMethod(EFindMethod findMethod) {
        if ( findMethod == null) {
            this.findMethod= EFindMethod.DEFAULT;
        }
        else {
            this.findMethod = findMethod;
        }
    }

    /**
     * @return the firstResult.
     */
    public int getFirstResult() {
        return firstResult;
    }

    /**
     * First row to be returned by the FindAll action. Use a value of zero
     * (default) to start with the first row.
     *
     * @param firstResult The firstResult to set. Must be &gt;= 0.
     */
    public void setFirstResult(int firstResult) {
        this.firstResult = firstResult;
    }

    /**
     * @return the maxResults.
     */
    public int getMaxResults() {
        return maxResults;
    }

    /**
     * Maximum number of rows returned by the FindAll action. Use a value of
     * zero (default) to avoid limiting the number of rows.
     *
     * @param maxResults The maxResults to set. Must be &gt;= 0.
     */
    public void setMaxResults(int maxResults) {
        this.maxResults = maxResults;
    }

    /**
     * @return the queryParameters.
     */
    public List<Object> getQueryParameters() {
        return queryParameters;
    }

    /**
     * @param queryParameters The queryParameters to set.
     */
    public void setQueryParameters(List<Object> queryParameters) {
        this.queryParameters = queryParameters;
    }

    /**
     * @return the customData
     */
    public Object getCustomData() {
        return customData;
    }

    /**
     * Set the customData object. This can be used for data
     * transfer for the <code>CUSTOM</code> find method.
     *
     * @param customData the customData to set
     */
    public void setCustomData(Object customData) {
        this.customData = customData;
    }

    /**
     * @return the order.
     */
    public List<FindOrder> getOrder() {
        return order;
    }

    /**
     * @param order The order to set.
     */
    public void setOrder(List<FindOrder> order) {
        this.order = order;
    }

    /**
     * @return the defaultMaxResults.
     */
    public static int getDefaultMaxResults() {
        return defaultMaxResults;
    }

    /**
     * @param defaultMaxResults The defaultMaxResults to set.
     */
    public static void setDefaultMaxResults(int defaultMaxResults) {
        FindData.defaultMaxResults = defaultMaxResults;
    }

    /**
     * @return the domainClass.
     */
    public Class<?> getDomainClass() {
        return domainClass;
    }

    /**
     * Set the domain class for CRITERIA and DEFAULT findMethods. This is the
     * domain class used in Find/FindAll
     *
     * @param domainClass The domainClass to set.
     */
    public void setDomainClass(Class<?> domainClass) {
        this.domainClass = domainClass;
    }

    /**
     * @return the example.
     */
    public Object getExample() {
        return example;
    }

    /**
     * Set the example object for the EXAMPLE findMethod. The example object is
     * used for "query by example".
     *
     * @param example The example to set.
     */
    public void setExample(Object example) {
        this.example = example;
    }

    /**
     * @return the key.
     */
    public Serializable getKey() {
        return key;
    }

    /**
     * Set the key (database primary key) for the DEFAULT findMethod.
     *
     * @param key The key to set.
     */
    public void setKey(Serializable key) {
        this.key = key;
    }

    /**
     * @return the criterion aliases map.
     */
    public Map<String, String> getCriterionAliases() {
        return Collections.unmodifiableMap(criterionAliases);
    }

    /**
     * @return the criterion join types map.
     */
    public Map<String, Object> getCriterionJoinTypes() {
        return Collections.unmodifiableMap(criterionJoinTypes);
    }
    
    /**
     * @return the criterion with clauses map.
     */
    public Map<String, Object> getCriterionWithClauses() {
        return Collections.unmodifiableMap(criterionWithClauses);
    }
    
    /**
     * @return the deepFindAll flag.
     */
    public boolean getDeepFindAll() {
        return this.deepFindAll;
    }

    /**
     * Set the deepFindAll flag.
     *
     * @param deepFindAll (missing javadoc)
     */
    public void setDeepFindAll(final boolean deepFindAll) {
        this.deepFindAll = deepFindAll;
    }

    /**
     * @return the criteriaProjection
     */
    public Object getCriteriaProjection() {
        return criteriaProjection;
    }

    /**
     * @param criteriaProjection the criteriaProjection to set
     */
    public void setCriteriaProjection(Object criteriaProjection) {
        this.criteriaProjection = criteriaProjection;
    }

    /**
     * If false, "old" style HQL positional parameters are used. HQL positional
     * parameters are marked with "?" in the query. HQL positional parameters
     * are deprecated from Hibernate 4.
     *
     * JPA positional parameters are marked with "?n" in the query, where n is a number
     * starting from 1.
     *
     * @return true if JPS style positional parameters should be used in HQL queries
     */
	public boolean getUseJpaPositionalParameters() {
		return useJpaPositionalParameters;
	}

    /**
     * If false, "old" style HQL positional parameters are used. HQL positional
     * parameters are marked with "?" in the query. HQL positional parameters
     * are deprecated from Hibernate 4.
     *
     * JPA positional parameters are marked with "?n" in the query, where n is a number
     * starting from 1.
     *
     * The default is to use JPA style positional parameters.
     *
     * @param useJpaPositionalParameters true if JPA style positional parameters should be used in HQL queries
     */
	public void setUseJpaPositionalParameters(boolean useJpaPositionalParameters) {
		this.useJpaPositionalParameters = useJpaPositionalParameters;
	}
}
