/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Instances of the class holds data for one object selection and points out the
 * root of interest and the role to operate on.
 *
 * Object selection data are always represented by a object tree starting at the
 * (sub)root object, according to the object selection model. Methods to get and
 * put object selection data are available for both single domain objects and
 * lists of domain objects.
 */
public class ObjectSelection implements Serializable {

    /**
     * The value of the attribute points to the current root in the roots
     * attribute. The value of the attribute corresponds to the key value of the
     * root of interest.
     */
    private String currentRootName;

    /**
     * The attribute points out the role in the current root in the object
     * selection that should be addressed.
     */
    private String targetRoleName;

    /**
     * The main key used to identify the target role instance. Primary key is
     * used if not set.
     */
    private String targetMainKey;

    /**
     * The attribute holds a Map of root-lists within the object selection. The
     * keys of the roots are their role names in the object selection.
     */
    private Map<String, List<Object>> roots =
            new HashMap<String, List<Object>>();

    /**
     * The attribute holds the name of the object selection.
     */
    private String objectSelectionName;

    /**
     * The attribute holds the name of the application.
     */
    private String applicationName;

    /**
     * The attribute holds detailed data for Find/FindAll actions.
     */
    private FindData findData;

    /**
     * Flag used between client and service to indicate the content of data in
     * many associations. Used in action Save, Update or Insert.
     *
     * true: All instances of objects reached through many associations are
     * present. Instances in the database not present in the object selection
     * will be deleted/disconnected from its parent.
     *
     * false: Only instances to be saved/updated/inserted are present, i.e. the
     * object selection does not contain the full set of objects. Instances in
     * the database not present in the object selection will not be touched by
     * the action.
     */
    private boolean setsContainAllRelated = false;

    /**
     * Map holding action code for each instance of the target node in the
     * object selection. Used in a Save action
     *
     * null: Do an ordinary save action, i.e. do save for all instances of the
     * target.
     *
     * != null: A map holding the desired action for each of the target
     * instances. Possible actions are Save, Insert, Update, Delete If an
     * instance is not present in the map a save action will be executed.
     *
     */
    private Map<Object, ActionType> actions = null;

    /**
     * Flag used to tell service if parent nodes and its up related nodes should
     * be saved. Used in action Save, Update, Insert or Delete
     */
    private boolean doParents = true;

    /**
     * Flag used to tell service how to treat one related nodes. In action Save,
     * Update and Insert it specifies that one related nodes should be saved
     * too. In action Find and FindAll it specifies that one related nodes
     * should be returned from JGrapeService.
     */
    private boolean doOneRelated = true;

    /**
     * Flag used to tell service how to treat 1-th level of many related nodes.
     * In action Save, Update and Insert it specifies that 1-th level of many
     * related nodes should be saved too. In action Find and FindAll it
     * specifies that 1-th level of many related nodes should be returned from
     * JGrapeService. This flag is ignored if the flag doManyRelated is true
     */
    private boolean doFirstLevelManyRelated = true;

    /**
     * Flag used to tell service how to treat any many related nodes. In action
     * Save, Update and Insert it specifies that related nodes should be saved
     * too. In action Find and FindAll it specifies that many related nodes
     * should be returned from JGrapeService.
     */
    private boolean doAnyLevelManyRelated = true;

    /**
     * The constructor to use when instantiating a completed ObjectSelection
     * object.
     *
     * @param currentRoot The current root role object
     * @param currentRootName The name of the current root role
     * @param targetRoleName The name of the target role for the action
     * @param applicationName The name of the application
     * @param objectSelectionName The name of the object selection
     */
    public ObjectSelection(Object currentRoot, String currentRootName,
            String targetRoleName, String applicationName,
            String objectSelectionName) {
        this(currentRootName, targetRoleName, applicationName,
                objectSelectionName);
        this.putRootObject(currentRootName, currentRoot);
    }

    /**
     * The constructor to use when instantiating a partially completed
     * ObjectSelection object.
     *
     * @param currentRootName The name of the current root role
     * @param targetRoleName The name of the target role for the action
     * @param applicationName The name of the application
     * @param objectSelectionName The name of the object selection
     */
    public ObjectSelection(String currentRootName, String targetRoleName,
            String applicationName, String objectSelectionName) {
        this.currentRootName = currentRootName;
        this.targetRoleName = targetRoleName;
        this.applicationName = applicationName;
        this.objectSelectionName = objectSelectionName;
    }

    /**
     * Create an empty ObjectSelection.
     */
    public ObjectSelection() {
        // Empty.
    }

    /**
     * Remove all entered root objects
     */
    public void removeAllRootObjects() {
        roots.clear();
    }

    /**
     * Put a domain object for the given root role name.
     *
     * @param rootName The name of the root role for the domain object, or
     *            <code>currentRootName</code> if null
     * @param rootObject The domain object
     */
    public void putRootObject(String rootName, Object rootObject) {
        List<Object> rootObjects = new LinkedList<Object>();
        rootObjects.add(rootObject);
        putRootObjects(rootName, rootObjects);
    }

    /**
     * Put a list of domain objects for the given root role name.
     *
     * @param rootName The name of the root role for the domain objects, or
     *            <code>currentRootName</code> if null
     * @param rootObjects The list of domain objects
     */
    public void putRootObjects(String rootName, List<Object> rootObjects) {
        String keyName = (rootName != null) ? rootName : this.currentRootName;
        this.roots.put(keyName, rootObjects);
    }

    /**
     * Get the domain object for the given root role name. If there are more
     * than one object for the root role name the first is returned.
     *
     * @param rootName The name of the root role for the domain object, or
     *            <code>currentRootName</code> if null
     * @return The first domain object for the given root role name
     */
    public Object getRootObject(String rootName) {
        List<Object> objects = getRootObjects(rootName);
        return (objects != null && objects.size() > 0) ? objects.get(0) : null;
    }

    /**
     * Get the list of domain objects for the given root role name.
     *
     * @param rootName The name of the root role for the domain object, or
     *            <code>currentRootName</code> if null
     * @return The list of domain objects for the given root role name
     */
    public List<Object> getRootObjects(String rootName) {
        String keyName = (rootName != null) ? rootName : this.currentRootName;
        return this.roots.get(keyName);
    }

    /**
     * Access method for the currentRootName property.
     *
     * @return the current value of the currentRootName property
     */
    public String getCurrentRootName() {
        return currentRootName;
    }

    /**
     * Sets the value of the currentRootName property.
     *
     * @param aCurrentRootName the new value of the currentRootName property
     */
    public void setCurrentRootName(String aCurrentRootName) {
        currentRootName = aCurrentRootName;
    }

    /**
     * Access method for the targetRoleName property.
     *
     * @return the current value of the targetRoleName property
     */
    public String getTargetRoleName() {
        return targetRoleName;
    }

    /**
     * Sets the value of the targetRoleName property.
     *
     * @param aTargetRoleName the new value of the targetRoleName property
     */
    public void setTargetRoleName(String aTargetRoleName) {
        targetRoleName = aTargetRoleName;
    }

    /**
     * Access method for the objectSelectionName property.
     *
     * @return the current value of the objectSelectionName property
     */
    public String getObjectSelectionName() {
        return objectSelectionName;
    }

    /**
     * Sets the value of the objectSelectionName property.
     *
     * @param aObjectSelectionName the new value of the objectSelectionName
     *            property
     */
    public void setObjectSelectionName(String aObjectSelectionName) {
        objectSelectionName = aObjectSelectionName;
    }

    /**
     * Access method for the applicationName property.
     *
     * @return the current value of the applicationName property
     */
    public String getApplicationName() {
        return applicationName;
    }

    /**
     * Sets the value of the applicationName property.
     *
     * @param aApplicationName the new value of the applicationName property
     */
    public void setApplicationName(String aApplicationName) {
        applicationName = aApplicationName;
    }

    /**
     * @return Returns the findData.
     */
    public FindData getFindData() {
        return findData;
    }

    /**
     * @param findData The findData to set.
     */
    public void setFindData(FindData findData) {
        this.findData = findData;
    }

    /**
     * @return Returns the targetMainKey
     */
    public String getTargetMainKey() {
        return targetMainKey;
    }

    /**
     * The main key used to identify the target role instance. Primary key is
     * used if not set.
     *
     * @param targetMainKey (missing javadoc)
     */
    public void setTargetMainKey(String targetMainKey) {
        this.targetMainKey = targetMainKey;
    }

    /**
     * @return Returns the setsContainAllRelated flag
     */
    public boolean getSetsContainAllRelated() {
        return setsContainAllRelated;
    }

    /**
     * Sets the setsContainAllRelated flag
     *
     * @param setsContainAllRelated (missing javadoc)
     */
    public void setSetsContainAllRelated(boolean setsContainAllRelated) {
        this.setsContainAllRelated = setsContainAllRelated;
    }

    /**
     * @return Returns the actions map
     */
    public Map<Object, ActionType> getActions() {
        return actions;
    }

    /**
     * Sets the actions map
     *
     * @param actions (missing javadoc)
     */
    public void setActions(Map<Object, ActionType> actions) {
        this.actions = actions;
    }

    /**
     * @return Returns the doParents flag
     */
    public boolean getDoParents() {
        return doParents;
    }

    /**
     * Sets the doParents flag
     *
     * @param doParents (missing javadoc)
     */
    public void setDoParents(boolean doParents) {
        this.doParents = doParents;
    }

    /**
     * @return Returns the doOneRelated flag
     */
    public boolean getDoOneRelated() {
        return doOneRelated;
    }

    /**
     * Sets the doOneRelated flag
     *
     * @param doOneRelated (missing javadoc)
     */
    public void setDoOneRelated(boolean doOneRelated) {
        this.doOneRelated = doOneRelated;
    }

    /**
     * @return Returns the doFirstLevelManyRelated flag
     */
    public boolean getDoFirstLevelManyRelated() {
        return doFirstLevelManyRelated;
    }

    /**
     * Sets the doFirstLevelManyRelated flag
     *
     * @param doFirstLevelManyRelated (missing javadoc)
     */
    public void setDoFirstLevelManyRelated(boolean doFirstLevelManyRelated) {
        this.doFirstLevelManyRelated = doFirstLevelManyRelated;
    }

    /**
     * @return Returns the doAnyLevelManyRelated flag
     */
    public boolean getDoAnyLevelManyRelated() {
        return doAnyLevelManyRelated;
    }

    /**
     * Sets the doAnyLevelManyRelated flag
     *
     * @param doAnyLevelManyRelated (missing javadoc)
     */
    public void setDoAnyLevelManyRelated(boolean doAnyLevelManyRelated) {
        this.doAnyLevelManyRelated = doAnyLevelManyRelated;
    }

    @Override
    public String toString() {
        return "ObjectSelection [currentRootName=" + currentRootName
                + ", targetRoleName=" + targetRoleName + ", targetMainKey="
                + targetMainKey + ", roots=" + roots + ", objectSelectionName="
                + objectSelectionName + ", applicationName=" + applicationName
                + ", findData=" + findData + ", setsContainAllRelated="
                + setsContainAllRelated + ", actions=" + actions
                + ", doParents=" + doParents + ", doOneRelated=" + doOneRelated
                + ", doFirstLevelManyRelated=" + doFirstLevelManyRelated
                + ", doAnyLevelManyRelated=" + doAnyLevelManyRelated + "]";
    }

}
