/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.Collator;
import java.text.ParseException;
import java.text.RuleBasedCollator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import no.g9.exception.G9BaseException;
import no.g9.exception.G9ClientFrameworkException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 * Various helper methods for type conversions etc. This class may change
 * without notice in a later version of g9.
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public class TypeTool {

    /** Default <code>Map</code> implementation */
    static Class<? extends Map> defaultMapType = HashMap.class;

    /** Default <code>Set</code> implementation */
    static Class<? extends Set> defaultSetType = HashSet.class;

    /** Default <code>List</code> implementation */
    static Class<? extends List> defaultListType = ArrayList.class;

    /** Default <code>Collection</code> implementation */
    static Class<? extends Collection> defaultCollectionType = HashSet.class;

    private static String msgIllegal(Class<?> argument, String type) {
        return "The specified type (" + argument.getName()
                + ") does not implement the " + type + " interface";
    }

    /**
     * Sorting collator used by compareTo(object,object) when comparing Strings
     */
    private static Collator collator = null;

    private static String msgInstantiate(String type) {
        return "Could not instantiate default " + type;
    }

    private static String msgAccess(String type) {
        return "Could not access default constructor "
                + "while trying to instansiate default " + type;
    }

    /**
     * Returns the class object representing the default map type.
     *
     * @return class representing the default map type.
     */
    public static final Class<? extends Map> getDefaultMapType() {
        return defaultMapType;
    }

    /**
     * Sets the default type for map. Consecutive invokations of
     * <code>getDefaultMap</code> returns instances of <code>mapType</code>
     *
     * @param mapType class implementing the Map interface.
     */
    public static final void setDefaultMapType(Class<? extends Map> mapType) {
        // Check type before assigning
        if (!defaultMapType.isAssignableFrom(mapType)) {
            throw new IllegalArgumentException(msgIllegal(mapType, "Map"));
        }
        defaultMapType = mapType;
    }

    /**
     * Returns a new instance of the default map type.
     *
     * @param <K> Key type for returned map
     * @param <V> Valie type for returned map
     * @return a new Map instance
     */
    public static final <K, V> Map<K, V> getDefaultMap() {
        try {
            return defaultMapType.newInstance();
        } catch (InstantiationException e) {
            throw new G9BaseException(msgInstantiate("Map"), e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(msgAccess("Map"), e);
        }
    }

    /**
     * Sets the default map type to be of the same type as <code>map</code>
     *
     * @param map object representing the (new) default map type.
     */
    public static final void setDefaultMap(Map<?, ?> map) {
        defaultMapType = map.getClass();
    }

    // ///////////////////////////// SET ///////////////////////////

    /**
     * Returns the class object representing the default set type.
     *
     * @return class representing the default set type.
     */
    public static final Class<? extends Set> getDefaultSetType() {
        return defaultSetType;
    }

    /**
     * Sets the default type for set. Consecutive invokations of
     * <code>getDefaultSet</code> returns instances of <code>setType</code>
     *
     * @param setType class implementing the Set interface.
     */
    public static final void setDefaultSetType(Class<? extends Set> setType) {
        // Check type before assigning
        if (!defaultSetType.isAssignableFrom(setType)) {
            throw new IllegalArgumentException(msgIllegal(setType, "Set"));
        }
        defaultSetType = setType;
    }

    /**
     * Returns a new instance of the default set type.
     *
     * @param <E> Element type for the Set
     * @return a new Set instance
     */
    public static final <E> Set<E> getDefaultSet() {
        try {
            return defaultSetType.newInstance();
        } catch (InstantiationException e) {
            throw new G9BaseException(msgInstantiate("Set"), e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(msgAccess("Set"), e);
        }
    }

    /**
     * Creates a new set, and adds all instances contained in the specified set
     * to the new set. The purpose of this method is to ensure that the new set
     * is hashed properly according to the invoking VM's hashing algorithm.
     *
     * @param <E> Type of set elements
     * @param serializedSet the set to be re-hashed.
     * @return the new rehashed set
     */
    public static <E> Set<E> reHash(Set<E> serializedSet) {
        Set<E> newSet = TypeTool.getDefaultSet();
        newSet.addAll(serializedSet);
        return newSet;
    }

    /**
     * Sets the default set type to be of the same type as <code>set</code>
     *
     * @param set object representing the (new) default set type.
     */
    public static final void setDefaultSet(Set<?> set) {
        defaultSetType = set.getClass();
    }

    // ///////////////////////////// LIST ///////////////////////////

    /**
     * Returns the class object representing the default list type.
     *
     * @return class representing the default list type.
     */
    public static final Class<? extends List> getDefaultListType() {
        return defaultListType;
    }

    /**
     * Sets the default type for list. Consecutive invokations of
     * <code>getDefaultList</code> returns instances of <code>listType</code>
     *
     * @param listType class implementing the List interface.
     */
    public static final void setDefaultListType(Class<? extends List> listType) {
        // Check type before assigning
        if (!defaultListType.isAssignableFrom(listType)) {
            throw new IllegalArgumentException(msgIllegal(listType, "List"));
        }
        defaultListType = listType;
    }

    /**
     * Returns a new instance of the default list type.
     *
     * @param <E> Element type for List
     * @return a new List instance
     */
    public static final <E> List<E> getDefaultList() {
        try {
            return defaultListType.newInstance();
        } catch (InstantiationException e) {
            throw new G9BaseException(msgInstantiate("List"), e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(msgAccess("List"), e);
        }
    }

    /**
     * Sets the default list type to be of the same type as <code>list</code>
     *
     * @param list object representing the (new) default list type.
     */
    public static final void setDefaultList(List<?> list) {
        defaultListType = list.getClass();
    }

    // ///////////////////////////// COLLECTION ///////////////////////////

    /**
     * Returns the class object representing the default collection type.
     *
     * @return class representing the default collection type.
     */
    public static final Class<? extends Collection> getDefaultCollectionType() {
        return defaultCollectionType;
    }

    /**
     * Sets the default type for collection. Consecutive invokations of
     * <code>getDefaultCollection</code> returns instances of
     * <code>collectionType</code>
     *
     * @param collectionType class implementing the Collection interface.
     */
    public static final void setDefaultCollectionType(
            Class<? extends Collection> collectionType) {
        // Check type before assigning
        if (!defaultCollectionType.isAssignableFrom(collectionType)) {
            throw new IllegalArgumentException(msgIllegal(collectionType,
                    "Collection"));
        }
        defaultCollectionType = collectionType;
    }

    /**
     * Returns a new instance of the default collection type.
     *
     * @param <E> Element type for Collection
     * @return a new Collection instance
     */
    public static final <E> Collection<E> getDefaultCollection() {
        try {
            return defaultCollectionType.newInstance();
        } catch (InstantiationException e) {
            throw new G9BaseException(msgInstantiate("Collection"), e);
        } catch (IllegalAccessException e) {
            throw new G9BaseException(msgAccess("Collection"), e);
        }
    }

    /**
     * Sets the default collection type to be of the same type as
     * <code>collection</code>
     *
     * @param collection object representing the (new) default map type.
     */
    public static final void setDefaultCollection(Collection<?> collection) {
        defaultCollectionType = collection.getClass();
    }

    /**
     * Check the java version. It simply checks to see if
     * System.getProperty("java.version") starts with the input string.
     *
     * @param version (missing javadoc)
     * @return true if java vm version property contains the version string
     */
    public static final boolean isVmVersion(String version) {
        String vmVersion = System.getProperty("java.version");
        return vmVersion != null && vmVersion.startsWith(version);
    }

    /**
     * Check if java vm version is 1.4
     *
     * @return true if java vm is 1.4
     */
    public static final boolean isVmVersion1_4() {
        return isVmVersion("1.4");
    }

    /**
     * This method puts leading zeroes in front of a number which is an int
     * intToConvert is the int that will used to add leading zeroes to length is
     * the length of the string that will be returned Example: String tmp =
     * leadingZeroes(2, 4); returns "0002"
     *
     * @param intToConvert the int to prefix with leading zeroes.
     * @param length the length of the text (including leading zeroes)
     * @return a text with leading zeroes
     * @deprecated Since 8.2.1b. Use java.text.DecimalFormat instead
     */
    @Deprecated
    public static String leadingZeroes(int intToConvert, int length) {
        String tall = "1";
        for (int i = 1; i < length; i++) {
            tall = tall + "0";
        }
        String tallSomSkal = Integer.toString(intToConvert);
        int diff = tall.length() - tallSomSkal.length();
        StringBuffer tmpTall = null;
        if (diff > 0) {
            tmpTall = new StringBuffer(tallSomSkal);
            for (int teller = 0; teller < diff; teller++) {
                tmpTall.insert(0, "0");
            }
        } else {
            return String.valueOf(intToConvert);
        }
        return tmpTall.toString();
    }

    /**
     * This method uses reflection to copy the field "field" from object "from"
     * to object "to". The field must be public. It support the regular Java
     * basic types, and in addition it support String, Date, G9s Numeric and
     * G9s Enumerators. (Enumerators must always inherit from
     * G9Enumerator, and nothing but that)
     *
     * @param from the object to copy from
     * @param field the attribute to copy
     * @param to the object to copy to
     */
    public static void reflectCopyMethod(Object from, String field, Object to) {
        if (from == null) {
            return;
        }

        String msgID = null;
        Exception ex = null;

        try {
            String capAttribute = field.substring(0, 1).toUpperCase()
                    + field.substring(1);
            Class<?> toClass = to.getClass();
            Method copyFromGetMethod = from.getClass().getMethod(
                    "get" + capAttribute, (Class[]) null);
            Class<?> fromFieldType = copyFromGetMethod.getReturnType();
            Class<?>[] paramSetMethod = { fromFieldType };
            Method copyToSetMethod = toClass.getMethod("set" + capAttribute,
                    paramSetMethod);
            // Work-around for bug #4533479 in java
            copyFromGetMethod.setAccessible(true);
            copyToSetMethod.setAccessible(true);
            if (fromFieldType == Numeric.class) {
                Object oldNumeric = copyFromGetMethod.invoke(from,
                        (Object[]) null);
                if (oldNumeric == null) {
                    return;
                }
                Method copyMethod = Numeric.class.getMethod("makeCopy",
                        (Class[]) null);
                // Work-around for bug #4533479 in java
                copyMethod.setAccessible(true);
                Object noParams[] = {};
                Object[] foo = { copyMethod.invoke(oldNumeric, noParams) };
                copyToSetMethod.invoke(to, foo);
            }
            /* Date ? */
            else if (fromFieldType == Date.class) {
                Object oldDate = copyFromGetMethod
                        .invoke(from, (Object[]) null);
                if (oldDate == null) {
                    Object[] nullparam = { null };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Method copyMethod = Date.class.getMethod("clone",
                        (Class[]) null);
                // Work-around for bug #4533479 in java
                copyMethod.setAccessible(true);
                Object noParams[] = {};
                Object[] foo = { copyMethod.invoke(oldDate, noParams) };
                copyToSetMethod.invoke(to, foo);
            } else if (fromFieldType == String.class) {
                Object fromString = copyFromGetMethod.invoke(from,
                        (Object[]) null);
                if (fromString != null) {
                    Object[] foo = { String.valueOf(fromString) };
                    copyToSetMethod.invoke(to, foo);
                }
            }
            /* G9 Enumerator ? */
            else if (fromFieldType.getSuperclass() == G9Enumerator.class) {
                Object newEnumObj = fromFieldType.newInstance();
                Object oldEnumObj = copyFromGetMethod.invoke(from,
                        (Object[]) null);
                /* No value - then do nothing */
                if (oldEnumObj == null) {
                    return;
                }
                Object[] foo = { newEnumObj };
                copyToSetMethod.invoke(to, foo);
                /* Get field for value */
                Field valueField = oldEnumObj.getClass().getField(
                        "currentValue");
                // Work-around for bug #4533479 in java
                valueField.setAccessible(true);
                /* Then set value */
                valueField.setInt(newEnumObj, valueField.getInt(oldEnumObj));
            } else if (fromFieldType == Integer.TYPE
                    || fromFieldType == Float.TYPE
                    || fromFieldType == Double.TYPE
                    || fromFieldType == Character.TYPE
                    || fromFieldType == Boolean.TYPE
                    || fromFieldType == Byte.TYPE || fromFieldType == Long.TYPE
                    || fromFieldType == Short.TYPE
                    || fromFieldType == Integer.class
                    || fromFieldType == Float.class
                    || fromFieldType == Character.class
                    || fromFieldType == Long.class
                    || fromFieldType == Boolean.class
                    || fromFieldType == Double.class
                    || fromFieldType == Byte.class
                    || fromFieldType == Short.class) {
                Object[] foo = { copyFromGetMethod
                        .invoke(from, (Object[]) null) };
                copyToSetMethod.invoke(to, foo);
            }
        } catch (NoSuchMethodException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (IllegalAccessException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InvocationTargetException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InstantiationException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (NoSuchFieldException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        from.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
    }

    /**
     * Replaces a string with a string in a string.
     *
     * @param str The original string.
     * @param o The string to replace.
     * @param n The string to change to.
     * @param all true if all occurrences should be changed, false if just the
     *            first.
     * @return the new string.
     */
    public static String replace(String str, String o, String n, boolean all) {
        if (str == null || o == null || o.length() == 0 || n == null) {
            throw new IllegalArgumentException("null or empty String");
        }
        StringBuffer result = null;
        int oldpos = 0;
        do {
            int pos = str.indexOf(o, oldpos);
            if (pos < 0) {
                break;
            }
            if (result == null) {
                result = new StringBuffer();
            }
            result.append(str.substring(oldpos, pos));
            result.append(n);
            pos += o.length();
            oldpos = pos;
        } while (all);
        if (oldpos == 0) {
            return str;
        }
        if (result != null){
        	result.append(str.substring(oldpos));
        	return result.toString();
        }
        return null;
    }

    /**
     * Returns a list of all elements from fromLista that are not contained in
     * fromListb(disjunct) Requires a comparator(with equals implemented) for
     * the objects contained in the two Lists. Requires that the list in
     * fromListb is possible to sort.
     *
     * @param fromLista list of candidates to return
     * @param fromListb the list of elements to not return
     * @param objComp the comparator
     * @return a new list containing elements unique to <code>fromLista</code>
     * @deprecated Since 8.2.1.b
     */
    // @SuppressWarnings("unchecked")
    @Deprecated
    public static List disjunctListBinary(List fromLista, List fromListb,
            G9Comparator objComp) {
        if (fromListb.isEmpty()) {
            return fromLista;
        }
        Collections.sort(fromListb, objComp);
        List notInListb = new Vector();
        Iterator listaIterator = fromLista.iterator();
        while (listaIterator.hasNext()) {
            // boolean found = false;
            Object a = listaIterator.next();
            if (Collections.binarySearch(fromListb, a, objComp) < 0) {
                notInListb.add(a);
            }
        }
        return notInListb;
    }

    /**
     * Copies the specified attribute from the <code>from</code> object to the
     * <code>to</code> object.
     *
     * @param from the object to copy from
     * @param field the field to copy
     * @param to the object to copy to.
     */
    public static void reflectCopy(Object from, String field, Object to) {
        if (from == null) {
            return;
        }
        String msgID = null;
        Exception ex = null;
        try {
            Field copyFromField = from.getClass().getField(field);
            Field copyToField = to.getClass().getField(field);
            // Work-around for bug #4533479 in java
            copyFromField.setAccessible(true);
            copyToField.setAccessible(true);
            Class<?> fromFieldType = copyFromField.getType();
            Class<?> copyToFieldType = copyToField.getType();
            /* Numeric? */
            if (fromFieldType == Numeric.class) {
                Object oldNumeric = copyFromField.get(from);
                if (oldNumeric == null) {
                    return;
                }
                Method copyMethod = Numeric.class.getMethod("makeCopy",
                        (Class[]) null);
                // Work-around for bug #4533479 in java
                copyMethod.setAccessible(true);
                Object noParams[] = {};
                Object foo = copyMethod.invoke(oldNumeric, noParams);
                copyToField.set(to, foo);
            }
            /* Date ? */
            else if (fromFieldType == Date.class) {
                Object oldDate = copyFromField.get(from);
                if (oldDate == null) {
                    copyToField.set(to, null);
                    return;
                }
                Method copyMethod = Date.class.getMethod("clone",
                        (Class[]) null);
                // Work-around for bug #4533479 in java
                copyMethod.setAccessible(true);
                Object noParams[] = {};
                Object foo = copyMethod.invoke(oldDate, noParams);
                copyToField.set(to, foo);
            }
            /* String ? */
            else if (fromFieldType == String.class) {
                Object fromString = copyFromField.get(from);
                if (fromString != null) {
                    copyToField.set(to, String.valueOf(fromString));
                }
            }
            /* G9 Enumerator ? */
            else if (fromFieldType.getSuperclass() == G9Enumerator.class) {
                Object newEnumObj = copyToFieldType.newInstance();
                Object oldEnumObj = copyFromField.get(from);
                /* No value - then do nothing */
                if (oldEnumObj == null) {
                    return;
                }
                /* Assign new object */
                copyToField.set(to, newEnumObj);
                /* Get field for value */
                Field valueField = copyToFieldType.getField("currentValue");
                // Work-around for bug #4533479 in java
                valueField.setAccessible(true);
                /* Then set value */
                valueField.setInt(newEnumObj, valueField.getInt(oldEnumObj));
            } else if (fromFieldType == Integer.TYPE) {
                copyToField.setInt(to, copyToField.getInt(from));
            } else if (fromFieldType == Float.TYPE) {
                copyToField.setFloat(to, copyToField.getFloat(from));
            } else if (fromFieldType == Double.TYPE) {
                copyToField.setDouble(to, copyToField.getDouble(from));
            } else if (fromFieldType == Character.TYPE) {
                copyToField.setChar(to, copyToField.getChar(from));
            } else if (fromFieldType == Boolean.TYPE) {
                copyToField.setBoolean(to, copyToField.getBoolean(from));
            } else if (fromFieldType == Byte.TYPE) {
                copyToField.setByte(to, copyToField.getByte(from));
            } else if (fromFieldType == Long.TYPE) {
                copyToField.setLong(to, copyToField.getLong(from));
            } else if (fromFieldType == Short.TYPE) {
                copyToField.setShort(to, copyToField.getShort(from));
            }
        } catch (NoSuchMethodException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (IllegalAccessException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InvocationTargetException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InstantiationException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (NoSuchFieldException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        from.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
    }

    /**
     * Returns the tab sequence map
     *
     * @param tabSequence array of tab sequence
     * @param next boolean flag
     * @return the tab sequence map
     */
    public static Map<String, String> quickIndex(String[] tabSequence,
            boolean next) {
        if (tabSequence == null || tabSequence.length == 0) {
            return new HashMap<String, String>();
        }
        HashMap<String, String> answer = new HashMap<String, String>();
        if (next) {
            for (int i = 0; i < tabSequence.length - 1; i++) {
                answer.put(tabSequence[i], tabSequence[i + 1]);
            }
            answer.put(tabSequence[tabSequence.length - 1], tabSequence[0]);
        } else {
            for (int i = tabSequence.length - 1; i >= 1; i--) {
                answer.put(tabSequence[i], tabSequence[i - 1]);
            }
            answer.put(tabSequence[0], tabSequence[tabSequence.length - 1]);
        }
        return answer;
    }

    /**
     * Returns a list of all elements from fromLista that are not contained in
     * fromListb(disjunct) Requires a comparator(with equals implemented) for
     * the objects caontained in the two Lists.
     *
     * @param fromLista the list containing all elements
     * @param fromListb the list containing the elements that will be removed
     * @param objComp the comparator
     * @return a list containing elements unique to the <code>fromLista</code>
     *         list.
     * @deprecated Since 8.2.1.b
     */
    @Deprecated
    public static List disjunctList(List fromLista, List fromListb,
            G9Comparator objComp) {
        if (fromListb.isEmpty()) {
            return fromLista;
        }
        List notInListb = new Vector();
        Iterator listaIterator = fromLista.iterator();
        Iterator listbIterator = fromListb.iterator();
        while (listaIterator.hasNext()) {
            boolean found = false;
            Object a = listaIterator.next();
            while (listbIterator.hasNext()) {
                Object b = listbIterator.next();
                if (objComp.equals(a, b)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                notInListb.add(a);
            }
            listbIterator = fromListb.iterator();
        }
        return notInListb;
    }

    /**
     * Returns a list which is the union of fromLista and fromListb(A u B)
     * Requires a comparator(with equals implemented) for the objects caontained
     * in the two Lists.
     *
     * @param fromLista a list
     * @param fromListb a list
     * @param objComp the comparator
     * @return a new list containg the elements present in both lists.
     * @deprecated Since 8.2.1.b
     */
    // @SuppressWarnings("unchecked")
    @Deprecated
    public static List unionLists(List fromLista, List fromListb,
            G9Comparator objComp) {
        List unionList = new Vector();
        Iterator listaIterator = fromLista.iterator();
        Iterator listbIterator = fromListb.iterator();
        while (listaIterator.hasNext()) {
            Object a = listaIterator.next();
            while (listbIterator.hasNext()) {
                Object b = listbIterator.next();
                if (objComp.equals(a, b)) {
                    unionList.add(a);
                }
            }
            listbIterator = fromListb.iterator();
        }
        return unionList;
    }

    /**
     * Assigns the specified value using reflection.
     *
     * @param to The object to assign the value to.
     * @param field The attribute to assign to.
     * @param value The value to assign.
     */
    public static void reflectAssignMethod(Object to, String field, String value) {
        if (value == null) {
            return;
        }
        String msgID = null;
        Exception ex = null;
        try {
            String capAttribute = field.substring(0, 1).toUpperCase()
                    + field.substring(1);
            Class<?> toClass = to.getClass();
            Method copyFromGetMethod = to.getClass().getMethod(
                    "get" + capAttribute, (Class[]) null);
            Class<?> copyToFieldType = copyFromGetMethod.getReturnType();
            Class<?>[] paramSetMethod = { copyToFieldType };
            Method copyToSetMethod = toClass.getMethod("set" + capAttribute,
                    paramSetMethod);
            // Work-around for bug #4533479 in java
            copyFromGetMethod.setAccessible(true);
            copyToSetMethod.setAccessible(true);

            if (copyToFieldType == Numeric.class) {
                /* Empty strings means assign to 0 */
                if (value.equals("")) {
                    Object[] nullparam = { null };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object numericObject = new Numeric(0, 0);
                Class<?>[] declParams = { String.class };
                Method assignMethod = numericObject.getClass().getMethod(
                        "setValue", declParams);
                // Work-around for bug #4533479 in java
                assignMethod.setAccessible(true);
                Object params[] = { String.valueOf(value) };
                assignMethod.invoke(numericObject, params);
                Object[] foo = { numericObject };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Date.class) {
                /* Empty strings means nothing here */
                if (value.trim().equals("")) {
                    Object[] nullparam = { null };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] newdate = { parse(value) };
                copyToSetMethod.invoke(to, newdate);
            } else if (copyToFieldType == String.class) {
                Object[] newString = { value };
                copyToSetMethod.invoke(to, newString);
            } else if (copyToFieldType.getSuperclass() == G9Enumerator.class) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    Object[] nullparam = { null };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object newEnumObj = copyToFieldType.newInstance();
                Field valueField = copyToFieldType.getField("currentValue");
                // Work-around for bug #4533479 in java
                valueField.setAccessible(true);
                valueField.setInt(newEnumObj, Integer.parseInt(value));
                Object[] foo = { newEnumObj };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Boolean.TYPE
                    || copyToFieldType == Boolean.class) {
                if (value.equals("")) {
                    return;
                }
                Object[] foo = { Boolean.valueOf(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Integer.TYPE
                    || copyToFieldType == Integer.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Integer.TYPE ? Integer.valueOf(
                            0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { Integer.valueOf(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Float.TYPE
                    || copyToFieldType == Float.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Float.TYPE ? new Float(
                            0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { new Float(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Double.TYPE
                    || copyToFieldType == Double.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Double.TYPE ? new Double(
                            0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { new Double(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Character.TYPE
                    || copyToFieldType == Character.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Character.TYPE ? Character.valueOf(
                            ' ')
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { Character.valueOf(value.toCharArray()[0]) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Byte.TYPE
                    || copyToFieldType == Byte.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Byte.TYPE ? Byte.valueOf(
                            (byte) 0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { Byte.valueOf(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Long.TYPE
                    || copyToFieldType == Long.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Long.TYPE ? Long.valueOf(
                            0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { Long.valueOf(value) };
                copyToSetMethod.invoke(to, foo);
            } else if (copyToFieldType == Short.TYPE
                    || copyToFieldType == Short.class) {
                if (value.equals("")) {
                    Object[] nullparam = { (copyToFieldType == Short.TYPE ? Short.valueOf(
                            (short) 0)
                            : null) };
                    copyToSetMethod.invoke(to, nullparam);
                    return;
                }
                Object[] foo = { Short.valueOf(value) };
                copyToSetMethod.invoke(to, foo);
            }
        } catch (NoSuchMethodException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (IllegalAccessException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InvocationTargetException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InstantiationException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (NoSuchFieldException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        to.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
    }

    /**
     * Performs assing using retrospection on field 'field' of type class
     * 'fieldClassName' in object 'to', with the value 'value'
     *
     * @param to The object to assign to
     * @param field The attribute to assign to
     * @param value The value to assign.
     */
    public static void reflectAssign(Object to, String field, String value) {
        String msgID = null;
        Exception ex = null;
        try {
            if (value == null) {
                return;
            }

            Field copyToField = to.getClass().getField(field);
            // Work-around for bug #4533479 in java
            copyToField.setAccessible(true);
            Class<?> copyToFieldType = copyToField.getType();
            if (copyToFieldType == Numeric.class) {
                /* Empty strings means assign to 0 */
                if (value.equals("")) {
                    copyToField.set(to, new Numeric(0, 0));
                    return;
                }
                Object numericObject = new Numeric(0, 0);
                Class<?> declParams[] = { String.class };
                Method assignMethod = numericObject.getClass().getMethod(
                        "setValue", declParams);
                // Work-around for bug #4533479 in java
                assignMethod.setAccessible(true);
                Object params[] = { String.valueOf(value) };
                assignMethod.invoke(numericObject, params);
                /* Assign to field */
                copyToField.set(to, numericObject);
            } else if (copyToFieldType == Date.class) {
                /* Empty strings means nothing here */
                if (value.trim().equals("")) {
                    copyToField.set(to, null);
                    return;
                }
                Date newDate = parse(value);
                copyToField.set(to, newDate);
            } else if (copyToFieldType == String.class) {
                String newString = value;
                copyToField.set(to, newString);
            } else if (copyToFieldType.getSuperclass() == G9Enumerator.class) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                Object newEnumObj = copyToFieldType.newInstance();
                Field valueField = copyToFieldType.getField("currentValue");
                // Work-around for bug #4533479 in java
                valueField.setAccessible(true);
                /* Then set value - it is a title here! */
                valueField.setInt(newEnumObj, Integer.parseInt(value));
                copyToField.set(to, newEnumObj);
            } else if (copyToFieldType == Boolean.TYPE) {
                if (value.equals("")) {
                    return;
                }
                copyToField.setBoolean(to,
                        (value.equals("true") ? true : false));
            } else if (copyToFieldType == Integer.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setInt(to, Integer.parseInt(value));
            } else if (copyToFieldType == Float.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setFloat(to, Float.parseFloat(value));
            } else if (copyToFieldType == Double.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setDouble(to, Double.parseDouble(value));
            } else if (copyToFieldType == Character.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setChar(to, value.toCharArray()[0]);
            } else if (copyToFieldType == Byte.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setByte(to, Byte.parseByte(value));
            } else if (copyToFieldType == Long.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setLong(to, Long.parseLong(value));
            } else if (copyToFieldType == Short.TYPE) {
                /* Empty strings means nothing here */
                if (value.equals("")) {
                    return;
                }
                copyToField.setShort(to, Short.parseShort(value));
            }
        } catch (NoSuchMethodException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (IllegalAccessException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InvocationTargetException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InstantiationException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (NoSuchFieldException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        to.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
    }

    /**
     * REturns the getter method name for the specified field
     *
     * @param owner the object with the field
     * @param field the attribute
     * @return the get-method name
     */
    public static String getAttributeClassNameMethod(Object owner, String field) {
        String msgID = null;
        Exception ex = null;
        if (owner == null) {
            return null;
        }
        try {
            String methodName = "get" + field.substring(0, 1).toUpperCase()
                    + field.substring(1);
            Method m = owner.getClass().getMethod(methodName, (Class[]) null);
            String name = m.getReturnType().getName();
            int lastDot = name.lastIndexOf('.');
            if (lastDot == -1) {
                return name;
            }
            return name.substring(lastDot + 1);
        } catch (NoSuchMethodException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        owner.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }

        return null;
    }

    /**
     * Returns the attribute class name
     *
     * @param owner the object with the attribute
     * @param field the name of the attribute
     * @return the attribute class name.
     */
    public static String getAttributeClassName(Object owner, String field) {
        if (owner == null) {
            return null;
        }
        try {
            Field f = owner.getClass().getField(field);
            String name = f.getType().getName();
            int lastDot = name.lastIndexOf('.');
            if (lastDot == -1) {
                return name;
            }
            return name.substring(lastDot + 1);
        } catch (NoSuchFieldException e) {
            Object[] msgArgs = { TypeTool.class.getClass(), field,
                    owner.getClass(), e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(
                    CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ClientFrameworkException(e, msg);
        }
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Object convertIgnore(int primitive, boolean ignoreZero) {
        if (ignoreZero) {
            return convert(primitive);
        }
        return Integer.valueOf(primitive);
    }

    /**
     * Converts an int to an Integer, or "" if it is 0
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Object convert(int primitive) {
        if (primitive == 0) {
            return "";
        }
        return Integer.valueOf(primitive);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Object convertIgnore(short primitive, boolean ignoreZero) {
        if (ignoreZero) {
            return convert(primitive);
        }
        return Short.valueOf(primitive);
    }

    /**
     * Converts a short to a Short
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Object convert(short primitive) {
        if (primitive == 0) {
            return "";
        }
        return Short.valueOf(primitive);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Object convertIgnore(long primitive, boolean ignoreZero) {
        if (ignoreZero) {
            return convert(primitive);
        }
        return Long.valueOf(primitive);
    }

    /**
     * Converts a long to a Long
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Object convert(long primitive) {
        if (primitive == 0) {
            return "";
        }
        return Long.valueOf(primitive);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Object convertIgnore(double primitive, boolean ignoreZero) {
        if (ignoreZero) {
            return convert(primitive);
        }
        return new Double(primitive);
    }

    /**
     * Converts a double to a Double
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Object convert(double primitive) {
        if (primitive == 0) {
            return "";
        }
        return new Double(primitive);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Object convertIgnore(float primitive, boolean ignoreZero) {
        if (ignoreZero) {
            return convert(primitive);
        }
        return new Float(primitive);
    }

    /**
     * Converts a float to a Float
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Object convert(float primitive) {
        if (primitive == 0) {
            return "";
        }
        return new Float(primitive);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
    static public Boolean convertIgnore(boolean primitive, boolean ignoreZero) {
        return Boolean.valueOf(primitive);
    }

    /**
     * Converts a boolean to a Boolean
     *
     * @param b the boolean to convert
     * @return the converted boolean.
     */
    static public Boolean convert(boolean b) {
        return Boolean.valueOf(b);
    }

    /**
     * Convert the primitive.
     *
     * @param primitive the primitive to wrap
     * @param ignoreZero if <code>true</code> zero values will converted to the
     *            empty string.
     * @return the converted object.
     */
	static public Character convertIgnore(char primitive, boolean ignoreZero) {
		return Character.valueOf(primitive);
	}

    /**
     * Converts a char to a Character
     *
     * @param primitive the primitive to convert.
     * @return the converted primitive.
     */
    static public Character convert(char primitive) {
        return Character.valueOf(primitive);
    }

    /**
     * Converts a numeric to it's string representation. If the ignoreZero flag
     * is true, zero values will be returned as an empty string ("").
     *
     * @param n the numeric to convert
     * @param ignoreZero boolean flag indicating if zero should be ignored.
     * @return the string representation of the numeric value.
     */
    static public String convertIgnore(Numeric n, boolean ignoreZero) {
        if (n == null) {
            if (ignoreZero) {
                return "";
            }
            return "0";
        }
        return n.toString();
    }

    /**
     * Converts the specified numeric to it's string representation.
     *
     * @param n the numeric to convert
     * @return the string representation of the numeric value.
     */
    static public String convert(Numeric n) {
        if (n == null) {
            return "";
        }
        return n.toString();
    }

    /**
     * BUGGY. Converts the enumerator to a string.
     *
     * @param n the enumerator to convert
     * @param ignoreZero the ignore zero flag.
     * @return the string representation of the specified enumerator.
     */
    static public String convertIgnore(G9Enumerator n, boolean ignoreZero) {
        return convert(n);
    }

    /**
     * BUGGY? Converts the specified enumerator to a string represenatation of
     * the current value.
     *
     * @param n the enumerator to convert
     * @return the string representation of the enumerator's current value
     */
    static public String convert(G9Enumerator n) {
        if (n == null) {
            return "";
        }
        return String.valueOf(n.currentValue);
    }

    /**
     * If a null object is converted, an empty string is returned
     *
     * @param o the object to convert
     * @param ignoreZero the ignore zero flag
     * @return the converted object.
     */
    static public Object convertIgnore(Object o, boolean ignoreZero) {
        return convert(o);
    }

    /**
     * If a null object is converted, an empty string is returned
     *
     * @param o the object to convert
     * @return the converted object.
     */
    static public Object convert(Object o) {
        if (o == null) {
            return "";
        }
        return o;
    }

    /**
     * Parses the string to a Date
     *
     * @param arg the string to parse
     * @return a date
     */
    @SuppressWarnings("deprecation")
    static public Date parse(String arg) {
        return CustomDate.parse(CustomDate.superDateFormat(arg));
    }

    /**
     * Converts the specifed day, month and year to a date.
     *
     * @param day the day
     * @param month the month
     * @param year the year
     * @param d this parameter is not used any more.
     * @return a date
     */
    static public Date convert(String day, String month, String year, Date d) {
        return parse(day + month + year);
    }

    /**
     * Converts a String into a byte.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a byte.
     */
    static public byte convert(String data, byte field) {
        if (data == null || data.equals("")) {
            return 0;
        }
        field = Byte.valueOf(data).byteValue();
        return field;
    }

    /**
     * Converts a byte into a byte.
     *
     * @param data The byte to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The data converted to a byte.
     */
    static public byte convert(byte data, byte field) {
        return data;
    }

    /**
     * Converts a String into a short.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a short.
     */
    static public short convert(String data, short field) {
        if (data == null || data.equals("")) {
            return 0;
        }

        field = Short.valueOf(data).shortValue();

        return field;
    }

    /**
     * Converts a String into an int.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to an int.
     */
    static public int convert(String data, int field) {
        if (data == null || data.equals("")) {
            return 0;
        }
        field = Integer.parseInt(data);

        return field;
    }

    /**
     * Converts a int into an int.
     *
     * @param data The int to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The int data converted to an int.
     */
    static public int convert(int data, int field) {
        return data;
    }

    /**
     * Converts a String into a boolean.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a boolean.
     */
    static public boolean convert(String data, boolean field) {
        if (data == null || data.equals("")) {
            return false;
        }

        field = Boolean.getBoolean(data);

        return field;
    }

    /**
     * Converts a boolean into a boolean.
     *
     * @param data The boolean to be copied.
     * @param field Field used to control which convert method is to be called.
     * @return The boolean data "converted" to a boolean.
     */
    static public boolean convert(boolean data, boolean field) {
        return data;
    }

    /**
     * Converts a String into a String. Quite stupid, but nice when you
     * autogenerate code and do not want to differ :-)
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a String
     */
    static public String convert(String data, String field) {
        return data;
    }

    /**
     * Converts a String into a char.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a char
     */
    static public char convert(String data, char field) {
        if (data == null || data.equals("")) {
            return 0;
        }

        field = data.charAt(0);

        return field;
    }

    /**
     * Converts a String into a float.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a float
     */
    static public float convert(String data, float field) {
        if (data == null || data.equals("")) {
            return 0;
        }

        field = Float.valueOf(data).floatValue();

        return field;
    }

    /**
     * Converts a String into a double.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a double
     */
    static public double convert(String data, double field) {
        if (data == null || data.equals("")) {
            return 0;
        }

        field = Double.valueOf(data).doubleValue();

        return field;
    }

    /**
     * Converts a String into a long.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a long
     */
    static public long convert(String data, long field) {
        if (data == null || data.equals("")) {
            return 0;
        }
        field = Long.parseLong(data);

        return field;
    }

    /**
     * Converts a String into a Date.
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String data converted to a Date using yyyy.MM.dd .
     * @see java.util.Date
     */
    static public Date convert(String data, Date field) {
        return convertToDate(data);
    }

    /**
     * Converts the specified string to a date
     *
     * @param data the string date
     * @return a date.
     */
    static public java.util.Date convertToDate(String data) {
        if (data == null || data.equals("")) {
            return null;
        }
        return parse(data);
    }

    /**
     * Converts a Date into a Date.
     *
     * @param data The Date to be returned. (No copy here)
     * @param field Field used to control which convert method is to be called.
     * @return The Date data returned as a Date
     */
    static public java.util.Date convert(Date data, Date field) {
        return data;
    }

    /**
     * Converts a String into a Numeric. <br>
     *
     * @param data The String to be converted.
     * @param field Field used to control which convert method is to be called.
     * @return The String converted to a Numeric.
     */
    static public Numeric convert(String data, Numeric field) {
        return new Numeric(data, 0);
    }

    /**
     * Converts the string to a G9 Enumerator.
     *
     * @param data the current value (an int as string)
     * @param field the enumerator.
     * @return the enumerator with current value updated.
     */
    static public G9Enumerator convert(String data, G9Enumerator field) {
        G9Enumerator g = new G9Enumerator();
        g.currentValue = Integer.parseInt(data);
        return g;
    }

    /**
     * Converts an Object to String using toString().
     *
     * @param ob the object to convert
     * @param ignoreZero the ignoreZero flag (not used by this method)
     * @return the converted string object
     */
    static public String convertToString(Object ob, boolean ignoreZero) {
        return convertToString(ob);
    }

    /**
     * Converts an Object to String using toString().
     *
     * @param o the object to convert
     * @return the converted string object
     */
    @SuppressWarnings("deprecation")
    static public String convertToString(Object o) {
        if (o == null) {
            return "";
        }
        if (o instanceof Date) {
            return String.valueOf(CustomDate.getDateFormatter().format(((Date) o)));
        }
        return String.valueOf(o);
    }

    /**
     * Converts an int to a String. Flag decides if zero should be ""
     *
     * @param i The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(int i, boolean ignoreZero) {
        if (ignoreZero) {
            return convertToString(i);
        }
        return String.valueOf(i);
    }

    /**
     * Converts an int to a String. Zero is set to ""
     *
     * @param i The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(int i) {
        if (i == 0) {
            return "";
        }
        return String.valueOf(i);
    }

    /**
     * Converts a boolean to a String.
     *
     * @param b The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(boolean b) {
        return String.valueOf(b);
    }

    /**
     * Converts a boolean to a String.
     *
     * @param b The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(boolean b, boolean ignoreZero) {
        return convertToString(b);
    }

    /**
     * Converts a char to a String.
     *
     * @param c The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(char c) {
        return String.valueOf(c);
    }

    /**
     * Converts a char to a String.
     *
     * @param c The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(char c, boolean ignoreZero) {
        return convertToString(c);
    }

    /**
     * Converts a float to a String. 0 is returned as "".
     *
     * @param f The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(float f) {
        if (f == 0) {
            return "";
        }
        return String.valueOf(f);
    }

    /**
     * Converts a float to a String. Flag decides if zero should be ""
     *
     * @param f The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(float f, boolean ignoreZero) {
        if (ignoreZero) {
            return convertToString(f);
        }
        return String.valueOf(f);
    }

    /**
     * Converts a double to a String. 0 is returned as "".
     *
     * @param d The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(double d) {
        if (d == 0) {
            return "";
        }
        return String.valueOf(d);
    }

    /**
     * Converts a double to a String. Flag decides if zero should be ""
     *
     * @param d The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(double d, boolean ignoreZero) {
        if (ignoreZero) {
            return convertToString(d);
        }
        return String.valueOf(d);
    }

    /**
     * Converts a long to a String. 0 is returned as "".
     *
     * @param l The primitive to convert
     * @return The converted primitive
     */
    static public String convertToString(long l) {
        if (l == 0) {
            return "";
        }
        return String.valueOf(l);
    }

    /**
     * Converts a long to a String.
     *
     * @param l The primitive to convert
     * @param ignoreZero The ignore zero flag
     * @return the converted primitive
     */
    static public String convertToString(long l, boolean ignoreZero) {
        if (ignoreZero) {
            return convertToString(l);
        }
        return String.valueOf(l);
    }

    /**
     * Converts a Boolean to a boolean. The method is null safe, that is returns
     * false if b is null.
     *
     * @param b An Boolean object
     * @return The boolean value held by b
     */
    static public boolean convertTobool(Boolean b) {
        return (b == null) ? false : b.booleanValue();
    }

    /**
     * Converts a boolean to a boolean. No convertion is needed, so the
     * primitive boolean value is return unchanged.
     *
     * @param b An boolean
     * @return The boolean value held by b
     */
    static public boolean convertTobool(boolean b) {
        return b;
    }

    /**
     * Dummy method - returns the specified int
     *
     * @param n the int to convert
     * @return the int
     */
    static public int convertToInt(int n) {
        return n;
    }

    /**
     * Converts the specified argument to an int.
     *
     * @param arg the argument to convert to an int
     * @return an int
     */
    static public int convertToInt(String arg) {
        if (arg == null) {
            return 0;
        }
        return Integer.parseInt(arg);
    }

    /**
     * Converts the specified argument to an int.
     *
     * @param arg the argument to convert to an int
     * @return an int
     */
    static public int convertToInt(G9Enumerator arg) {
        if (arg == null) {
            return 0;
        }
        return arg.currentValue;
    }

    /**
     * Converts the specified argument to an int.
     *
     * @param arg the argument to convert to an int
     * @return an int
     */
    static public int convertToInt(Numeric arg) {
        if (arg == null) {
            return 0;
        }
        return arg.intValue();
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public byte safeCopy(byte field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public boolean safeCopy(boolean field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public char safeCopy(char field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public short safeCopy(short field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public int safeCopy(int field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public long safeCopy(long field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public float safeCopy(float field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public double safeCopy(double field) {
        return field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public String safeCopy(String field) {
        return field == null ? null : field;
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public Date safeCopy(Date field) {
        return field == null ? null : new Date(field.getTime());
    }

    /**
     * Used to make sure that a field is safely copied.
     *
     * @param field the value to copy
     * @return the copied value
     * @see #safeCopy(String f)
     */
    static public G9Enumerator safeCopy(G9Enumerator field) {
        if (field == null) {
            return null;
        }
        String msgID = null;
        Exception ex = null;
        G9Enumerator g = null;
        try {
            g = field.getClass().newInstance();
            g.currentValue = field.currentValue;
            return g;
        } catch (IllegalAccessException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } catch (InstantiationException e) {
            ex = e;
            msgID = CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD;
        } finally {
            if (ex != null) {
                Object[] msgArgs = { TypeTool.class.getClass(), field,
                        field.getClass(), ex.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(msgID, msgArgs);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9ClientFrameworkException(ex, msg);
            }
        }
        return null;
    }

    /**
     * Used to make sure that a field is safely copied. This one makes a new
     * copy of the Numeric field.
     *
     * @param field the value to copy
     * @return the copied value
     */
    static public Numeric safeCopy(Numeric field) {
        if (field != null) {
            field = (Numeric) (field.clone());
        }
        return field;
    }

    /**
     * A breakpoint.
     *
     * @deprecated Since 8.2.1.b
     */
    @Deprecated
    static public void breakpoint() {
        System.out.println("breakpoint");
    }

    /**
     * Regular equals. Compares two integer numbers and returns true if they are
     * equal.
     *
     * @param a the first number to compare
     * @param b the second number to compare
     * @return a == b
     */
    public static boolean equals(long a, long b) {
        return a == b;
    }

    /**
     * Regular equals. Compares two numbers and returns true if they are equal
     *
     * @param a the first number to compare
     * @param b the second number to compare
     * @return a == b
     */
    public static boolean equals(double a, double b) {
        return a == b;
    }

    /**
     * Regular equals. Compares two booleans and returns true if they are equal
     *
     * @param a the first boolean to compare
     * @param b the second boolean to compare
     * @return a == b
     */
    public static boolean equals(boolean a, boolean b) {
        return a == b;
    }

    /**
     * Null safe equals. Compares two object references for equality. The
     * alogoritm is as follows: <code>
     * if (a == null &amp;&amp; b == null) {
     * 		return true;
     * 	} else {
     * 		return a != null &amp;&amp; a.equals(b);
     * 	}
     *</code>
     *
     * @param a the first object to compare
     * @param b the second object to compare
     * @return true if both object are null or they are equal.
     */
    public static boolean equals(Object a, Object b) {
        if (a == null) {
            return b == null;
        }
        return a.equals(b);
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(int a, int b) {
        return a - b;
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(boolean a, boolean b) {
        if (a == b) {
            return 0;
        }
        if (a) {
            return -1;
        }
        if (b) {
            return 1;
        }
        return 0;
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(long a, long b) {
        return (int) (a - b);
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(char a, char b) {
        return (a - b);
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(float a, float b) {
        return (int) (a - b);
    }

    /**
     * Regular compare to.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    static public int compareTo(double a, double b) {
        return (int) (a - b);
    }

    /**
     * null safe compareTo.
     *
     * @param a A
     * @param b B
     * @return a - b
     */
    // @SuppressWarnings("unchecked")
    static public int compareTo(Object a, Object b) {
        if (a == null) {
            if (b == null) {
                return 0;
            }
            return -1;
        }
        if (b == null) {
            return 1;
        }
        if (a instanceof String && b instanceof String) {
            if (collator == null) {
                collator = Collator.getInstance();
                Locale locale = Locale.getDefault();
                if (isVmVersion1_4()
                        && collator instanceof RuleBasedCollator
                        && (locale.getLanguage().equals(
                                new Locale("no").getLanguage()) || locale
                                .getLanguage().equals(
                                        new Locale("da").getLanguage()))) {
                    String r = ((RuleBasedCollator) collator).getRules();
                    r = r.concat("&v,V<w,W");
                    try {
                        collator = new RuleBasedCollator(r);
                        collator.setDecomposition(Collator.NO_DECOMPOSITION);
                    } catch (ParseException e) {
                        collator = Collator.getInstance();
                    }
                }
            }
            String aLow = ((String) a).toLowerCase();
            String bLow = ((String) b).toLowerCase();

            return collator.compare(aLow, bLow);
        }
        if (a instanceof Date && b instanceof Date) {
            return ((Date) a).compareTo((Date) b);
        }

        if (a instanceof Boolean && b instanceof Boolean) {
            return TypeTool.compareTo(((Boolean) a).booleanValue(),
                    ((Boolean) b).booleanValue());
        }

        return ((Comparable) a).compareTo(b);
    }

    /**
     * Check if the field is an enumerator
     *
     * @param owner the object with the field
     * @param field the name of the field
     * @return <code>true</code> if the field is an enumerator.
     */
    static public boolean isEnumeratorMethod(Object owner, String field) {
    	String capAttribute = field.substring(0, 1).toUpperCase()
    			+ field.substring(1);
        try {
            Method m = owner.getClass().getMethod("get" + capAttribute,
                    (Class[]) null);
            return m.getReturnType().getSuperclass() == G9Enumerator.class || m.getReturnType().isEnum();
        } catch (NoSuchMethodException e) {
        	// Try "is"
        	try {
        		Method m = owner.getClass().getMethod("is" + capAttribute,
        				(Class[]) null);
        		return m.getReturnType().getSuperclass() == G9Enumerator.class || m.getReturnType().isEnum();
        	} catch (NoSuchMethodException e2) {
        		Object[] msgArgs = { TypeTool.class.getClass(), field,
        				owner.getClass(), e.getMessage() };
        		Message msg = MessageSystem.getMessageFactory().getMessage(
        				CRuntimeMsg.CF_UNABLE_TO_ACCESS_FIELD_OR_METHOD, msgArgs);
        		MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        		throw new G9ClientFrameworkException(e, msg);
        	}
        }
    }

    /**
     * Dummy method, checks if the specified field is an enumerator. Will always
     * returne false.
     *
     * @param a ignored
     * @param field ignored
     * @return false
     */
    static public boolean isEnumeratorMethod(int a, String field) {
        return false;
    }

    /**
     * Dummy method, checks if the specified field is an enumerator. Will always
     * returne false.
     *
     * @param a ignored
     * @param field ignored
     * @return false
     */
    static public boolean isEnumeratorMethod(boolean a, String field) {
        return false;
    }

    /**
     * Dummy method, checks if the specified field is an enumerator. Will always
     * returne false.
     *
     * @param a ignored
     * @param field ignored
     * @return false
     */
    static public boolean isEnumeratorMethod(char a, String field) {
        return false;
    }

    /**
     * Check if the <code>owner</code> is an enumerator.
     *
     * @param owner the object to check
     * @param field ignored
     * @return <code>true</code> if <code>owner</code> is an instance of
     *         G9Enumerator.
     */
    static public boolean isEnumerator(Object owner, String field) {
        return owner instanceof G9Enumerator || owner.getClass().isEnum();
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(int a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(boolean a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(long a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(float a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(char a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return false.
     *
     * @param a ignored.
     * @param field ignored.
     * @return false
     */
    static public boolean isEnumerator(byte a, String field) {
        return false;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, int i) {
        return true;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, float i) {
        return true;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, double i) {
        return true;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, byte i) {
        return true;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, char i) {
        return true;
    }

    /**
     * Dummy method. Will always return <code>true</code>
     *
     * @param t ignored.
     * @param i ignored
     * @return true
     */
    static public boolean isConsistant(String t, boolean i) {
        return true;
    }

    /**
     * Check if the specified string represents a valid date.
     *
     * @param t a string representation of a date
     * @param d a date
     * @return <code>true</code> if <code>t</code> can be parsed to a date.
     */
    static public boolean isConsistant(String t, java.util.Date d) {
        return convert(t, d) != null;
    }

    /**
     * Dummy method. Will always return true.
     *
     * @param t ingored.
     * @param o ignored.
     * @return true
     */
    static public boolean isConsistant(String t, Object o) {
        return true;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return true
     */
    static public boolean isBoolean(boolean x) {
        return true;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(double x) {
        return false;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(String x) {
        return false;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(int x) {
        return false;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(float x) {
        return false;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(long x) {
        return false;
    }

    /**
     * Check if the specified date is a boolean.
     *
     * @param x the date to check
     * @return false
     */
    static public boolean isBoolean(Date x) {
        return false;
    }

    /**
     * Check if the specified primitive is a boolean.
     *
     * @param x the primitive to check
     * @return false
     */
    static public boolean isBoolean(char x) {
        return false;
    }

    /**
     * Check if the specified object is a boolean.
     *
     * @param x the object to check
     * @return false
     */
    static public boolean isBoolean(Object x) {
        return false;
    }

    /**
     * Returns true if the string is not null and not empty.
     *
     * @param t the string to check
     * @return <code>true</code> if the string is not null or empty.
     */
    static public boolean hasValue(String t) {
        return t != null && !t.equals("");
    }

    /**
     * Checks if the date field is assigned to a Date object.
     *
     * @param d the date to check
     * @return <code>true</code> if the date is not null.
     */
    static public boolean hasValue(Date d) {
        return d != null;
    }

    /**
     * Checks if the Numeric field is assigned to a Numeric object.
     *
     * @param n the numeric to check
     * @return <code>true</code> if the numeric is not null and not zero.
     */
    static public boolean hasValue(Numeric n) {
        return n != null && n.compareTo(0) != 0;
    }

    /**
     * Returns true if the G9Enumeator is not null and has a different value
     * than 0.
     *
     * @param e the enumerator to check
     * @return <code>true</code> if not null and not zero.
     */
    static public boolean hasValue(G9Enumerator e) {
        return e != null && e.currentValue != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(int primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero (always true for boolean)
     *
     * @param primitive the primitive to check
     * @return <code>true</code>.
     */
    static public boolean hasValue(boolean primitive) {
        return true;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(float primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(double primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(long primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(char primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(short primitive) {
        return primitive != 0;
    }

    /**
     * Check that the specified primitive is not zero.
     *
     * @param primitive the primitive to check
     * @return <code>true</code> if the primitive is not zero.
     */
    static public boolean hasValue(byte primitive) {
        return primitive != 0;
    }

    /**
     * Returns a list with unique elements based on the GenovaComparator.
     *
     * @param l the list of elements
     * @param comp the comparator specifying when elements are equal.
     * @return a list of unique elements.
     * @deprecated Since 8.2.1.b
     */
    @Deprecated
    static public List uniqueList(List l, G9Comparator comp) {
        if (l == null || l.size() == 0) {
            return l;
        }
        Collections.sort(l, comp);
        Vector retVal = new Vector();
        Object current;
        Iterator i = l.iterator();
        current = i.next();
        retVal.add(current);
        while (i.hasNext()) {
            Object next = i.next();
            if (!comp.equals(current, next)) {
                retVal.add(next);
                current = next;
            }
        }
        return retVal;
    }

    /**
     * Formats the specified string according to CustomDate.superDateFormat.
     *
     * @param d the string to format
     * @return the formatted string.
     */
    public static String superDateFormat(String d) {
        return CustomDate.superDateFormat(d);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param i (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(int i) {
        return Integer.valueOf(i);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param l (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(long l) {
        return Long.valueOf(l);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param b (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(byte b) {
        return Byte.valueOf(b);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param s (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(short s) {
        return Short.valueOf(s);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param b (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(boolean b) {
        return Boolean.valueOf(b);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param d (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(double d) {
        return Double.valueOf(d);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param f (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(float f) {
        return Float.valueOf(f);
    }

    /**
     * Convert a primary type to wrapper type.
     *
     * @param c (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(char c) {
        return Character.valueOf(c);
    }

    /**
     * Just return the object...
     *
     * @param o (missing javadoc)
     * @return new wrapper object
     */
    public static Object toObject(Object o) {
        return o;
    }

    /**
     * Construct a JavaBean style getter method name from the given field name.
     *
     * @param fieldName (missing javadoc)
     * @return "get&lt;Property&gt;"
     */
    public static String asBeanGetter(String fieldName) {
        StringBuffer methodName = new StringBuffer(fieldName);
        methodName.setCharAt(0, Character.toUpperCase(methodName.charAt(0)));
        return "get" + methodName.toString();
    }

    /**
     * Construct a JavaBean style getter method name from the given field name, boolean "is" style.
     *
     * @param fieldName (missing javadoc)
     * @return "is&lt;Property&gt;"
     */
    public static String asBeanBooleanGetter(String fieldName) {
        StringBuffer methodName = new StringBuffer(fieldName);
        methodName.setCharAt(0, Character.toUpperCase(methodName.charAt(0)));
        return "is" + methodName.toString();
    }

    /**
     * Construct a JavaBean style setter method name from the given field name.
     *
     * @param fieldName (missing javadoc)
     * @return "set&lt;Property&gt;"
     */
    public static String asBeanSetter(String fieldName) {
        StringBuffer methodName = new StringBuffer(fieldName);
        methodName.setCharAt(0, Character.toUpperCase(methodName.charAt(0)));
        return "set" + methodName.toString();
    }

    /**
     * Convenience method, checks if the specified String is either empty (""),
     * or consists entirely of blanks (" "), or is a null-pointer, in which case
     * it returns <code>true</code>
     *
     * @param s the string to check
     * @return <code>true</code> if the string is either empty or
     *         <code>null</code>
     */
    public static boolean isEmpty(String s) {
        return s == null || s.trim().length() == 0;
    }

    /**
     * Strip trailing whitespace from end of string.
     *
     * @param str input string
     * @return a new string with trailing whitespace removed, or input string if
     *         null, empty or without trailing whitespace
     */
    public static String stripEnd(final String str) {
        if (str == null || str.length() == 0
                || !Character.isWhitespace(str.charAt(str.length() - 1))) {
            return str;
        }
        char[] cs = str.toCharArray();
        int counter;
        for (counter = cs.length - 2; counter >= 0
                && Character.isWhitespace(cs[counter]); --counter) {
            /* emtpy */
        }
        if (counter == -1) {
            return "";
        }
        return new String(cs, 0, counter + 1);
    }

    /**
     * Returns the role name prefix of the specified String. If
     * <code>roleAttributeName</code> contais a '.', the substring from index 0
     * to the index of the '.' is returned. Otherwise the returned value is
     * <code>null</code>.
     * <p>
     * Examples: <code>getRoleNamePrefix("foo.bar");</code> returns "foo". <br>
     * <code>getRoleNamePrefix("foobar");</code> returns <code>null</code>.<br>
     * <code>getRoleNamePrefix(".bar"); </code> returns "".<br>
     *
     * @param roleAttributeName the string specifying an attribute and possibly
     *            a role name.
     * @return the role name
     */
    public static String getRoleNamePrefix(String roleAttributeName) {
        if (roleAttributeName == null)
            return null;
        int dotIndex = roleAttributeName.indexOf('.');
        if (dotIndex >= 0) {
            return roleAttributeName.substring(0, dotIndex);
        }
        return null;

    }

    /**
     * Removes the specified role name prefix from the supplied attribute
     * string. The role name prefix is only removed if the <code>attr</code>
     * string starts with the string and the first character after the string is
     * a '.'. Thus <code>stripRoleNamePrefix("foo.bar", "foo")</code> returns
     * "bar" <code>stripRoleNamePrefix("foox.bar", "foo"</code> returns
     * "foox.bar"
     *
     * @param attr the string with attribute name and possibly a role name
     * @param roleName the role name to strip from the <code>attr</code> string
     * @return the name of the attribute without the role name prefix.
     */
    public static String stripRoleNamePrefix(String attr, String roleName) {
        if (attr != null && attr.startsWith(roleName)
                && attr.length() > roleName.length() + 1
                && attr.charAt(roleName.length()) == '.') {
            return attr.substring(roleName.length() + 1);
        }
        return attr;
    }

    /**
     * Adds the specified roleName as prefix to the attribute name. If the
     * attributeName is already prefixed with the roleName the attributeName is
     * returned unaltered. Else it is prefixed with the role name and returned.
     *
     * @param attributeName the name of the attribute.
     * @param roleName the name of the role.
     * @return the prefixed attribute name.
     */
    public static String addRoleNamePrefix(String attributeName, String roleName) {
        if (!roleName.equals(TypeTool.getRoleNamePrefix(attributeName))) {
            attributeName = roleName + "." + attributeName;
        }
        return attributeName;
    }

    /**
     * Check if the two value differs. This method does not rely upon the
     * symmetry contract of equals, since it is broken in at least one common
     * case (timestamp vs. date). Both values might be null (considered not
     * different).
     *
     * @param value1 the first value to compare
     * @param value2 the second value to compare
     * @return <code>true</code> if the values differ.
     */
    public final static boolean differ(Object value1, Object value2) {
        if (value1 == null) {
            return value2 != null;
        }
        if (value2 == null) {
            return true;
        }
        // Here we know none of them are null
        return !(value1.equals(value2) || value2.equals(value1));
    }

    /**
     * Check if the two field values differs. This method is similar to
     * {@link #differ(Object, Object)}, but with special treatment of Strings:
     * Trailing blanks are ignored, and an empty string is considered
     * equal to a <code>null</code> string.
     *
     * @param fieldValue1 the first value to compare
     * @param fieldValue2 the second value to compare
     * @return <code>true</code> if the values differ.
     */
    public final static boolean viewFieldDiffer(Object fieldValue1,
            Object fieldValue2) {
        boolean differ = differ(fieldValue1, fieldValue2);
        if (differ) {
            boolean stringComparison = false;
            String s1 = "", s2 = "";
            if (fieldValue1 instanceof String) {
                s1 = stripEnd((String) fieldValue1);
                stringComparison = true;
            }
            if (fieldValue2 instanceof String) {
                s2 = stripEnd((String) fieldValue2);
                stringComparison = true;
            }
            if (stringComparison) {
                return !s1.equals(s2);
            }
            boolean boolComparison = false;
            Boolean b1 = Boolean.FALSE, b2 = Boolean.FALSE;
            if (fieldValue1 instanceof Boolean) {
                b1 = (Boolean) fieldValue1;
                boolComparison = true;
            }
            if (fieldValue2 instanceof Boolean) {
                b2 = (Boolean) fieldValue2;
                boolComparison = true;
            }
            if (boolComparison) {
                return !b1.equals(b2);
            }
        }
        return differ;
    }

}
