/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.convert;

import java.util.HashMap;


/**
 * A container that holds converters, and that returns a converter on demand.
 *
 * All converters are assosiated with an attribute name, but they can be
 * registered in 4 levels; role, object selection, class and global. The
 * registry will always return registered converter of the highest level
 * (when we have more than one match).
 *
 * Global converters may be returned for an attribute if they were registered
 * with that attribute's name.
 * Class converters may be returned for an attribute if they were registered
 * with the same class name, and with the same sttribute name.
 * Os-level converters may be returned for an atribute if they were registered
 * for the object selection the attribute has as context.
 * A role-level converter will be returned for an attribute if it was
 * registered with that attribute's name, and the name of the role the
 * attribute is in context of.
 *
 */
public class ConverterRegistry {

    private ValueConverter defaultConverter;

    /** Maps attribute names (String) to a ConvertersForOneAttribute */
    private HashMap<String, ConvertersForOneAttribute> convertersForAttributes;

    private static class ConvertersForOneAttribute {
        public HashMap<String, ValueConverter> roleConverters = new HashMap<String, ValueConverter>();
        public HashMap<String, ValueConverter> classConverters = new HashMap<String, ValueConverter>();
        public HashMap<String, ValueConverter> osConverters = new HashMap<String, ValueConverter>();
        public ValueConverter globalConverter = null;
    }

    /**
     * Default constructor
     */
    public ConverterRegistry() {
        convertersForAttributes = new HashMap<String, ConvertersForOneAttribute>();
        defaultConverter = new SimpleValueConverter();
    }

    /**
     * Constructs a new regustry and sets its default converter
     *
     * @param defaultConverter (missing javadoc)
     */
    public ConverterRegistry(ValueConverter defaultConverter) {
        convertersForAttributes = new HashMap<String, ConvertersForOneAttribute>();
        this.defaultConverter = defaultConverter;
    }

    /**
     * Registers a ValueConverter for an attribute for a given role name.
     *
     * @param attributeName (missing javadoc)
     * @param roleName (missing javadoc)
     * @param converter (missing javadoc)
     * @return this
     */
    public synchronized ConverterRegistry registerRoleConverterForAttribute(String attributeName, String roleName, ValueConverter converter) {
        ConvertersForOneAttribute c = convertersForAttributes.get(attributeName);
        if (c == null) {
            c = new ConvertersForOneAttribute();
            convertersForAttributes.put(attributeName, c);
        }
        c.roleConverters.put(roleName, converter);
        return this;
    }

    /**
     * Registers a ValueConverter for an attribute for a given object selection
     * name.
     *
     * @param attributeName (missing javadoc)
     * @param osName (missing javadoc)
     * @param converter (missing javadoc)
     * @return this (missing javadoc)
     */
    public synchronized ConverterRegistry registerOsConverterForAttribute(String attributeName, String osName, ValueConverter converter) {
        ConvertersForOneAttribute c = convertersForAttributes.get(attributeName);
        if (c == null) {
            c = new ConvertersForOneAttribute();
            convertersForAttributes.put(attributeName, c);
        }
        c.osConverters.put(osName, converter);
        return this;
    }

    /**
     * Registers a ValueConverter for an attribute for a given class name.
     *
     * @param attributeName (missing javadoc)
     * @param className (missing javadoc)
     * @param converter (missing javadoc)
     * @return this
     */
    public synchronized ConverterRegistry registerClassConverterForAttribute(String attributeName, String className, ValueConverter converter) {
        ConvertersForOneAttribute c = convertersForAttributes.get(attributeName);
        if (c == null) {
            c = new ConvertersForOneAttribute();
            convertersForAttributes.put(attributeName, c);
        }
        c.classConverters.put(className, converter);
        return this;
    }

    /**
     * Registers a ValueConverter for an attribute name, regardless of os, role
     * or class name
     *
     * @param attributeName (missing javadoc)
     * @param converter (missing javadoc)
     * @return this (missing javadoc)
     */
    public synchronized ConverterRegistry registerGlobalConverterForAttribute(String attributeName, ValueConverter converter) {
        ConvertersForOneAttribute c = convertersForAttributes.get(attributeName);
        if (c == null) {
            c = new ConvertersForOneAttribute();
            convertersForAttributes.put(attributeName, c);
        }
        c.globalConverter = converter;
        return this;
    }

    /**
     * Returns a ValueConverter for a given context, or null if none is
     * registered for that context.
     *
     * @param context (missing javadoc)
     * @return A ValueConverter for the given context, or null if none is
     *         registered for that context.
     */
    public synchronized ValueConverter getConverter(OSConvertContext context) {
        ConvertersForOneAttribute c = convertersForAttributes.get(context.getAttributeName());
        if (c == null) return defaultConverter;
        if (c.roleConverters.containsKey(context.getRoleName()))
            return c.roleConverters.get(context.getRoleName());
        if (c.osConverters.containsKey(context.getObjectSelectionName()))
            return c.osConverters.get(context.getObjectSelectionName());
        if (context.getDomainObject() != null) {
            String className = context.getDomainObject().getClass().getName();
            if (c.classConverters.containsKey(className)) return c.classConverters.get(className);
        }
        if (c.globalConverter != null) return c.globalConverter;
        return defaultConverter;
    }

    /**
     * Sets the default value converter
     *
     * @param defaultConverter (missing javadoc)
     */
    public synchronized void setDefaultValueConverter(ValueConverter defaultConverter) {
        this.defaultConverter = defaultConverter;
    }

}
