/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.filter;

import java.util.HashSet;
import java.util.Set;


/**
 * A simple filter that filters roles based on their names only, and attributes
 * based on their name and the role they belong to.
 *
 */
public class SimpleFilter implements AttributeFilter, RoleFilter {

    /** Sets of role names to disqualify */
    protected HashSet<String> excludedRoleNames;

    /** Set of attribute names to disqualify */
    protected HashSet<String> excludedAttributeNames;

    /**
     * Constructs a new {@link SimpleFilter}
     */
    public SimpleFilter() {
        excludedRoleNames = new HashSet<String>();
        excludedAttributeNames = new HashSet<String>();
    }

    @Override
    public boolean isRoleIncluded(String roleName, Object domainObject) {
        return !excludedRoleNames.contains(roleName);
    }

    @Override
    public boolean isAttributeIncluded(String attributeName, Object domainObject) {
        return !excludedAttributeNames.contains(attributeName);
    }

    /**
     * Adds one excluded name to the list of excluded role names.
     *
     * @param roleName (missing javadoc)
     * @return this
     */
    public SimpleFilter addExcludedRoleName(String roleName) {
        excludedRoleNames.add(roleName);
        return this;
    }

    /**
     * Adds a set of names to the list of excluded role names.
     *
     * @param roleName (missing javadoc)
     * @return this
     */
    public SimpleFilter addExcludedRoleNames(Set<String> roleName) {
        excludedRoleNames.addAll(roleName);
        return this;
    }

    /**
     * Adds excluded role names, given as an array of Strings
     *
     * @param roleNames (missing javadoc)
     * @return this
     */
    public SimpleFilter addExcludedRoleNames(String[] roleNames) {
        for (String s : roleNames) addExcludedRoleName(s);
        return this;
    }

    /**
     * Adds one excluded name to the list of excluded attribute names.
     *
     * @param attributeName The attribute to exclude, given as "role.attribute"
     * @return this
     */
    public SimpleFilter addExcludedAttributeName(String attributeName) {
        excludedAttributeNames.add(attributeName);
        return this;
    }

    /**
     * Adds a set of names to the list of excluded attribute names.
     *
     * @param attributeNames The attribute to exclude, given as a Set of Strings
     *            "role.attribute"
     * @return this
     */
    public SimpleFilter addExcludedAttributeNames(Set<String> attributeNames) {
        excludedAttributeNames.addAll(attributeNames);
        return this;
    }

    /**
     * Adds excluded attribute names, given as an array of Strings
     *
     * @param attributeNames The attribute to exclude, given as an array of
     *            Strings "role.attribute"
     */
    public void addExcludedAttributeNames(String[] attributeNames) {
        for (String s : attributeNames) addExcludedAttributeName(s);
    }

}
