/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.transport;

import java.util.Collection;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;

import no.esito.log.Logger;
import no.g9.os.AssociationRoleConstant;
import no.g9.os.OSRole;
import no.g9.os.RelationCardinality;
import no.g9.os.RoleConstant;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.PropertyAccessorUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.CollectionFactory;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.TypeDescriptor;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

/**
 * This class handles the conversion to and from domain and transport objects.
 * This conversion is a deep copy as described by the hierarchy in the supplied
 * <code>OSRole</code> root node.
 * <p>
 * The shallow copy (the object creation and transfer of attribute values) is
 * done by an implementation of @{link AbstractDomainTransportConverter} (one
 * converter per domain/transport-pair, generated by g9).
 *
 *
 */
@Service
public class DomainTransportConversionService implements ConversionService {

    /** the converter registry to use *. */
    private DomainTransportConverterRegistry domainTransportConverterRegistry;

    /*** Conversion service used for simple type conversion **/
    private ConversionService domainTransportTypeConversionService;

    private final static Logger logger = Logger.getLogger(DomainTransportConversionService.class);

    @Override
    public boolean canConvert(Class<?> sourceType, Class<?> targetType) {
        if (targetType == null) {
            throw new IllegalArgumentException("The targetType to convert to cannot be null");
        }
        return canConvert(sourceType != null ? TypeDescriptor.valueOf(sourceType) : null,
                TypeDescriptor.valueOf(targetType));
    }

    @Override
    public boolean canConvert(TypeDescriptor sourceType, TypeDescriptor targetType) {
        return domainTransportTypeConversionService.canConvert(sourceType, targetType)
                || domainTransportConverterRegistry.canConvert(sourceType, targetType);

    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T convert(final Object source, Class<T> targetType) {
        Assert.notNull(targetType, "The targetType to convert to cannot be null");
        return (T) convert(source, TypeDescriptor.forObject(source), TypeDescriptor.valueOf(targetType));
    }

    @Override
    public Object convert(final Object source, TypeDescriptor sourceType, TypeDescriptor targetType) {
        if (source == null) {
            return null;
        }

        if (source instanceof Collection<?>) {
            return convert((Collection<?>) source, targetType);

        }
        if (domainTransportTypeConversionService.canConvert(sourceType, targetType)) {
            // simple type conversion
            return domainTransportTypeConversionService.convert(source, sourceType, targetType);
        }
        DomainTransportTransfer<Object, Object> converter = getDomainTransportConverterRegistry().getConverter(
                sourceType, targetType);
        Assert.notNull(converter, "No converter registered for source '" + source.getClass() + "' and target '"
                + targetType + "'");
        OSRole<?> rootRole = converter.getOSRole();
        ConversionDirection conversionDirection = converter.getTransportType() == targetType.getType() ? ConversionDirection.DOMAIN2TRANSPORT
                : ConversionDirection.TRANSPORT2DOMAIN;
        Assert.notNull(rootRole, "rootRole must not be null");
        if (!rootRole.isRoot()) {
            throw new IllegalArgumentException("Only conversion of root roles is supported");
        }
        IdentityHashMap<Object, Object> processedObjects = new IdentityHashMap<Object, Object>();
        BeanWrapperImpl convertTo = new BeanWrapperImpl();
        BeanWrapperImpl convertFrom = new BeanWrapperImpl(source);
        convertNode(rootRole, convertFrom, convertTo, conversionDirection, processedObjects, "");
        return convertTo.getRootInstance();
    }

    /**
     * Convert a <code>Collection</code> of transport objects to a
     * <code>Collection</code> of domain objects.
     *
     * @param <T>
     *            the generic type of the transport class
     * @param <D>
     *            the generic type of the domain class
     * @param transportObjects
     *            the <code>Collection</code> of transport objects
     * @param rootRole
     *            the root role to use to navigate the object graph
     * @return the collection
     */
    @SuppressWarnings("unchecked")
    private <S, T> Collection<T> convert(final Collection<S> sourceCollection, TypeDescriptor itemTargetType) {
        if (sourceCollection == null) {
            return null;
        }
        Collection<T> targetCollection = CollectionFactory.createApproximateCollection(sourceCollection, 16);
        if (sourceCollection.isEmpty()) {
            return targetCollection;
        }
        Object firstElement = sourceCollection.iterator().next();
        Class<?> collectionType = sourceCollection.getClass();
        TypeDescriptor itemSourceType = TypeDescriptor.forObject(firstElement);
        TypeDescriptor sourceType = TypeDescriptor.collection(collectionType, itemSourceType);
        TypeDescriptor targetType = TypeDescriptor.collection(collectionType, itemTargetType);
        if (getDomainTransportTypeConversionService().canConvert(sourceType, targetType)) {
            return (Collection<T>) getDomainTransportTypeConversionService().convert(sourceCollection, sourceType,
                    targetType);

        }

        for (Iterator<S> iter = sourceCollection.iterator(); iter.hasNext();) {
            S sourceObject = iter.next();
            targetCollection.add((T) convert(sourceObject, itemSourceType, itemTargetType));
        }
        return targetCollection;
    }

    /**
     * Converts a node in the object graph if it has not already been converted.
     *
     * Uses an {@link IdentityHashMap} internally to keep track of converted
     * nodes.
     *
     * @see IdentityHashMap
     *
     * @param <S>
     *            the generic type of the source object
     * @param <T>
     *            the generic type of the target object
     * @param role
     *            the role to use for limiting the conversion
     * @param convertFrom
     *            the <code>BeanWrapper</code> holding the object to convert
     *            from
     * @param convertTo
     *            the <code>BeanWrapper</code> holding the object to convert to
     * @param conversionDirection
     *            the direction to convert, domain to transport or transport to
     *            domain
     * @param processedObjects
     *            a <code>IdentityHashMap</code> of already converted objects,
     *            handles cyclic dependencies.
     * @param propertyPath
     *            the the property path of the current object to convert, an
     *            empty String ("") if it is the root
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    private <S, T> void convertNode(OSRole<?> role, BeanWrapperImpl convertFrom, BeanWrapperImpl convertTo,
            ConversionDirection conversionDirection, IdentityHashMap<Object, Object> processedObjects,
            String propertyPath) {

        assertNotNull(role, convertFrom, convertTo, conversionDirection, propertyPath);

        Object fromObject = null;

        fromObject = role.isRoot() ? convertFrom.getRootInstance() : convertFrom.getPropertyValue(propertyPath);

        if (fromObject == null) {
            return;
        }

        Object processedObject = processedObjects.get(fromObject);

        if (processedObject != null) {
            // already converted higher up in the object graph
            convertTo.setPropertyValue(propertyPath, processedObject);
            return;
        }

        if (logger.isTraceEnabled()) {
            logger.trace("Started convertNode() for role: [" + role + "], propertyPath: ['" + propertyPath
                    + "]', isRoot: [" + role.isRoot() + "]");
        }

        Object shallowCopy = null;
        DomainTransportTransfer<S, T> converter = getConverterForRole(role);

        if (fromObject instanceof Collection<?>) {
            shallowCopy = handleCollection(role, conversionDirection, processedObjects, propertyPath,
                    (Collection) fromObject);
        } else {
            shallowCopy = createShallowCopy(conversionDirection, fromObject, converter);
        }

        processedObjects.put(fromObject, shallowCopy);

        if (role.isRoot()) {
            convertTo.setWrappedInstance(shallowCopy);
        } else {
            if (role.getRelationCardinality() == RelationCardinality.MANY) {
                Collection convertedCollection = (Collection) convertTo.getPropertyValue(propertyPath);
                if (shallowCopy != null && convertedCollection == null) {
                    convertedCollection = new HashSet();
                    convertTo.setPropertyValue(propertyPath, convertedCollection);
                }
                if (shallowCopy != null) {
                    convertedCollection.addAll((Collection) shallowCopy);
                }
            } else {
                convertTo.setPropertyValue(propertyPath, shallowCopy);
                handleIfNavigateToParent(convertFrom, processedObjects, convertTo, converter);
            }
        }
        handleIfParentOne(role, convertFrom, convertTo, conversionDirection, processedObjects);
        if (logger.isTraceEnabled()) {
            logger.trace("End convertNode() for role: [" + role + "], propertyPath: ['" + propertyPath
                    + "'], isRoot: [" + role.isRoot() + "]");
        }
    }

    private <S, T> void handleIfNavigateToParent(BeanWrapper convertFrom,
            IdentityHashMap<Object, Object> processedObjects, BeanWrapper convertTo,
            DomainTransportTransfer<S, T> converter) {
        if (converter.getParentAssociationRoleName() != null) {
            Object parent = convertFrom.getPropertyValue(converter.getParentAssociationRoleName());
            Object convertedParent = processedObjects.get(parent);
            if (convertedParent == null) {
                logger.warn("Parent association not found in object graph for property name "
                        + converter.getParentAssociationRoleName());
            } else {
                if (convertTo.isWritableProperty(converter.getParentAssociationRoleName())) {
                    convertTo.setPropertyValue(converter.getParentAssociationRoleName(), convertedParent);
                } else {
                    logger.warn(String.format(
                            "Cannot set navigation to parent, property '%s' isn't writeable on class '%s'",
                            converter.getParentAssociationRoleName(), convertTo.getWrappedClass()));
                }
            }
        }
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private <S, T> Object handleCollection(OSRole<?> role, ConversionDirection conversionDirection,
            IdentityHashMap<Object, Object> processedObjects, String propertyPath, Collection fromObject) {
        BeanWrapperImpl newWrapperFrom = new BeanWrapperImpl();
        BeanWrapperImpl newWrapperTo = new BeanWrapperImpl();
        DomainTransportTransfer<S, T> converter = getConverterForRole(role);
        Collection convertedCollection = null;
        if (fromObject.size() > 0) {
            convertedCollection = CollectionFactory.createApproximateCollection(fromObject, 16);
            for (Iterator iter = fromObject.iterator(); iter.hasNext();) {
                Object toConvert = iter.next();
                Object convertedObject = createShallowCopy(conversionDirection, toConvert, converter);
                convertedCollection.add(convertedObject);
                processedObjects.put(toConvert, convertedObject);
                newWrapperFrom.setWrappedInstance(toConvert);
                newWrapperTo.setWrappedInstance(convertedObject);
                handleIfNavigateToParent(newWrapperFrom, processedObjects, newWrapperTo, converter);
                handleIfParentMany(role, newWrapperFrom, newWrapperTo, conversionDirection, processedObjects);
            }

            if (logger.isTraceEnabled()) {
                logger.trace("Handled Collection for role: [" + role + "], propertyPath: ['" + propertyPath + "']");
            }
        }
        return convertedCollection;
    }

    @SuppressWarnings("unchecked")
    private <S, T> Object createShallowCopy(ConversionDirection conversionDirection, Object source,
            DomainTransportTransfer<S, T> converter) {
        Class<?> targetType = getTargetType(conversionDirection, converter);
        Object target = BeanUtils.instantiate(targetType);
        if (conversionDirection == ConversionDirection.DOMAIN2TRANSPORT) {
            converter.transferToTransport((S) source, (T) target);
        } else {
            converter.transferToDomain((T) source, (S) target);
        }

        return target;

    }

    /**
     * @param <S>
     *            The generic source type
     * @param <T>
     *            The generic target type
     * @param role
     *            the root role to do base the conversion with
     * @return the converter for shallow copy of <S> to and from <T>
     */
    <S, T> DomainTransportTransfer<S, T> getConverterForRole(OSRole<?> role) {
        DomainTransportTransfer<S, T> converter = getDomainTransportConverterRegistry().getConverter(role);
        Assert.notNull(converter, "Converter not found for role '" + role + "'");
        return converter;
    }

    private <S, T> Class<?> getTargetType(ConversionDirection conversionDirection,
            DomainTransportTransfer<S, T> converter) {
        return conversionDirection == ConversionDirection.DOMAIN2TRANSPORT ? converter.getTransportType() : converter
                .getDomainType();
    }

    private <D> void handleIfParentOne(OSRole<D> parentRole, BeanWrapperImpl convertFrom, BeanWrapperImpl convertTo,
            ConversionDirection direction, IdentityHashMap<Object, Object> processedObjects) {
        for (OSRole<?> childRole : parentRole.getChildren()) {
            if (!parentRole.isMany()) {
                String newPropertyPath = getFullPropertyPath(childRole);
                if (logger.isTraceEnabled()) {
                    logger.trace("handleIfParentOne() for role: [" + childRole + "] with propertyPath ['"
                            + newPropertyPath + "']");
                }
                convertNode(childRole, convertFrom, convertTo, direction, processedObjects, newPropertyPath);
            }
        }
    }

    private <D> void handleIfParentMany(OSRole<D> parentRole, BeanWrapperImpl convertFrom, BeanWrapperImpl convertTo,
            ConversionDirection direction, IdentityHashMap<Object, Object> processedObjects) {
        if (logger.isTraceEnabled()) {
            logger.trace("handleIfParentMany() for role: [" + parentRole + "]");
        }
        for (OSRole<?> childRole : parentRole.getChildren()) {
            if (logger.isTraceEnabled()) {
                logger.trace("handleIfParentMany() for role: [" + parentRole + "] for child " + childRole
                        + ": ParentIsMany: " + parentRole.isMany() + " IsNavigableToParent: "
                        + childRole.isNavigableToParent() + " IsUpRelated: " + childRole.isUpRelated());
            }
            if (parentRole.isMany() && (childRole.isNavigableToParent() || childRole.isUpRelated())) {
                String newPropertyPath = getLastPropertyPathElement(childRole);
                if (logger.isTraceEnabled()) {
                    logger.trace("handleIfParentMany(), handling property for child role: [" + childRole
                            + "] with propertyPath ['" + newPropertyPath + "']");
                }
                convertNode(childRole, convertFrom, convertTo, direction, processedObjects, newPropertyPath);
            }
        }
    }

    /**
     * Gets the full property path for a given role.
     *
     * @param oSRole
     *            the role to calculate the property path from
     * @return the full property path represented by this role
     */
    String getFullPropertyPath(OSRole<?> oSRole) {
        Assert.isInstanceOf(AssociationRoleConstant.class, oSRole.getRoleConstant());
        AssociationRoleConstant roleConstant = (AssociationRoleConstant) oSRole.getRoleConstant();
        String fullPropertyPath = null;
        if (oSRole.getParent().isRoot()) {
            fullPropertyPath = String.valueOf(roleConstant.getAssociationRoleName());
        } else if (oSRole.isMany()) {
            fullPropertyPath = getFullPropertyPath(oSRole.getParent()) + "." + roleConstant.getAssociationRoleName();
        } else {
            StringBuilder sb = new StringBuilder();
            Iterator<RoleConstant> it = oSRole.getRoleConstant().getRolePath().iterator();
            while (it.hasNext()) {
                AssociationRoleConstant rc = (AssociationRoleConstant) it.next();
                sb.append(rc.getAssociationRoleName());
                if (it.hasNext()) {
                    sb.append(".");
                }
            }
            fullPropertyPath = sb.toString();
        }
        return fullPropertyPath;
    }

    /**
     * Gets the last property path element.
     *
     * @param oSRole
     *            the role to map
     * @return the last property path element Given "BeanA.BeanB.BeanD" this
     *         should return "BeanD"
     */
    String getLastPropertyPathElement(OSRole<?> oSRole) {
        String fullPath = getFullPropertyPath(oSRole);
        return fullPath.substring(PropertyAccessorUtils.getLastNestedPropertySeparatorIndex(fullPath) + 1);
    }

    /**
     * Gets the registry that contains all converters for the transport and
     * domain types.
     *
     * @return the domain transport converter registry
     */
    DomainTransportConverterRegistry getDomainTransportConverterRegistry() {
        return domainTransportConverterRegistry;
    }

    /**
     * Set the registry that contains all converters for the transport and
     * domain types.
     *
     * @param domainTransportConverterRegistry
     *            the new domain transport converter registry
     */
    @Autowired
    @Required
    public void setDomainTransportConverterRegistry(DomainTransportConverterRegistry domainTransportConverterRegistry) {
        this.domainTransportConverterRegistry = domainTransportConverterRegistry;
    }

    /**
     * Gets the domain transport type conversion service.
     *
     * @return the domain transport type conversion service
     */
    ConversionService getDomainTransportTypeConversionService() {
        return domainTransportTypeConversionService;
    }

    /**
     * Sets the domain transport type conversion service.
     *
     * @param domainTransportTypeConversionService
     *            the new domain transport type conversion service
     */
    @Autowired
    @Required
    public void setDomainTransportTypeConversionService(ConversionService domainTransportTypeConversionService) {
        this.domainTransportTypeConversionService = domainTransportTypeConversionService;
    }

    /**
     * Internal enum used to mark conversion direction.
     */
    private enum ConversionDirection {

        DOMAIN2TRANSPORT, TRANSPORT2DOMAIN
    }

    private void assertNotNull(OSRole<?> role, BeanWrapperImpl convertFrom, BeanWrapperImpl convertTo,
            ConversionDirection direction, String associationPath) {
        Assert.notNull(role, "role must not be null");
        Assert.notNull(convertFrom, "convertFrom must not be null");
        Assert.notNull(convertTo, "convertTo must not be null");
        Assert.notNull(direction, "direction must not be null");
        Assert.notNull(associationPath, "associationPath must not be null");
    }

}
