/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.ws;

import java.util.Collection;

import javax.annotation.Resource;

import no.esito.log.Logger;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.convert.ConversionService;
import org.springframework.ws.client.core.WebServiceTemplate;

/**
 * Abstract template for the generated Spring WS Client implementations.
 * 
 */
public abstract class AbstractWSClient implements InitializingBean {

    private final static Logger logger = Logger.getLogger(AbstractWSClient.class);

    /** For web service invocation **/
    private WebServiceTemplate webServiceTemplate;

    /*** Conversion service used for conversion to/from domain types etc. **/
    private ConversionService domainTransportConversionService;

    /*** The web service URI to connect to **/
    private String webServiceURI;

    /*** Root URI for the service to connect to **/
    private String webServiceRootURI;

    /**
     * Sends a web service message that contains the given payload, marshalled
     * by the configured Marshaller. Returns the unmarshalled payload of the
     * response message, if any.
     * 
     * @param requestPayload
     *            the request payload
     * @return the response payload
     */
    protected Object marshalSendAndReceive(Object requestPayload) {
        return webServiceTemplate.marshalSendAndReceive(this.webServiceURI, requestPayload);
    }

    /**
     * Get the name of this service. This will be the camel cased intereface
     * name.
     * 
     * @return the current service name
     */
    protected abstract String getServiceRequestName();

    /**
     * @param webServiceTemplate
     *            the <code>WebServiceTemplate</code> to use
     */
    @Required
    @Autowired
    public void setWebServiceTemplate(WebServiceTemplate webServiceTemplate) {
        this.webServiceTemplate = webServiceTemplate;
    }

    /**
     * Convenience method that converts the <code>source</code> to
     * <code>targetType</code> .
     * 
     * @param <C>
     *            the generic target type
     * @param source
     *            the source to convert
     * @param targetType
     *            the target type to convert to
     * @return the converted object
     */
    protected <C> C convert(Object source, Class<C> targetType) {
        return this.domainTransportConversionService.convert(source, targetType);
    }
    
    /**
     * Convenience method that converts the <code>source</code> Collection to
     * Collection of <code>targetType</code> objects.
     * 
     * @param <C>
     *            the generic target type
     * @param source
     *            the source Collection to convert
     * @param targetType
     *            the target type to convert to
     * @return the converted collection
     */
    @SuppressWarnings("unchecked")
    protected <C> Collection<C> convert(Collection<?> source, Class<C> targetType) {
        return (Collection<C>) this.domainTransportConversionService.convert(source, targetType);
    }


    /**
     * @param domainTransportConversionService
     *            the <code>ConversionService</code> to use for domain and type
     *            conversions
     */
    @Required
    @Resource
    public void setDomainTransportConversionService(ConversionService domainTransportConversionService) {
        this.domainTransportConversionService = domainTransportConversionService;
    }

    /**
     * @param webServiceRootURI
     *            the root URI of the services on the server. The Request-URI
     *            will be appended.
     */
    @Required
    @Resource
    public void setWebServiceRootURI(String webServiceRootURI) {
        this.webServiceRootURI = webServiceRootURI;
    }

    /**
     * @throws Exception
     *             when invalid property found
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        if (!this.webServiceRootURI.endsWith("/")) {
            this.webServiceRootURI = this.webServiceRootURI + "/";
        }
        this.webServiceURI = this.webServiceRootURI + getServiceRequestName();
        logger.info("Configured with web service URI: " + this.webServiceURI);
    }

}
