/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.support.xml;

import java.util.ArrayList;

import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;
import no.g9.support.convert.ConverterRegistry;
import no.g9.support.convert.OSConvertContext;
import no.g9.support.convert.SimpleValueConverter;
import no.g9.support.convert.ValueConverter;
import no.g9.support.filter.AttributeFilter;
import no.g9.support.filter.RoleFilter;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * This class serves as base class for all generated XML converters, and it
 * contains a common way for all XML converters to handle role- and attribute
 * filters, and value converters.
 *
 * <h3>Custom filters</h3>
 * <p>Generated subclasses will call {@link #isRoleIncluded(String, Object)}
 * and {@link #isAttributeIncluded(String, Object)} to check if a role or
 * attribute should be included in the generated XML Document. This behaviour
 * can be used in two ways to add custom filters;
 *
 * <p>The obvious way is to override these methods, and take care of filtering
 * in the subclass. However, there is support in the default implementation for
 * adding filters, and here is a small example that shows how to utilize this.
 *
 * <p>Lets say we wish exclude a role named "Artist", and that we also wish to
 * exclude the "email" atributes of role "Customer" in an object selection named
 * "OrderOs". We then need to add the following code to the constructor of the
 * generated class <code>OrderOsXmlConverter</code>;
 *
 * <pre>
 *     SimpleFilter filter = new SimpleFilter();
 *     filter.addExcludedRoleName("Artist");
 *     filter.addExcludedAttributeName("Customer.email");
 *     addAttributeFilter(filter);
 *     addRoleFilter(filter);
 * </pre>
 *
 * <p>Have a look at {@link AttributeFilter}, {@link RoleFilter} and
 * {@link no.g9.support.filter.SimpleFilter} for a better understanding of
 * how this works.
 *
 * <h3>Custom value converters</h3>
 * <p>The generated subclass will call
 * {@link #convertObjectToString(Object, OSConvertContext)} for all included attributes
 * to convert them to a String. This method can be overridden to add custom
 * value convertion. However, there is support for adding custom value
 * converters for attributes in the default implementation. The following
 * example will show how to take advantage of this.
 *
 * <p>Lets say we wish to convert all attributes with name "specialPrice" from
 * an enumeration (<code>{FullPrice, MediumPrice, LowPrice}</code>) to the
 * discounted price in the OrderOs object selection. We then need to add the
 * following lines to the constructor;
 *
 * <pre>
 *     ConverterRegistry cReg = new ConverterRegistry();
 *     cReg.registerGlobalConverterForAttribute("specialPrice", new SimpleValueConverter {
 *         public String convertToString(Object value, ConvertContext context) {
 *             Record record = (Record) context.getDomainObject();
 *             switch (record.getSpecialPrice().toValue()) {
 *             case EPriceCategory.LowPrice:
 *                 return "" + (record.getPrice() * 0.50);
 *             case EPriceCategory.MediumPrice:
 *                 return "" + (record.getPrice() * 0.75);
 *             default:
 *                 return "" + record.getPrice();
 *             }
 *         }
 *     });
 *     setConverterRegistry(cReg);
 * </pre>
 *
 * <p>The idea behind this approach is to gather all value convertions (in XML
 * context) for an application in as few places as possible. For instance, an
 * application needs only to create one instance of <code>ConverterRegistry
 * </code>. That registry can contain convertion logic for all object selection
 * used in the application.
 *
 * <p>Fields in the object selection of g9 type DATE, TIME or TIMESTAMP are
 * treated differently. The value is given as a pre-formatted ISO-8601 String.
 * The raw value is available through convertContext.getDomainObject().getXXX().
 *
 * <p>Have a look at @{@link ValueConverter} and {@link ConverterRegistry} to
 * get a better understanding of how to add custom value converters to XML
 * conversion.
 *
 */
public abstract class AbstractXmlConverter implements XmlConverter {

    /** List of all attribute filters */
    protected ArrayList<AttributeFilter> attributeFilters;

    /** List of all role filters */
    protected ArrayList<RoleFilter> roleFilters;

    /** Registry of converters */
    protected ConverterRegistry converterRegistry;

    /**
     * Default constructor
     */
    protected AbstractXmlConverter() {
        roleFilters = new ArrayList<RoleFilter>();
        attributeFilters = new ArrayList<AttributeFilter>();
        converterRegistry = new ConverterRegistry(new SimpleValueConverter());
    }

    /*
     * Defined and documented in no.g9.support.xml.XmlConverter
     */
    @Override
    public abstract Document convert(ObjectSelection os, ClientContext ctx);

    /**
     * Adds a filter that will be called for all roles
     * @param filter The {@link RoleFilter} to add
     * @return this
     */
    protected AbstractXmlConverter addRoleFilter(RoleFilter filter) {
        roleFilters.add(filter);
        return this;
    }

    /**
     * Adds a filter that will be called for all attributes
     * @param filter The {@link AttributeFilter} to add
     * @return this
     */
    protected AbstractXmlConverter addAttributeFilter(AttributeFilter filter) {
        attributeFilters.add(filter);
        return this;
    }

    /**
     * Sets the converter registry to use to get ValueConverters for attributes.
     * @param converterRegistry The {@link ConverterRegistry} to use
     */
    protected void setConverterRegistry(ConverterRegistry converterRegistry) {
        this.converterRegistry = converterRegistry;
    }

    /**
     * Returns false if an added filter dictates that the given role of the
     * given domainObject should be excluded.
     * @param roleName Name of the role
     * @param domainObject The domain object
     * @return false if any of the added filters excludes the domain object
     */
    protected boolean isRoleIncluded(String roleName, Object domainObject) {
        for (RoleFilter rf : roleFilters) {
            if (!rf.isRoleIncluded(roleName, domainObject)) return false;
        }
        return true;
    }

    /**
     * Returns false if an added filter dictates that the given attribute in
     * the given domain object should be removed.
     * @param attributeName Attribute name
     * @param domainObject Domain object
     * @return true if the attribute passes all added filters
     */
    protected boolean isAttributeIncluded(String attributeName, Object domainObject) {
        for (AttributeFilter af : attributeFilters) {
            if (!af.isAttributeIncluded(attributeName, domainObject)) return false;
        }
        return true;
    }

    /**
     * Gets a converter from the converter registry, and applies it to the
     * value in the given context, and returns the result.
     * @param value The value Object to convert
     * @param context The {@link OSConvertContext}
     * @return The value objected represented as String
     */
    protected String convertObjectToString(Object value, OSConvertContext context) {
        ValueConverter converter = converterRegistry.getConverter(context);
        return converter.convertToString(value, context);
    }

    /**
     * Helper function that creates a document node from an atribute
     * @param attrValue Attribute value
     * @param doc The DOcument the attribute is to be included in
     * @param oSConvertContext Convert context
     * @return A Node representing the attribute
     */
    protected Node makeAttributeNode(Object attrValue, Document doc, OSConvertContext oSConvertContext) {
        if (!isAttributeIncluded(oSConvertContext.getRoleName() + "." + oSConvertContext.getAttributeName(), oSConvertContext.getDomainObject())) return null;
        String value = convertObjectToString(attrValue, oSConvertContext);
        if (value == null) return null;
        Node n  = doc.createElement(oSConvertContext.getAttributeName());
        n.appendChild(doc.createTextNode(value));
        return n;
    }

}
