/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.faces.component.UIComponent;
import javax.faces.event.ActionEvent;

import no.esito.jvine.communication.SystemMessageUtils;
import no.esito.jvine.controller.DialogInstanceKey;
import no.esito.jvine.view.ViewModelImpl;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogObjectType;
import no.g9.client.core.view.BooleanProperty;
import no.g9.client.core.view.EffectProperty;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.faces.FacesComparatorMap.ComparatorPropertyType;
import no.g9.exception.G9BaseException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.os.RoleConstant;

import org.icefaces.ace.component.submenu.Submenu;
import org.icefaces.ace.model.MenuModel;
import org.icefaces.ace.model.table.RowStateMap;
import org.icefaces.ace.model.tree.NodeStateMap;
import org.springframework.util.StringUtils;

/**
 * Common superclass for all JSF backing beans for the dialogs in the
 * application.
 *
 * Note: this class and its generated subclasses is intended as a JSF interface
 * to the view, and should normally not be used by the application programmer
 * (unless the properties accessed from the JSF markup needs customization).
 * Care should be taken if the public or protected methods are overridden in the
 * generated &lt;dialog&gt;Bean subclass.
 */
@SuppressWarnings({"rawtypes", "serial"})
public abstract class FacesDialogBean implements Serializable {

	private DialogInstance instance;

	/** Set from the constructor. The <code>instance</code> is set during initialization */
	private DialogConstant tmpDialog;

	/** A copy of the same from the dialog view, set during initialization */
	private String messageBundleName;

    private FacesApplicationBean applicationBean;

	private FacesPropertyMap<BooleanProperty> propertyEditable;

	private FacesPropertyMap<BooleanProperty> propertyEnabled;

	private FacesPropertyMap<BooleanProperty> propertyMandatory;

	private FacesPropertyMap<BooleanProperty> propertyShown;

	private FacesPropertyMap<BooleanProperty> propertyExpanded;

	private FacesPropertyMap<EffectProperty> propertyEffect;

	private FacesTitleMap propertyTitle;

	private FacesComparatorMap propertySortPriority;

	private FacesComparatorMap propertySortAscending;

	/**
	 * Create a new backing bean for the given dialog.
	 *
	 * @param dialog
	 *            the dialog for this bean.
	 */
	public FacesDialogBean(DialogConstant dialog) {
		this.tmpDialog = dialog;
	}


	/**
	 * @return the Map for the EDITABLE property
	 */
	public Map<?, ?> getEditable() {
		return propertyEditable;
	}

	/**
	 * @return the Map for the ENABLED property
	 */
	public Map<?, ?> getEnabled() {
		return propertyEnabled;
	}

	/**
	 * @return the Map for the MANDATORY property
	 */
	public Map<?, ?> getMandatory() {
		return propertyMandatory;
	}

	/**
	 * @return the Map for the SHOWN property
	 */
	public Map<?, ?> getShown() {
		return propertyShown;
	}

	/**
	 * @return the Map for the EXPANDED property
	 */
	public Map<?, ?> getExpanded() {
		return propertyExpanded;
	}

	/**
	 * @return the Map for the EFFECT property
	 */
	public Map<?, ?> getEffect() {
		return propertyEffect;
	}

    /**
     * @return the Map for the title of a field
     */
    public Map<?, ?> getTitle() {
        return propertyTitle;
    }

    /**
     * @return the Map for sort priority
     */
    public Map<?, ?> getSortPriority() {
        return propertySortPriority;
    }
    
    /**
     * @return the Map for sort ascending
     */
    public Map<?, ?> getSortAscending() {
        return propertySortAscending;
    }
    
    /**
     * Get the row state map for the given role.
     *
     * @param role the role to get the row state map for
     * @return the rowStateMap
     */
    protected RowStateMap getRowStateMap(RoleConstant role) {
    	return getDialogView().getRowStateMap(role);
    }

    /**
     * Update the row states for the given role.
     *
     * @param role the owning role of the table
     */
    protected void updateRowState(RoleConstant role) {
    	getDialogView().updateRowState(role);
    }

    /**
     * Get the node state map for the given dialog constant.
     *
     * @param diaConst the dialog constant to get the node state map for
     * @return the nodeStateMap
     */
    protected NodeStateMap getNodeStateMap(DialogObjectConstant diaConst) {
    	return getDialogView().getNodeStateMap(diaConst);
    }

    /**
     * Set the node state map for the given dialog constant.
     *
     * @param diaConst  the dialog constant to set the node state map for
     * @param nodeStateMap the nodeStateMap to set
     */
    protected void setNodeStateMap(DialogObjectConstant diaConst, NodeStateMap nodeStateMap) {
        getDialogView().setNodeStateMap(diaConst, nodeStateMap);
    }

    /**
	 * Initialize the property maps for this dialog bean. The property maps are
	 * accessed from JSF markup, and routes the access to the property manager
	 * for the dialog view.
	 *
	 * @param enumClass
	 *            - the class for the const enum for this dialog.
	 */
	@SuppressWarnings("unchecked")
	protected void initPropertyMaps(Class enumClass) {
		propertyEditable = new FacesPropertyMap(enumClass, this, BooleanProperty.EDITABLE);
		propertyEnabled = new FacesPropertyMap(enumClass, this, BooleanProperty.ENABLED);
		propertyMandatory = new FacesPropertyMap(enumClass, this, BooleanProperty.MANDATORY);
		propertyShown = new FacesPropertyMap(enumClass, this, BooleanProperty.SHOWN);
		propertyExpanded = new FacesPropertyMap(enumClass, this, BooleanProperty.EXPANDED);
		propertyEffect = new FacesPropertyMap(enumClass, this, EffectProperty.STD_EFFECT);
		propertyTitle = new FacesTitleMap(enumClass, getViewModel().getDialogController().getDialogView());
		propertySortPriority = new FacesComparatorMap(enumClass, this, ComparatorPropertyType.PRIORITY);
		propertySortAscending = new FacesComparatorMap(enumClass, this, ComparatorPropertyType.ASCENDING);
	}

    /**
     * Get the current Notebook page from the view.
     *
     * @param notebook the Notebook in question.
     * @return the current page for the Notebook.
     */
    protected DialogObjectConstant getCurrentNotebookPage(DialogObjectConstant notebook) {
        return getDialogView().getCurrentNotebookPage(notebook);
    }

    /**
     * Set the current Notebook page in the view.
     *
     * @param notebook the Notebook.
     * @param page the new current page for the Notebook.
     */
    protected void setCurrentNotebookPage(DialogObjectConstant notebook, DialogObjectConstant page) {
        getDialogView().setCurrentNotebookPage(notebook, page);
    }

    /**
     * Get the current Notebook index from the view.
     *
     * @param notebook the Notebook in question.
     * @return the current index for the Notebook.
     */
    protected int getCurrentNotebookIndex(DialogObjectConstant notebook) {
        return getDialogView().getCurrentNotebookIndex(notebook);
    }

    /**
     * Set the current Notebook index in the view.
     *
     * @param notebook the Notebook.
     * @param index the new current page index for the Notebook.
     */
    protected void setCurrentNotebookIndex(DialogObjectConstant notebook, int index) {
        getDialogView().setCurrentNotebookIndex(notebook, index);
    }

    /**
     * Get the old Notebook index from the view.
     *
     * @param notebook the Notebook in question.
     * @return the old index for the Notebook.
     */
    protected int getOldNotebookIndex(DialogObjectConstant notebook) {
        return getDialogView().getOldNotebookIndex(notebook);
    }

    /**
     * Set the old Notebook index in the view.
     *
     * @param notebook the Notebook.
     * @param index the new old page index for the Notebook.
     */
    protected void setOldNotebookIndex(DialogObjectConstant notebook, int index) {
        getDialogView().setOldNotebookIndex(notebook, index);
    }

	/**
	 * Delegate the list selection handling to the view.
	 *
	 * @param listRole
	 *            - the object selection role for the list
	 * @param selected
	 *            - true if the row is to be selected
	 * @param rowNo
	 *            - the list row in question
	 * @param emptyRow
	 *            - an empty list row containing a collection of all the fields
	 *            in a list row
	 * @return true if the selection in the list was changed, false if the
	 *         operation was canceled by the user
	 */
	protected boolean handleListSelection(RoleConstant listRole,
			boolean selected, int rowNo, ListRow emptyRow) {
		return getDialogView().handleListSelection(listRole, selected, rowNo,
				emptyRow);
	}
	
	/**
	 * Sets the selected row num in the dialog view, so that
	 * it may be fetched during action Select later.
	 * 
	 * @param rowNum row number
	 */
	protected void setListSelectionRowNum(int rowNum) {
	    getDialogView().setListSelectionRowNum(rowNum);
	}

	/**
	 * Delegate the show operation to the view.
	 *
	 * @param target
	 *            the dialog object to show.
	 */
	protected void show(DialogObjectConstant target) {
		getDialogView().show(target);
	}

	/**
	 * Delegate the hide operation to the view.
	 *
	 * @param target
	 *            the dialog object to hide.
	 */
	protected void hide(DialogObjectConstant target) {
		getDialogView().hide(target);
	}

	/**
	 * Delegate the event dispatch to the view.
	 *
	 * @param trigger
	 *            the cause of the event.
	 * @param event
	 *            the type of event.
	 * @param method
	 *            the event method name.
	 */
	protected void dispatchEvent(DialogObjectConstant trigger, Object event,
			String method) {
	    clearOuputMessageQueue();
		dispatchEvent(null, trigger, event, method);
		sendOutputMessageQueue();
	}

    /**
     * Dispatch the JSON messages.
     * Messages where the application menu is the port are dispatched directly, with the
     * message payload as the event method.
     * Other messages are forwarded to the application controller, and will normally contain
     * G9 actions in the payload.
     *
     * @param trigger the cause of the event
     * @param systemMessagesInJson JSON coded array of messages
     */
    protected void dispatchSystemMessages(DialogObjectConstant trigger, String systemMessagesInJson) {
        if (getDialogView().canDispatchEvents()) {
            List<SystemMessage> messages = SystemMessageUtils.fromJSon(systemMessagesInJson);
            for (SystemMessage systemMessage : messages) {
                if (SystemMessage.MENU_PORT.equals(systemMessage.port)) {
                    dispatchEvent(trigger, systemMessage, systemMessage.payload);
                }
                else {
                    clearOuputMessageQueue();
                    getApplicationBean().getApplicationView().getApplicationController().forward(systemMessage);
                    sendOutputMessageQueue();
                }
            }
        }
        else {
            getApplicationBean().getApplicationView().sendElevateMessage();
            sendOutputMessageQueue();
        }
    }

    private void clearOuputMessageQueue() {
        getApplicationBean().getApplicationView().getOutputMessages().clear();
    }

    private void sendOutputMessageQueue() {
        for (SystemMessage outputMessage : getApplicationBean().getApplicationView().getOutputMessages()) {
            getApplicationBean().getApplicationView().forward(outputMessage);
        }
        clearOuputMessageQueue();
    }

    /**
	 * Delegate the event dispatch to the view, with listRow.
	 *
	 * @param listRow
	 *            the ListRow in which the event occurred.
	 * @param trigger
	 *            the cause of the event.
	 * @param event
	 *            the type of event.
	 * @param method
	 *            the event method name.
	 */
	protected void dispatchEvent(ListRow listRow, DialogObjectConstant trigger,
			Object event, String method) {
		getDialogView().dispatchEvent(listRow, trigger, event, method);
	}

	/**
	 * Get the JSF facelet dynamic include path for the toolbar of the current
	 * dialog.
	 *
	 * @return the dynamic include path for the current dialog toolbar.
	 */
	public String getDialogToolbarIncludePath() {
		return getApplicationBean().getDialogToolbarIncludePath();
	}

	/**
	 * @return the application menu model (for the menu bar), as created by the
	 *         application view.
	 */
	public MenuModel getApplicationMenuModel() {
		return getApplicationBean().getApplicationView().getApplicationMenuModel();
	}

	/**
	 * @return the application menu as a list of menus, as created by the
	 *         application view.
	 */
	public List<Submenu> getApplicationMenu() {
	    return getApplicationBean().getApplicationView().getApplicationMenu();
	}

    /**
     * Get the message text for the given key. The default is to look for the
     * message in the generated messages.properties file for a dialog.
     *
     * This method is used from the generated xhtml markup, to allow
     * for providing other message sources than the default resource bundle.
     *
     * @param key the message key for the message
     * @return the message text for the given key
     */
    public String msg(String key) {
    	return getApplicationBean().getFacesMessage(getMessageBundleName(), key);
    }

	/**
	 * Delegate method for the view model.
	 *
	 * @param field
	 *            the field with the input value.
	 * @param value
	 *            the field value to copy.
	 */
	protected void copyToEquivalentFields(DialogObjectConstant field,
			Object value) {
		getViewModel().copyToEquivalentFields(field, value);
	}

	/**
	 * Get the field (view model value) for the given field.
	 *
	 * @param field
	 *            the field holds the value for this field.
	 * @return the field.
	 */
	protected Object getField(DialogObjectConstant field) {
		return getViewModel().getField(field);
	}

	/**
	 * Set the field (view model value) for the given field. If the new value
	 * differs from the old value, the field is marked as changed.
	 *
	 * @param field
	 *            the field holds the value for this field.
	 * @param value
	 *            the new field value.
	 */
	protected void setField(DialogObjectConstant field, Object value) {
		getViewModel().setField(field, value);
	}

	/**
	 * Delegate method for the view model.
	 *
	 * @param field The (generated) enum denoting the role attribute
	 * @return the specified field value (possibly converted from view to model value)
	 */
	protected Object getFieldValue(DialogObjectConstant field) {
		return getViewModel().getFieldValue(field);
	}

	/**
	 * Delegate method for the view model.
	 *
	 * @param field The (generated) enum denoting the role attribute
	 * @param fieldValue The field value to display (possibly converted from model to view value)
	 */
	protected void setFieldValue(DialogObjectConstant field, Object fieldValue) {
		getViewModel().setFieldValue(field, fieldValue);
	}

	/**
	 * Delegate method for the view model.
	 *
	 * @param role
	 *            the role which owns the field collection.
	 * @return the field collection.
	 */
	protected Collection<DialogObjectConstant> getRoleFields(RoleConstant role) {
		return getViewModel().getRoleFields(role);
	}

	/**
	 * Get the JSF managed bean which references the application view.
	 *
	 * @return the JSF application bean.
	 */
	public FacesApplicationBean getApplicationBean() {
		return applicationBean;
	}

	/**
	 * Set the JSF backing bean for the application. This done by JSF when the
	 * dialog is opened for the first time, injected as the applicationBean
	 * property in the config file. The initialization of the bean in the
	 * initBean method is essential to get a properly working dialog.
	 *
	 * @param applicationBean
	 *            the new JSF application bean.
	 */
	public void setApplicationBean(FacesApplicationBean applicationBean) {
		this.applicationBean = applicationBean;
		if (instance != null) {
		    initInstance();
		}
	}

    /**
     * Gets the 1-based dialog instance number.
     *
     * @return the dialog instance number
     */
    public int getDialogInstanceNumber() {
        return instance.getDialogInstanceNumber();
    }

    /**
     * Sets the 1-based dialog instance number.
     *
     * @param dialogInstanceNumber the new dialog instance number.
     *
     * @see DialogConstant#getMaximumNumberOfInstances()
     */
    public void setDialogInstanceNumber(int dialogInstanceNumber) {
        this.instance = new DialogInstanceKey(tmpDialog, dialogInstanceNumber);
        if (applicationBean != null) {
            initInstance();
        }
    }

    /**
     * Set this dialog instance as the active instance for this dialog model.
     */
    public void setAsActiveInstance() {
        FacesDialogView view = getDialogView();
        view.getApplicationView().getApplicationController().setActiveDialogInstance(view.getDialogInstance());
    }

    private void initInstance() {
        if (instance == null) {
            throw new IllegalStateException("Trying to initialize an unknown instance");
        }
        if (applicationBean == null) {
            throw new IllegalStateException("Trying to initialize an instance without an application");
        }
        FacesApplicationView appView = applicationBean.getApplicationView();
        FacesDialogView dialogView = appView.createDialogView(instance);
        setMessageBundleName(dialogView.getMessageBundleName());
        initBean();
        dialogView.setOpen(true);
    }

    /**
	 * Initialization of the bean. This method is implemented in the generated
	 * subclasses.
	 */
	protected abstract void initBean();

	/**
     * @param <T> The dialog view type
	 * @return the view for this dialog.
	 */
	@SuppressWarnings("unchecked")
    protected <T extends FacesDialogView> T getDialogView() {
		FacesApplicationView appView = getApplicationBean().getApplicationView();
		return (T) appView.getDialogView(instance);
	}

	/**
	 * @return true if this dialog has an open view
	 */
	protected boolean hasOpenDialogView() {
	    FacesApplicationView appView = getApplicationBean().getApplicationView();
	    return appView.hasOpenDialogView(instance);
	}

	/**
	 * @return the view model for this dialog.
	 */
	protected ViewModelImpl getViewModel() {
		return (ViewModelImpl) getDialogView().getViewModel();
	}

    /**
     * Validate the new value for the given field. If the new value does not
     * pass the validation, an error message is shown to the user.
     *
     * @param field the field to validate.
     * @param newValue the new field value (which is to be validated).
     * @param oldValue the old field value.
     *
     * @return true if the validation passed, false otherwise.
     */
    protected boolean validateField(DialogObjectConstant field, Object newValue, Object oldValue) {
        return validateField(null, field, newValue, oldValue);
    }

    /**
     * Validate the new value for the given field. If the new value does not
     * pass the validation, an error message is shown to the user.
     *
     * @param listRow the ListRow in which the event occurred.
     * @param field the field to validate.
     * @param newValue the new field value (which is to be validated).
     * @param oldValue the old field value.
     *
     * @return true if the validation passed, false otherwise.
     */
    protected boolean validateField(ListRow listRow,
            DialogObjectConstant field, Object newValue, Object oldValue) {
        return getDialogView().validateField(listRow, field, newValue);
    }

    /**
     * Get the title for this dialog.
     *
     * @return the title
     */
    public String getDialogTitle() {
        return getApplicationBean().getDialogTitle(instance);
    }

    /**
     * Get the ListRow for the given context value. The context value is taken
     * from the JSF display event, and may be of type ListRow or String.
     *
     * @param listRole the role for the list which contains the row.
     * @param contextValue the context value from the JSF event.
     * @return the found ListRow, or null if not found.
     */
    protected ListRow getListRow(RoleConstant listRole, Object contextValue) {
        if (contextValue instanceof ListRow) {
            return (ListRow) contextValue;
        }
        if (contextValue instanceof String) {
            String rowId = (String) contextValue;
            List<ListRow> list = getViewModel().getDisplayList(listRole);
            ListRow listRow = null;
            for (ListRow row : list) {
                if (rowId.equals(row.toString())) {
                    listRow = row;
                    break;
                }
            }
            return listRow;
        }
        return null;
    }

    /**
     * @return the messageBundleName
     */
    public String getMessageBundleName() {
        return messageBundleName;
    }

    /**
     * @param messageBundleName the messageBundleName to set
     */
    private void setMessageBundleName(String messageBundleName) {
        this.messageBundleName = messageBundleName;
    }

    /**
     * Execute the action listener method for the current default button.
     * The default button must be a button in the dialog model for this dialog,
     * and it must have a Clicked event.
     *
     * @param event the JSF event
     */
    public void defaultButtonActionListener(ActionEvent event) {
        DialogObjectConstant defaultButton = getDialogView().getDefaultButton();
        if (defaultButton != null && defaultButton.getType() == DialogObjectType.Button) {
            String methodName = StringUtils.uncapitalize(defaultButton.getG9Name()) + "_Clicked";
            try {
                Method method = this.getClass().getMethod(methodName, ActionEvent.class);
                method.invoke(this, event);
            } catch (Exception e) {
                String msg = "Failed to execute action listener method " + methodName;
                Message message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_CAUGHT, msg);
                message.setException(e);
                throw new G9BaseException(message);
            }
        }
    }

    /**
     * Delegate to the view.
     * 
     * @param target the dialog object to look up
     * @return the faces ID for target
     */
    protected String getFacesId(DialogObjectConstant target) {
    	return getDialogView().getFacesId(target);
    }

    /**
     * Delegate to the view.
     *
     * @param comp the parent component, or null
     * @param key the JSF component id
     * @return the JSF component with the given key
     */
    protected UIComponent getFacesComponent(UIComponent comp, String key) {
    	return getDialogView().getFacesComponent(comp, key);
    }

}
