/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.text.ParseException;

import javax.faces.component.PartialStateHolder;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;

import no.esito.log.Logger;
import no.esito.util.NumberFormat;
import no.g9.support.FormatHelper;
import no.g9.support.G9Consts;
import no.g9.support.Numeric;

/**
 * JSF converter class for number types in G9.
 */
public class NumberConverter implements Converter, G9Converter, PartialStateHolder {

    private static final Logger log = Logger.getLogger(NumberConverter.class);

    private Integer inputLength;
    private Integer storedLength;
    private Integer scale;
    private Boolean blankWhenZero;
    private NumberFormat numberFormat = null;
    private ConverterHelper converterHelper;
    private Boolean isG9Numeric = false;

    /**
     * Create a new number converter.
     */
    public NumberConverter() {
        converterHelper = new ConverterHelper();
        if (log.isTraceEnabled()) {
            log.trace("Created instance: " + this);
        }
    }

    @Override
    public Object getAsObject(FacesContext context, UIComponent component, String value) {

        Object retVal = null;

        if (context == null || component == null) {
            throw new NullPointerException();
        }

        if (log.isTraceEnabled()) {
            log.trace("getAsObject: " + this + ", component: " + component.getId() + ", value: " + value);
        }

        // Use the old value if it was shown as overflow (####) in the view
        if (isOverflow(value)) {
            retVal = converterHelper.getComponentValue(component);
            if (log.isTraceEnabled()) {
                log.trace("getAsString: " + this + ", component: " + component.getId() + ", restore overflow value: " + retVal);
            }
            return retVal;
        }

        if (value != null) {
            if (value.trim().length() < 1) {
                return null;
            }
            try {
                retVal = getNumberFormat().parse(value, false);
                if (retVal instanceof Numeric && !isG9Numeric) {
                	retVal = ((Numeric)retVal).getValue();
                }
            } catch (ParseException e) {
                throw new ConverterException(converterHelper.getConversionErrorMessage());
            }
        }
        return retVal;
    }

    @Override
    public String getAsString(FacesContext context, UIComponent component, Object value) {

        if (context == null || component == null) {
            throw new NullPointerException();
        }

        if (log.isTraceEnabled()) {
            log.trace("getAsString: " + this + ", component: " + component.getId() + ", value: " + value);
        }

        if (value == null) {
            return "";
        }
        return getOutputText(value);
    }

    private String getOutputText(Object value) {
        String strValue = null;
        if (value != null) {
            if (isBlankWhenZero() && NumberFormat.isZero(value)) {
                strValue = "";
            }
            else {
                strValue = getNumberFormat().format(FormatHelper.getNumericFormat(getDisplayRule()), value, false);
            }
        }
        return strValue;
    }


    private int getDataTypeAsInt() {
        return G9Consts.Converter.fromName(getDataType());
    }

    private NumberFormat getNumberFormat() {
        if (numberFormat == null) {
            numberFormat = new NumberFormat(getDataTypeAsInt(), getDisplayRule());
        }
        return numberFormat;
    }

    private boolean isOverflow(String value) {
        if (value != null && value.matches("#*")) {
            return true;
        }
        return false;
    }

    @Override
    public String getDataType() {
        return converterHelper.getDataType();
    }

    /**
     * Set the data type.
     * If the number formatter is already created, we need to create a new one
     * with the current data type.
     *
     * @param dataType the name of the data type.
     */
    @Override
    public void setDataType(String dataType) {
        if (log.isTraceEnabled()) {
            log.trace("setDataType: " + this + ", dataType: " + dataType);
        }

        clearInitialState();

        converterHelper.setDataType(dataType);
        if (numberFormat != null) {
            numberFormat = new NumberFormat(getDataTypeAsInt(), getDisplayRule());
        }
    }

    /**
     * @return the displayRule.
     */
    @Override
    public String getDisplayRule() {
        return FormatHelper.getDisplayrule(getDataTypeAsInt(), converterHelper.getDisplayRule(),
                getInputLength(), getStoredLength(), getScale());
    }

    /**
     * Set the display rule.
     * If the number formatter is already created, we need to create a new one
     * with the current display rule.
     *
     * @param displayRule the display rule to set.
     */
    @Override
    public void setDisplayRule(String displayRule) {
        clearInitialState();

        converterHelper.setDisplayRule(displayRule);
        if (numberFormat != null) {
            numberFormat = new NumberFormat(getDataTypeAsInt(), getDisplayRule());
        }
    }

    /**
     * @return the inputLength.
     */
    public int getInputLength() {
        return inputLength;
    }

    /**
     * @param inputLength the inputLength to set.
     */
    public void setInputLength(int inputLength) {
        clearInitialState();

        this.inputLength = inputLength;
    }

    /**
     * @return the storedLength.
     */
    public int getStoredLength() {
        return storedLength;
    }

    /**
     * @param storedLength the storedLength to set.
     */
    public void setStoredLength(int storedLength) {
        clearInitialState();

        this.storedLength = storedLength;
    }

    /**
     * @return the scale.
     */
    public int getScale() {
        return scale;
    }

    /**
     * @param scale the scale to set.
     */
    public void setScale(int scale) {
        clearInitialState();

        this.scale = scale;
    }

    /**
     * @return the blankWhenZero.
     */
    public boolean isBlankWhenZero() {
        return blankWhenZero;
    }

    /**
     * @param blankWhenZero the blankWhenZero to set.
     */
    public void setBlankWhenZero(boolean blankWhenZero) {
        clearInitialState();

        this.blankWhenZero = blankWhenZero;
    }

	public Boolean getIsG9Numeric() {
		return isG9Numeric;
	}

	public void setIsG9Numeric(Boolean isG9Numeric) {
		this.isG9Numeric = isG9Numeric;
	}

    @Override
    public String getTitle() {
        return converterHelper.getTitle();
    }

    @Override
    public void setTitle(String title) {
        clearInitialState();

        converterHelper.setTitle(title);
    }

    @Override
    public Object saveState(FacesContext context) {

        if (context == null) {
            throw new NullPointerException();
        }
        if (!initialStateMarked()) {
            Object values[] = new Object[7];

            values[0] = inputLength;
            values[1] = storedLength;
            values[2] = scale;
            values[3] = blankWhenZero;
            values[4] = numberFormat;
            values[5] = converterHelper;
            values[5] = isG9Numeric;
            return (values);
        }
        return null;
    }

    @Override
    public void restoreState(FacesContext context, Object state) {

        if (context == null) {
            throw new NullPointerException();
        }
        if (state != null) {
            Object values[] = (Object[]) state;
            inputLength = (Integer) values[0];
            storedLength = (Integer) values[1];
            scale = (Integer) values[2];
            blankWhenZero = (Boolean) values[3];
            numberFormat = (NumberFormat) values[4];
            converterHelper = (ConverterHelper) values[5];
            isG9Numeric = (Boolean) values[6];
        }
    }

    private boolean transientFlag = false;

    @Override
    public boolean isTransient() {
        return (transientFlag);
    }

    @Override
    public void setTransient(boolean transientFlag) {
        this.transientFlag = transientFlag;
    }

    private boolean initialState;

    @Override
    public void markInitialState() {
        initialState = true;
    }

    @Override
    public boolean initialStateMarked() {
        return initialState;
    }

    @Override
    public void clearInitialState() {
        initialState = false;
    }

}
