/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces.tree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import no.esito.jvine.view.ViewModelImpl;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.BooleanProperty;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.ViewModel;
import no.g9.client.core.view.table.RowFilter;
import no.g9.client.core.view.tree.TreeModel;
import no.g9.client.core.view.tree.TreeNode;
import no.g9.client.core.view.tree.TreeNodeComparator;

import org.icefaces.ace.model.tree.NodeStateMap;

/**
 * The default tree model implementation. Contains the complete collection of tree nodes. 
 *
 * @param <T> the generated FacesTreeNode class
 * @param <L> the generated ListRow class
 */
public class FacesDefaultTreeModel <T extends FacesTreeNode, L extends ListRow> implements TreeModel<T, L>, Observer {
	
    /**
     * The state map contains the state of a particular node object,
     * as well as reverse look-ups to get node objects with a particular state.
     */
	private final NodeStateMap nodeStateMap;
	
	/** The list of filtered root nodes */
	protected final FacesNodeList<T> treeView = new FacesNodeList<>();
	
	/** The list of all root nodes */
	protected final FacesNodeList<T> treeData = new FacesNodeList<>();
	
	/** The list of row filters */
	private final List<RowFilter<?, ListRow>> rowFilters = new ArrayList<RowFilter<?, ListRow>>();

	protected final ViewModelImpl viewModel;
	
	/**
	 * Create a new eagerly fetched tree model. 
	 * 
	 * @param viewModel the dialog's viewmodel
	 * @param nodeStateMap the state map for the nodes
	 */
	public FacesDefaultTreeModel(ViewModel viewModel, NodeStateMap nodeStateMap) {
		this.viewModel = (ViewModelImpl) viewModel;
		this.nodeStateMap = nodeStateMap;
		this.treeData.addObserver(this);
	}
	

	@Override
	public void update(Observable o, Object arg) {
		filterTree(treeData, treeView);
		sortTree(treeView);
	}
	
    @Override
	public FacesNodeList<T> getTreeView() {
		return treeView;
	}

	
	@Override
	public FacesNodeList<T> getTreeData() {
		return treeData;
	}

	@Override
	public Object getDataModel() {
		return treeView;
	}
	
	  ////////////////////////////////
	 /// Filters ////////////////////
	////////////////////////////////
	
    @Override
    public void addRowFilter(RowFilter<?, ListRow> rowFilter) {
        rowFilters.add(rowFilter);
        filterTree(treeData, treeView);
    }

    @Override
    public void addRowFilters(Collection<RowFilter<?, ListRow>> rowFilter) {
        rowFilters.addAll(rowFilters);
        filterTree(treeData, treeView);
    }

    @Override
    public void removeRowFilter(RowFilter<?, ListRow> rowFilter) {
        rowFilters.remove(rowFilter);
        filterTree(treeData, treeView);
    }

    @Override
    public void removeRowFilters(Collection<RowFilter<?, ListRow>> rowFilter) {
        rowFilters.removeAll(rowFilter);
        filterTree(treeData, treeView);
    }
    
	@SuppressWarnings({ "unchecked"})
	@Override
	public void filterTree(List<T> srcList, List<T> dstList) {
		dstList.clear();

		T currNode = null;
		T currViewNode = null;
		T currViewRootNode = null;
		
		for(T rootNode : srcList) {

			currNode = rootNode;
			currViewRootNode = currViewNode = (T) rootNode.clone();
			
			while(currNode != null) {
				
				if(passFilters(currNode) == true) {
					
					if(currNode == rootNode) {
						dstList.add(currViewRootNode);
					} else {
						currViewNode.add((T) currNode.clone());
					}

					T nxtNode = (T) currNode.getNextNode(); // next node, pre-order traversal.

					if(nxtNode == null) {
						currNode = null;
						break;
					}

					if(currNode.isNodeDescendant(nxtNode)) { // update view node to point to 'next' node
						if(currNode != rootNode) {
							currViewNode = (T) currViewNode.getLastChild();
						}
					} else { // need to backtrack to common ancestor
						T ancestorNode = (T) currNode.getSharedAncestor(nxtNode);
						int backtrack = currNode.getLevel() - ancestorNode.getLevel();
						while(backtrack-- > 1) {
							currViewNode = (T) currViewNode.getParent();
						}
					}
					
				} else {
					while(currNode.isNodeDescendant(currNode.getNextNode()) ) { // "skip" subtree located at currNode
						currNode = (T) currNode.getNextNode();
					}
				} 
				currNode = (T) currNode.getNextNode();
			}

		}
	}

	private boolean passFilters(T node) {
		ListRow row = node.getListRow();
        for (RowFilter<?, ListRow> filter : rowFilters) {
            if (!filter.passFilter(row)) {
                return false;
            }
        }
        return true;
	}

	  ////////////////////////////////
	 /// Cell values ////////////////
	////////////////////////////////
	
	@Override
    public Object getValueAt(T node, DialogObjectConstant columnIdentifier) {
    	return node.getListRow().getValue(columnIdentifier);
    }

	@Override
    public void setValueAt(T node, DialogObjectConstant columnIdentifier, Object value) {
    	node.getListRow().setValue(columnIdentifier, value);
    }

	  ////////////////////////////////
	 /// Node counts ////////////////
	////////////////////////////////
	
	@Override
	public int getTreeDataNodeCount() {
		return nodeCount(treeData);
	}

	@Override
	public int getTreeViewNodeCount() {
		return nodeCount(treeView);
	}

	@SuppressWarnings("unchecked")
	private int nodeCount(List<T> rootNodeList) {
		int retval = 0;
		for(T node : rootNodeList) {
			Enumeration<T> e = node.breadthFirstEnumeration();
			while(e.hasMoreElements()) {
				e.nextElement();
				retval++;
			}
		}
		return retval;
	}
	
	@Override
	public void clear() {
		treeData.clear();
		nodeStateMap.clear();
	}

	@Override
	public boolean isEmpty() {
		return treeData.isEmpty();
	}

	@Override
	public void setRootNodes(List<T> rootNodeList){
		treeData.addAll(rootNodeList);
	}
	
	  ////////////////////////////////
	 /// Selected & Expanded ////////
	////////////////////////////////

    protected SelectionModel selectionModel = SelectionModel.DEFAULT;

	@Override
	public void setSelectionModel(SelectionModel selectionModel) {
		this.selectionModel = selectionModel;
		if(selectionModel == SelectionModel.NO_SELECT) {
			unselectAll();
		}
	}
    
    @Override
    public SelectionModel getSelectionModel() {
        return selectionModel;
    }
	
	@Override
	public void setSelected(T node, boolean selected) {

		switch (selectionModel) {
	        case SINGLE_SELECT:
	            unselectAll();
	            //$FALL-THROUGH$
	        case MULTI_SELECT:
	            nodeStateMap.get(node).setSelected(selected);
	            node.getListRow().setSelected(selected);
	            break;
	        case NO_SELECT:
	            if (selected) {
	                throw new IllegalStateException("Current selection model prohibits selection of a node");
	            }
	            nodeStateMap.get(node).setSelected(selected);
	            node.getListRow().setSelected(selected);
	            break;
	        default:
	            break;
        }
	}
	
    @SuppressWarnings("unchecked")
	@Override
    public void setSelected(boolean selected) {
    	
        if (selected && (selectionModel == SelectionModel.SINGLE_SELECT || selectionModel == SelectionModel.NO_SELECT)) {
            throw new IllegalStateException("Current selection model prohibits selection of a node");
        }

        nodeStateMap.setAllSelected(selected);
        
        for(T rootNode : treeData) {
        	Enumeration<T> e = rootNode.breadthFirstEnumeration();
        	while(e.hasMoreElements()) {
        		T n = e.nextElement();
        		n.getListRow().setSelected(selected);
        	}
        }
    }

    @SuppressWarnings("unchecked")
	private void unselectAll() {
        for(T rootNode : treeData) {
        	Enumeration<T> e = rootNode.breadthFirstEnumeration();
        	while(e.hasMoreElements()) {
        		T n = e.nextElement();
        		n.getListRow().setSelected(Boolean.FALSE);
        	}
        }
        
        nodeStateMap.setAllSelected(false);
    }
	
	@Override
    public boolean isSelected(T node) {
        return node.getListRow().isRowSelected();
    }

    @SuppressWarnings("unchecked")
	@Override
    public List<T> getSelected() {
        return nodeStateMap.getSelected();
    }
    
    @Override
	public void setExpanded(T node, boolean expanded) {
		nodeStateMap.get(node).setExpanded(expanded);
	}

    @Override
	public void setExpanded(boolean expanded) {
		nodeStateMap.setAllExpanded(expanded);
	}

	@Override
	public boolean isExpanded(T node) {
		return nodeStateMap.get(node).isExpanded();
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<T> getExpanded() {
		return nodeStateMap.getExpanded();
	}
    
	  ////////////////////////////////
	 /// Enabled & Shown ////////////
	////////////////////////////////
	
    @Override
    public void setEnabled(boolean enabled) {
        setProperty(BooleanProperty.ENABLED, Boolean.valueOf(enabled));
    }

    @Override
    public void setNodeEnabled(T node, boolean enabled) {
        setNodeProperty(node, BooleanProperty.ENABLED, enabled);
    }

    @Override
    public void setColumnEnabled(DialogObjectConstant columnIdentifier, boolean enabled) {
        setColumnProperty(columnIdentifier, BooleanProperty.ENABLED, enabled);
    }

    @Override
    public void setCellEnabled(T node, DialogObjectConstant columnIdentifier, boolean enabled) {
        this.setCellProperty(node, columnIdentifier, BooleanProperty.ENABLED, enabled);
    }

    @Override
    public boolean isCellEnabled(T node, DialogObjectConstant columnIdentifier) {
        return getCellProperty(node, columnIdentifier, BooleanProperty.ENABLED);
    }    
    
    @Override
    public void setShown(boolean shown) {
        setProperty(BooleanProperty.SHOWN, Boolean.valueOf(shown));
    }

    @Override
    public void setNodeShown(T node, boolean shown) {
        setNodeProperty(node, BooleanProperty.SHOWN, shown);
    }

    @Override
    public void setColumnShown(DialogObjectConstant columnIdentifier, boolean shown) {
        setColumnProperty(columnIdentifier, BooleanProperty.SHOWN, shown);
    }

    @Override
    public void setCellShown(T node, DialogObjectConstant columnIdentifier, boolean shown) {
        setCellProperty(node, columnIdentifier, BooleanProperty.SHOWN, shown);
    }

    @Override
    public boolean isCellShown(T node, DialogObjectConstant columnIdentifier) {
        return getCellProperty(node, columnIdentifier, BooleanProperty.SHOWN);
    }    
    
	  ////////////////////////////////
	 /// Properties /////////////////
	////////////////////////////////

    @SuppressWarnings("unchecked")
	@Override
    public <U> void setProperty(Property<U> property, U propertyValue) {
    	for(T rootNode : treeData) {
    		Enumeration<T> e = rootNode.preorderEnumeration();
    		while(e.hasMoreElements()) {
    			T node = e.nextElement();
        		setNodeProperty(node, property, propertyValue);
    		}
    	}
    }
    
	@Override
    public <U> void setNodeProperty(T node, Property<U> property, U propertyValue) {
    	L row =  node.getListRow();
        for (DialogObjectConstant columnIdentifier : row.getFields()) {
            row.<U> setProperty(columnIdentifier, property, propertyValue);
        }
    }
    
    @SuppressWarnings("unchecked")
	@Override
    public <U> void setColumnProperty(DialogObjectConstant columnIdentifier, Property<U> propertyName, U value) {
    	for(T rootNode : treeData) {
    		Enumeration<T> e = rootNode.preorderEnumeration();
    		while(e.hasMoreElements()) {
    			T node = e.nextElement();
    			node.getListRow().<U> setProperty(columnIdentifier, propertyName, value);
    		}
    	} 
    }

	@Override
    public <U> void setCellProperty(T node, DialogObjectConstant columnIdentifier, Property<U> propertyName, U propertyValue) {
    	node.getListRow().<U> setProperty(columnIdentifier, propertyName, propertyValue);
    }

	@Override
    public <U> U getCellProperty(T node, DialogObjectConstant columnIdentifier, Property<U> propertyName) {
    	return node.getListRow().getProperty(columnIdentifier, propertyName);
    }

	  ////////////////////////////////
	 /// Sorting & Comparing ////////
	////////////////////////////////
	
    private Map<DialogObjectConstant, TreeNodeComparator> comparatorMap = new HashMap<>();
    
    @Override
    public void addTreeNodeComparator(DialogObjectConstant nodeConst, TreeNodeComparator comparator) {
        comparatorMap.put(nodeConst, comparator);
    }
	
	@Override 
	public void sortTree(List<T> nodeList) {

		if(!nodeList.isEmpty()) {
			T node = nodeList.get(0);
			if(node.getNodeConst() != null) {
				Comparator<TreeNode> comparator = comparatorMap.get(node.getNodeConst().getParent());
				if(comparator != null){
					Collections.sort(nodeList, comparator);
				}
			}
		}
		
		for(T rootNode : nodeList) {
			sortChildren(rootNode);
		}
		
	}

	@SuppressWarnings("unchecked")
	private void sortChildren(T node) {
		
		Enumeration <T> e = node.children();
		List<T> childNodeList = new ArrayList<>();
		
		while(e.hasMoreElements()) {
			childNodeList.add(e.nextElement());
		}
		
		node.removeAllChildren();
		
		if(childNodeList.size() != 0) {
			Comparator<TreeNode> comparator = comparatorMap.get(node.getNodeConst());
			if(comparator != null) {
				Collections.sort(childNodeList, comparator);
			}
		}
		
		for(T c : childNodeList) {
			sortChildren(c);
			node.add(c);
		}
	}

}
