/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces.tree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Observer;

import javax.swing.tree.TreeNode;

/**
 * Represents a list of root nodes. Extends ArrayList and notifies
 * registered observers about changes to the list.
 *
 * @param <E> the generated FacesTreeNode class
 */
@SuppressWarnings("serial")
public class FacesNodeList<E extends FacesTreeNode> extends ArrayList<E> {

	private NodeObservable obs = new NodeObservable();
	private Observer observer;
	
	/**
	 * Adds an observer to notify about changes to the node
	 * 
	 * @param obs the observer
	 */
	public void addObserver(Observer obs) {
		this.obs.addObserver(obs);
		observer = obs;
	}
	
	/**
	 * Deletes all observers registrered
	 */
	public void deleteObservers() {
		this.obs.deleteObservers();
		observer = null;
	}
	
	private void markElementsForObservation(Collection<? extends E> c) {
		for(E e: c) {
			markElementForObservation(e);
		}
	}
	
	@SuppressWarnings("unchecked")
	private void markElementForObservation(E e) {
		if(observer != null) {
			e.addObserver(observer);
			Enumeration<TreeNode> en  = e.breadthFirstEnumeration();
			while(en.hasMoreElements()) {
				((FacesTreeNode) en.nextElement()).addObserver(observer);
			}
		}
	}
	
	@Override
	public boolean add(E e) {
        boolean retval = super.add(e);
        obs.notifyObservers();
        markElementForObservation(e);
        return retval;
	}

	@Override
	public boolean remove(Object o) {
		boolean retval = super.remove(o);
		obs.notifyObservers();
        return retval;
	}

	@Override
	public boolean addAll(Collection<? extends E> c) {
        boolean retval = super.addAll(c);
        obs.notifyObservers();
        markElementsForObservation(c);
        return retval;
	}

	@Override
	public boolean addAll(int index, Collection<? extends E> c) {
        boolean retval =  super.addAll(index, c);
        obs.notifyObservers();
        markElementsForObservation(c);
        return retval;
	}

	@Override
	public boolean removeAll(Collection<?> c) {
        boolean retval = super.removeAll(c);
        obs.notifyObservers();
        return retval;
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		boolean retval = super.retainAll(c);
		obs.notifyObservers();
        return retval;
	}

	@Override
	public void clear() {
		super.clear();
		obs.notifyObservers();
	}

	@Override
	public E set(int index, E element) {
        E retval = super.set(index, element);
        obs.notifyObservers();
        markElementForObservation(element);
        return retval;
	}

	@Override
	public void add(int index, E element) {
        super.add(index, element);
        obs.notifyObservers();
        markElementForObservation(element);
	}

	@Override
	public E remove(int index) {
        E retval = super.remove(index);
        obs.notifyObservers();
        return retval;
	}

	
	
}
