/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.ConverterException;

import no.esito.log.Logger;
import no.esito.util.DateUtil;

/**
 * The Class DateTimeConverter is a custom converter which handles G9
 * style input, and reports errors with the component title in the details
 * part of the FacesMessage.
 */
public class DateTimeConverter extends javax.faces.convert.DateTimeConverter {

    private static final Logger log = Logger.getLogger(DateTimeConverter.class);

    /**
     * Set the converter time zone to the current default time zone.
     */
    public DateTimeConverter() {
        super();
        setTimeZone(TimeZone.getDefault());
    }

    @Override
    public String getAsString(FacesContext context, UIComponent component, Object value) {
        setPatternFromComponent(component);
        return super.getAsString(context, component, value);
    }

    @Override
    public Object getAsObject(FacesContext context, UIComponent component, String value) {
        if (value == null) {
            return null;
        }
        value = value.trim();
        if (value.isEmpty()) {
            return null;
        }
        try {
            setPatternFromComponent(component);
            DateFormat parser = null;
            if (getType().equals("date") || getType().equals("timestamp")) {
                parser = parseLocaleFormat(value);
                if (parser == null) {
                    parser = DateUtil.getParseFormat(value, getLocale());
                    if (parser == null) {
                        parser = getDateFormat();
                    }
                }
                if (parser == null) {
                    throw new ParseException("No date parser available", 0);
                }

            } else {
                parser = getDateFormat();
            }
            return parser.parse(value);
        } catch (Exception e) {
            throw new ConverterException(ConverterHelper.getConversionErrorMessage(component));
        }
    }

    /**
     * Try parsing the value according to the locale. If successful, return the
     * date format. Otherwise, null is returned.
     *
     * @param value the value.
     * @return the date format.
     */
    private DateFormat parseLocaleFormat(String value) {
        DateFormat localeDateFormat = (DateFormat.getDateInstance(DateFormat.MEDIUM, getLocale()));
        localeDateFormat.setLenient(false);
        try {
            localeDateFormat.parse(value);
        } catch (ParseException e) {
            localeDateFormat = null;
        }
        return localeDateFormat;
    }

    /**
     * Gets the date format according to the pattern and type.
     *
     * @return the date format
     */
    private DateFormat getDateFormat() {
        if (getPattern() == null && getType() == null) {
            throw new IllegalArgumentException("Either pattern or type must be specified.");
        }
        DateFormat df = null;
        String format = getPattern();
        if (format != null && !format.isEmpty()) {
            df = new SimpleDateFormat(format, getLocale());
        } else if (getType().equals("both"))
            df = DateFormat.getDateTimeInstance(getStyle(getDateStyle()), getStyle(getTimeStyle()), getLocale());
        else if (getType().equals("date"))
            df = DateFormat.getDateInstance(getStyle(getDateStyle()), getLocale());
        else if (getType().equals("time"))
            df = DateFormat.getTimeInstance(getStyle(getTimeStyle()), getLocale());
        else
            throw new IllegalArgumentException("Invalid type: " + getType());
        df.setLenient(false);
        return df;
    }

    /**
     * Gets the style to be used in DateFormat.
     *
     * @param name the name
     * @return the DateFormat style
     */
    private int getStyle(String name) {
        if (name.equals("short"))
            return DateFormat.SHORT;
        if (name.equals("medium"))
            return DateFormat.MEDIUM;
        if (name.equals("long"))
            return DateFormat.LONG;
        if (name.equals("full"))
            return DateFormat.FULL;
        return DateFormat.DEFAULT;
    }

    private void setPatternFromComponent(UIComponent component) {
        if (getPattern() == null) {
            String pattern = (String) component.getAttributes().get("pattern");
            log.trace("Pattern from component: " + pattern);
            setPattern(pattern);
        }
    }

}
