/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.faces.component.StateHolder;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;

import no.g9.support.G9Enumerator;
import no.g9.support.ObjectFactory;

/**
 * Converter class for enums based on G9Enumerator.
 *
 */
public class EnumConverter implements Converter, StateHolder {

    private String enumClassName = null;

    private boolean transientFlag = false;

    /**
     * The map from enum value title to enum value name.
     * The map will normally be initialized from a ResourceBundle, but can initialized locally for
     * testing purposes.
     */
    protected static final Map<String, String> titles = new HashMap<String, String>();

    /**
     * Special value used for handling null values.
     */
    public static final String NULL_VALUE = "EnumConverter.NULL_VALUE";

    private static final String NAME_METHOD = "name";
    private static final String GET_BY_NAME_METHOD = "getByName";

    /**
     * Get the enum class name for this converter.
     *
     * @return the enum class name.
     */
    public String getEnumClass() {
        return this.enumClassName;
    }

    /**
     * Set the enum class name for this converter.
     *
     * @param enumClass the new enum class name.
     */
    public void setEnumClass(final String enumClass) {
        this.enumClassName = enumClass;
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * javax.faces.convert.Converter#getAsObject(javax.faces.context.FacesContext
     * , javax.faces.component.UIComponent, java.lang.String)
     */
    @Override
    public Object getAsObject(FacesContext context, UIComponent component, String value) {

        Object retVal = null;

        if (context == null || component == null) {
            throw new NullPointerException();
        }
        if (value == null) {
            return null;
        }
        if (value.isEmpty() || EnumConverter.NULL_VALUE.equals(value)) {
            return null;
        }
        if (EnumConverter.titles.get(value) == null) {
        	throw new ConverterException("Unknown title for enum value: " + value);
        }
        try {
        	String enumValueName = EnumConverter.titles.get(value).trim();
        	if (isJavaEnumClass()) {
        		retVal = getEnumObject(enumValueName);
        	}
        	else {
        		G9Enumerator geVal = (G9Enumerator) ObjectFactory.newObject(enumClassName);
        		geVal.assignFromName(enumValueName);
        		retVal = geVal;
        	}
        } catch (ConverterException e) {
            throw e;
        } catch (Exception e) {
            throw new ConverterException(e);
        }
        return retVal;
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * javax.faces.convert.Converter#getAsString(javax.faces.context.FacesContext
     * , javax.faces.component.UIComponent, java.lang.Object)
     */
    @Override
    @SuppressWarnings("deprecation")
    public String getAsString(FacesContext context, UIComponent component, Object value) {

        if (context == null || component == null) {
            throw new NullPointerException();
        }
        if (value == null || EnumConverter.NULL_VALUE.equals(value)) {
            return EnumConverter.NULL_VALUE;
        }
        String name = null;
        if (value instanceof G9Enumerator) {
        	G9Enumerator enumVal = (G9Enumerator) value;
        	name = enumVal.currentAsName();
        	if (name == null) {
        		throw new ConverterException("Unknown enumeration value: " + enumVal.currentValue);
        	}
        }
        else if (value instanceof Enum) {
        	name = getEnumName(value);
        }
        else {
        	throw new ConverterException("Unknown class for enumeration value: " + value.getClass());
        }
        return getTitle(context, name);
    }

    private boolean isJavaEnumClass() {
    	try {
			Class<?> enumClass = Class.forName(enumClassName);
			if (enumClass.isEnum()) {
				return true;
			}
			return false;
		} catch (ClassNotFoundException e) {
			throw new ConverterException(e);
		}

    }

    private Enum<?> getEnumObject(String name) {
    	try {
    		Class<?> enumClass = Class.forName(enumClassName);
			Method getByName = enumClass.getMethod(GET_BY_NAME_METHOD, String.class);
			return (Enum<?>) getByName.invoke(null, name);
		} catch (Exception e) {
			throw new ConverterException(e);
		}
    }

    private String getEnumName(Object enumObject) {
    	try {
			Method name = enumObject.getClass().getMethod(NAME_METHOD, (Class<?>[])null);
			return (String) name.invoke(enumObject, (Object[])null);
		} catch (Exception e) {
			throw new ConverterException(e);
		}
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * javax.faces.component.StateHolder#saveState(javax.faces.context.FacesContext
     * )
     */
    @Override
    public Object saveState(FacesContext context) {
        return enumClassName;
    }

    /*
     * (non-Javadoc)
     *
     * @seejavax.faces.component.StateHolder#restoreState(javax.faces.context.
     * FacesContext, java.lang.Object)
     */
    @Override
    public void restoreState(FacesContext context, Object state) {
        enumClassName = (String) state;
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.faces.component.StateHolder#isTransient()
     */
    @Override
    public boolean isTransient() {
        return (transientFlag);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.faces.component.StateHolder#setTransient(boolean)
     */
    @Override
    public void setTransient(boolean transientFlag) {
        this.transientFlag = transientFlag;
    }

    private String getTitle(FacesContext context, String name) {

        String title = getTitleFromMap(name);
        if (title == null) {
            String key = makeKey(name);
            ResourceBundle bundle = getBundle(context);
            try {
                title = bundle.getString(key);
            } catch (MissingResourceException e) {
                title = "???" + key + "???";
            }
            EnumConverter.titles.put(title, name);
        }
        return title;
    }

    private String getTitleFromMap(String name) {
        if (name != null) {
            for (String title : EnumConverter.titles.keySet()) {
                if (name.equals(EnumConverter.titles.get(title))) {
                    return title;
                }
            }
        }
        return null;
    }

    private ResourceBundle getBundle(FacesContext context) {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        if (loader == null) {
            loader = this.getClass().getClassLoader();
        }
        String bundleName = context.getApplication().getMessageBundle();
        ResourceBundle bundle = ResourceBundle.getBundle(bundleName, context.getViewRoot().getLocale(), loader);
        return bundle;
    }

    private String makeKey(String name) {
        String enumName = getEnumClass();
        int i = enumName.lastIndexOf('.');
        if (i != -1) {
            enumName = enumName.substring(i+1);
        }
        return enumName + "." + name.toUpperCase() + ".title";
    }

}
