/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.el.MethodExpression;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.event.AbortProcessingException;
import javax.faces.event.ActionEvent;
import javax.faces.event.ActionListener;
import javax.faces.event.MethodExpressionActionListener;

import no.esito.jvine.action.GuiTask;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.jvine.view.faces.ApplicationInfo;
import no.esito.jvine.view.faces.FacesMessageUtil;
import no.esito.jvine.view.faces.MenuHelper;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.view.menu.Menu;
import no.g9.client.core.view.menu.MenuBase;
import no.g9.client.core.view.menu.MenuItem;
import no.g9.client.core.view.menu.Separator;
import no.g9.exception.G9BaseException;
import no.g9.message.MessageReplyType;

import org.icefaces.ace.component.menuseparator.MenuSeparator;
import org.icefaces.ace.component.submenu.Submenu;
import org.icefaces.ace.model.DefaultMenuModel;
import org.icefaces.ace.model.MenuModel;
import org.icefaces.util.JavaScriptRunner;

/**
 * Superclass for the generated JSF application view, which holds a reference to
 * the application controller.
 *
 * A reference to the JSF backing bean is set after construction.
 */
public abstract class FacesApplicationView extends AbstractApplicationView {

    private FacesApplicationBean applicationBean;

    private List<Menu> applicationMenu = new ArrayList<Menu>();

    /** The menu title prefix used for applications which are not available in a partitioned application. */
    public static final String NOT_AVAILABLE_PREFIX = "Not available";

    /**
     * Create a new application view for the given application controller.
     *
     * @param applicationController the application controller to reference.
     */
    public FacesApplicationView(ApplicationController applicationController) {
        super(applicationController);
    }

    /**
     * Get the referenced application bean.
     *
     * @return the application bean.
     */
    public FacesApplicationBean getApplicationBean() {
        return applicationBean;
    }

    /**
     * Set the referenced application bean.
     *
     * @param applicationBean the new application bean.
     */
    protected void setApplicationBean(FacesApplicationBean applicationBean) {
        this.applicationBean = applicationBean;
    }

    private String getDialogInternalName(DialogInstance instance) {
        return instance.getDialogConstant().getInternalName() + instance.getDialogInstanceNumber();
    }

    /**
     * Add the given menu to the application menu.
     *
     * @param menu - the menu to add
     */
    public void addMenu(Menu menu) {
        applicationMenu.add(menu);
    }

    /**
     * Add the given menus to the application menu.
     *
     * @param menus - the menus to add
     */
    public void addMenus(List<Menu> menus) {
        for (Menu menu : menus) {
            addMenu(menu);
        }
    }

    /**
     * Get the dynamic menu model for the application window.
     *
     * @return the list of menus.
     */
    public MenuModel getApplicationMenuModel() {
        MenuModel appMenu = new DefaultMenuModel();

        // Move the dialog menus to be merged into a separate list
        List<Menu> dialogMenu = getCurrentDialogMenu();
        List<Menu> dialogMergeMenu = new ArrayList<Menu>();
        for (Menu menu : applicationMenu) {
            for (Menu diaMenu : dialogMenu) {
                if (equalsMenu(menu, diaMenu)) {
                    dialogMergeMenu.add(diaMenu);
                }
            }
            dialogMenu.removeAll(dialogMergeMenu);
        }

        // Create the application menu, merging and inserting from the dialog
        Menu.MENU_TYPE insertPoint = MenuHelper.getInsertPoint(applicationMenu);
        for (Menu menu : applicationMenu) {
            if (menu.getType().equals(insertPoint)) {
                addDialogMenu(appMenu, dialogMenu);
            }
            FacesSubmenu submenu = mergeDialogMenu(menu, dialogMergeMenu);
            if (menu.getType() == Menu.MENU_TYPE.WINDOWMENU) {
                addWindowMenuDialogs(submenu);
            }
            appMenu.addSubmenu(submenu);
        }
        if (insertPoint == null) {
            addDialogMenu(appMenu, dialogMenu);
        }
        return appMenu;
    }

    /**
     * Get the dynamic list of menus for the application window.
     *
     * @return the list of menus.
     */
    public List<Submenu> getApplicationMenu() {
        MenuModel model = getApplicationMenuModel();
        List<Submenu> menus = new ArrayList<Submenu>();
        for (UIComponent uiComp : model.getSubmenus()) {
            menus.add((Submenu)uiComp);
        }
        return menus;
    }

    private FacesSubmenu mergeDialogMenu(Menu menu, List<Menu> dialogMenu) {
        FacesSubmenu submenu = asFacesSubmenu(menu);
        Iterator<Menu> it = dialogMenu.iterator();
        while (it.hasNext()) {
            Menu diaMenu = it.next();
            if (equalsMenu(menu, diaMenu)) {
                submenu.getChildren().addAll(asFacesSubmenu(diaMenu).getChildren());
                it.remove();
            }
        }
        return submenu;
    }

    private void addDialogMenu(MenuModel appMenu, List<Menu> dialogMenu) {
        for (Menu menu : dialogMenu) {
            FacesSubmenu submenu = asFacesSubmenu(menu);
            appMenu.addSubmenu(submenu);
        }
    }

    private boolean equalsMenu(Menu a, Menu b) {
        if (a.getType().equals(b.getType())
                && (a.getType() != Menu.MENU_TYPE.USERMENU || a.getId().equals(
                        b.getId()))) {
            return true;
        }
        return false;
    }

    private List<Menu> getCurrentDialogMenu() {
        List<Menu> dialogMenu = new ArrayList<Menu>();
        FacesDialogView view = getDialogView(getCurrentDialog(false));
        if (view != null) {
            dialogMenu.addAll(view.getDialogMenu());
        }
        return dialogMenu;
    }

    private void addWindowMenuDialogs(FacesSubmenu windowMenu) {
        List<FacesMenuItem> dialogs = new ArrayList<FacesMenuItem>();
        for(DialogInstance instance : getOpenDialogs()) {
            dialogs.add(0, createWindowMenuItem(getDialogInternalName(instance), getDialogTitle(instance), windowMenu.getStyleClass()));
        }
        if (!dialogs.isEmpty()) {
            if (windowMenu.getChildCount() > 0) {
                windowMenu.getChildren().add(new FacesMenuItem());
            }
            windowMenu.getChildren().addAll(dialogs);
        }
    }

    private FacesMenuItem createWindowMenuItem(String id, String title, String styleClass) {
        FacesMenuItem menuItem = new FacesMenuItem();
        menuItem.setId(id);
        menuItem.setValue(title);
        menuItem.setStyleClass(styleClass);
        menuItem.addActionListener(new ActionListener() {
            @Override
            public void processAction(ActionEvent e)
            throws AbortProcessingException {
                String compId = null;
                if (e.getSource() instanceof UIComponent) {
                    compId = ((UIComponent) e.getSource()).getId();
                }
                DialogInstance foundInstance = null;
                for (DialogInstance instance : getOpenDialogs()) {
                    if (getDialogInternalName(instance).equals(compId)) {
                        foundInstance = instance;
                    }
                }
                if (foundInstance != null) {
                    show(foundInstance);
                }
            }
        });
        MenuHelper.addAjaxBehaviorTo(menuItem);
        return menuItem;
    }

    /**
     * Create a FacesMenuItem from the given menu item.
     *
     * @param item the menu item to convert
     * @return a new FacesMenuItem
     */
    public FacesMenuItem asFacesMenuItem(MenuItem item) {
        FacesDialogView view = MenuHelper.getDialogView(item, this);
        FacesMenuItem facesItem = new FacesMenuItem();
        if (view != null) {
            facesItem.setId(item.getId());
            MenuHelper.addActionListeners(facesItem, item, view);
            MenuHelper.addAjaxBehaviorTo(facesItem);
            facesItem.setDisabled(!view.isEnabled(view.getDialogObjectFromName(item.getId())));
            facesItem.setIcon(MenuHelper.getMenuIconClass(item, view));
        }
        else {
            facesItem.setId(item.getDialogId() + "_" + item.getId());
            SystemMessage menuMessage = MenuHelper.getExternalMessage(item);
            if (menuMessage != null) {
                facesItem.setMessage(menuMessage.code());
                facesItem.setOnclick("send('" + facesItem.getMessage() + "')");
            }
        }
        facesItem.setValue(item.getTitle());
        facesItem.setMnemonic(item.getMnemonic());
        facesItem.setAccelerator(item.getAccelerator());
        facesItem.setStyleClass(MenuHelper.getStyleClass(item));
        return facesItem;
    }

    /**
     * Create a MenuSeparator from the given separator.
     *
     * @param separator the separator to convert
     * @return a new MenuSeparator
     */
    public MenuSeparator asFacesMenuSeparator(Separator separator) {
        MenuSeparator facesSeparator = new MenuSeparator();
        facesSeparator.setId(separator.getDialogId() + "_" + separator.getId());
        return facesSeparator;
    }

    /**
     * Create a FacesSubmenu from the given menu.
     * Menu items and sub menus are included.
     *
     * @param menu the menu to convert
     *
     * @return a new FacesSubmenu
     */
    public FacesSubmenu asFacesSubmenu(Menu menu) {
        FacesDialogView view = MenuHelper.getDialogView(menu, this);
        FacesSubmenu facesMenu = new FacesSubmenu();
        if (view != null) {
            facesMenu.setId(menu.getId());
            facesMenu.setDisabled(!view.isEnabled(view.getDialogObjectFromName(menu.getId())));
            facesMenu.setRendered(view.isShown(view.getDialogObjectFromName(menu.getId())));
            facesMenu.setIcon(MenuHelper.getMenuIconClass(menu, view));
        }
        else {
            facesMenu.setId(menu.getDialogId() + "_" + menu.getId());
        }
        facesMenu.setLabel(menu.getTitle());
        facesMenu.setMnemonic(menu.getMnemonic());
        facesMenu.setAccelerator(menu.getAccelerator());
        facesMenu.setStyleClass(MenuHelper.getStyleClass(menu));

        for (MenuBase menuBase : menu.getChildren()) {
            UIComponent facesComp = null;
            if (menuBase instanceof MenuItem) {
                facesComp = asFacesMenuItem((MenuItem) menuBase);
            } else if (menuBase instanceof Separator) {
                facesComp = asFacesMenuSeparator((Separator) menuBase);
            } else {
                facesComp = asFacesSubmenu((Menu) menuBase);
            }
            facesMenu.getChildren().add(facesComp);
            facesComp.setParent(facesMenu);
            if (view != null) {
                facesComp.setRendered(view.isShown(view.getDialogObjectFromName(menuBase.getId())));
            }
        }
        return facesMenu;
    }

    /**
     * Set the reply string before calling the default implementation.
     */
    @Override
    public void handleMessageReply(MessageReplyType reply) {
        getMessage().setReplString(getApplicationBean().getMessageInput());
        super.handleMessageReply(reply);
        for (SystemMessage outputMessage : getOutputMessages()) {
            forward(outputMessage);
        }
    }

    /**
     * Load a string from the JSF message bundle.
     *
     * @param bundleName - the JSF message file name, null if default bundle.
     * @param key - the lookup key.
     * @return the found message string, or an error string if not found.
     */
    public String getFacesMessage(String bundleName, String key) {
        return FacesMessageUtil.getBundleMessage(bundleName, key);
    }

    /**
     * Create a new action listener for the action expression given.
     *
     * @param actionExpression - the string representing the action expression.
     *
     * @return a MethodBinding for the action listener.
     */
    public ActionListener createActionListener(String actionExpression) {
        ActionListener listener = null;
        if (actionExpression != null) {
            FacesContext context = FacesContext.getCurrentInstance();
            MethodExpression methodExpression = context.getApplication().getExpressionFactory().createMethodExpression(
                    context.getELContext(), actionExpression, null, new Class[] { ActionEvent.class });
            listener = new MethodExpressionActionListener(methodExpression);
        }
        return listener;
    }

    @Override
    public void openResource(final String resourceId, final boolean purge) {
        guiInvoke(new GuiTask("openResource") {
            @Override
            public void run() {
                JavaScriptRunner.runScript(FacesContext.getCurrentInstance(),
                        "window.open('" + G9ResourceServlet.URL_SUFFIX + "?"
                        + G9ResourceServlet.ID_PARAM + "=" + resourceId
                        + "&purge=" + purge + "', 'myWindow');");
            }
        });
    }

    @Override
    public void forward(final SystemMessage message) {
        guiInvoke(new GuiTask("forward") {
            @Override
            public void run() {
                JavaScriptRunner.runScript(FacesContext.getCurrentInstance(),
                        "send('" + message.code() + "');");
            }
        });
    }

    /**
     * Get a list containing all static application menus for the applications
     * given in the applicationUrls map. The map key is the application name, and the
     * value is the URL for the application.
     *
     * @param setupUrl the URL where the setup JSON is found
     * @param dialog the owning dialog for created menus
     * @return the list of all menus for all application partitions
     */
    public static List<Menu> getAllStaticApplicationMenus(String setupUrl, DialogConstant dialog) {
        List<Menu> allMenus = new ArrayList<Menu>();
        Map<String, ApplicationInfo> applicationUrls = MenuHelper.getApplications(setupUrl);
        for (String applicationName : applicationUrls.keySet()) {
            try {
                ApplicationInfo info = applicationUrls.get(applicationName);
                if (info.title.startsWith(FacesApplicationView.NOT_AVAILABLE_PREFIX)) {
                    allMenus.add(MenuHelper.createApplicationMenu(applicationName, info.title, dialog));
                }
                else {
                    List<Menu> menus = MenuHelper.getApplicationMenus(info);
                    if (menus != null && !menus.isEmpty()) {
                        Menu appMenu = MenuHelper.createApplicationMenu(applicationName, info.title, dialog);
                        for (Menu menu : menus) {
                            appMenu.addMenu(menu);
                        }
                        allMenus.add(appMenu);
                    }
                }
            } catch (G9BaseException e) {
                log.info(e.getMessage(), e);
            }
        }
        return allMenus;
    }

}
