/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view.faces;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.validator.ValidateContext;
import no.g9.message.Message;
import no.g9.support.ResourceProvider;

/**
 * Various utility methods for JSF messages.
 */
public class FacesMessageUtil {

    /** The bundle lookup key for the conversion error message */
    public static final String CONVERSION_ERROR = "converter.conversionError";

    /** The bundle lookup key for the illegal parameter message */
    public static final String ILLEGAL_PARAMETER = "converter.illegalParameter";

    /** An installed provider for resources. */
    private static ResourceProvider<String> externalResourceProvider;

    /**
     * Returns the external resource provider.
     *
     * @return externalResourceProvider
     */
    public static ResourceProvider<String> getExternalResourceProvider() {
        return externalResourceProvider;
    }

    /**
     * Sets the external resource provider.
     *
     * @param resourceProvider New resource provider (or null, to erase)
     */
    public static void setExternalResourceProvider(ResourceProvider<String> resourceProvider) {
        externalResourceProvider = resourceProvider;
    }

    /**
     * Add a message to the JSF context.
     *
     * @param component
     *            the JSF component for the message
     * @param severity
     *            the severity of the message
     * @param key
     *            the message lookup key for the message bundle
     * @param title
     *            of the field
     */
    public static void addContextMessage(UIComponent component,
            FacesMessage.Severity severity, String key, String title) {
        FacesContext context = FacesContext.getCurrentInstance();
        String summary = FacesMessageUtil.getBundleMessage(key);
        String clientID = null;
        if (component != null) {
            clientID = component.getClientId(context);
        }
        FacesMessage facesMessage = new FacesMessage(severity, summary, title);
        context.addMessage(clientID, facesMessage);
    }

    /**
     * Recursively check component children for a clientId match.
     *
     * @param context
     *            FacesContext context
     * @param uic
     *            UIComponent
     * @param key
     *            DialogObjectConstant internalName
     * @return clientId
     */
    private static String getCurrentCompClientId(FacesContext context,
            UIComponent uic, String key) {
        if (uic.getId().equals(key)) {
            return uic.getClientId(context);
        }
        for (UIComponent uiComponent : uic.getChildren()) {
            if (uiComponent.getId().equals(key)) {
                return uiComponent.getClientId(context);
            }
            String tmp = getCurrentCompClientId(context, uiComponent, key);
            if (tmp != null) {
                return tmp;
            }
        }

        return null;
    }

    /**
     * Add a message to the message context.
     *
     * @param message
     *            the message to add
     */
    public static void addContextMessage(Message message) {
        FacesContext context = FacesContext.getCurrentInstance();
        String detail = message.getMessageText();
        String summary = message.getTitle();
        ValidateContext con = (ValidateContext) message.getContext();
        String clientId = null;
        if (con != null) {
            String internalName = con.getDialogObjectConstant()
                    .getInternalName();
            clientId = getCurrentCompClientId(context, context.getViewRoot(),
                    internalName);
        }

        FacesMessage facesMessage = new FacesMessage(summary, detail);
        context.addMessage(clientId, facesMessage);
    }

    /**
     * @return true if there are any messages in the JSF context.
     */
    public static boolean hasContextMessages() {
        FacesContext context = FacesContext.getCurrentInstance();
        if (context == null) return false;
        return null != context.getMaximumSeverity();
    }

    /**
     * @param target the dialog object to check
     * @return true if there are any messages in the JSF context for the given
     *         target
     */
    public static boolean hasContextMessages(DialogObjectConstant target) {
        return !FacesMessageUtil.getContextMessages(target).isEmpty();
    }

    /**
     * Return a list of JSF context messages for the given target.
     *
     * @param target
     *            the dialog object to check. If null, all messages which do not
     *            belong to a specific JSF component are returned
     * @return the list of messages, possibly empty
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public static List getContextMessages(DialogObjectConstant target) {
        List messages = new LinkedList();
        String clientId = null;
        if (target != null) {
            clientId = FacesMessageUtil.getClientId(target);
        }
        FacesContext context = FacesContext.getCurrentInstance();
        Iterator<FacesMessage> msgIter = context.getMessages(clientId);
        while (msgIter.hasNext()) {
            FacesMessage msg = msgIter.next();
            messages.add(msg.getSummary());
        }
        return messages;
    }

    private static String getClientId(DialogObjectConstant target) {
        return "iceform:" + target.getInternalName();
    }

    /**
     * Clear all messages from the JSF context.
     */
    public static void clearContextMessages() {
        FacesContext context = FacesContext.getCurrentInstance();
        if (context == null) return;
        Iterator<?> messages = context.getMessages();
        while (messages.hasNext()) {
            messages.next();
            messages.remove();
        }
    }

    /**
     * Load a string from the default JSF message bundle.
     *
     * @param key the lookup key
     * @return the found message string, or an error string if not found
     */
    public static String getBundleMessage(String key) {
        return FacesMessageUtil.getBundleMessage(null, key);
    }

    /**
     * Load a string from the JSF message bundle.
     *
     * @param bundleName the JSF message file name, null if default bundle
     * @param key the lookup key
     * @return the found message string, or an error string if not found
     */
    public static String getBundleMessage(String bundleName, String key) {
        Locale locale = null;
        FacesContext context = FacesContext.getCurrentInstance();
        if (context != null) {
            if (bundleName == null) {
                bundleName = context.getApplication().getMessageBundle();
            }
            locale = context.getViewRoot().getLocale();
        }
        return FacesMessageUtil.getBundleMessage(bundleName, key, locale);
    }

    /**
     * Load a string from the JSF message bundle.
     *
     * @param bundleName the JSF message file name, null if default bundle
     * @param key the lookup key
     * @param locale the message locale. If null, the JVM default is used
     * @return the found message string, or an error string if not found
     */
    public static String getBundleMessage(String bundleName, String key, Locale locale) {
        String text = getExternalResource(bundleName, key, locale);
        if ( text != null ) {
        	return text;
        }
        if (locale == null) {
            locale = Locale.getDefault();
        }
        ResourceBundle bundle = ResourceBundle.getBundle(bundleName, locale, FacesMessageUtil.getCurrentClassLoader(key));
        try {
            text = bundle.getString(key);
        } catch (MissingResourceException e) {
            text = "???" + key + "???";
        }
        return text;
    }

    private static ClassLoader getCurrentClassLoader(Object defaultObject) {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        if (loader == null) {
            loader = defaultObject.getClass().getClassLoader();
        }
        return loader;
    }

    private static String getExternalResource(String bundleName, String key, Locale locale) {
        if (externalResourceProvider == null) {
        	return null;
        }
        return externalResourceProvider.getResource(bundleName, key, locale);
    }

}
