/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import no.g9.client.core.controller.DialogObjectConstant;


/**
 * The AutoCompleteConfig is a configuration for a given autocomplete widget from the dialog model.
 * The configuration values may be changed at runtime, but note that changing the result column names
 * to an array with less entries than given in the dialog model may lead to an error, as the markup
 * expects a given number of result columns present in the search result list.  
 */
public class AutoCompleteConfig {

    private DialogObjectConstant widget;

    private final List<String> resultColumns = new ArrayList<>();

    private int minChars;

    private int maxResults;

	/**
	 * Create a new AutoCompleteConfig with values for all fields.
	 *
	 * @param widget the autocomplete widget from the dialog model
	 * @param resultColumns the names of the result columns (to be shown in the result list)
	 * @param minChars the minimum number of characters to be entered before a search is triggered
	 * @param maxResults the maximum number of result list entries to return
	 */
	public AutoCompleteConfig(DialogObjectConstant widget, String[] resultColumns, int minChars, int maxResults) {
		super();
		this.widget = widget;
		setResultColumns(resultColumns);
		this.minChars = minChars;
		this.maxResults = maxResults;
	}

	/**
	 * The AutoComplete widget.
	 *
	 * @return the widget
	 */
	public DialogObjectConstant getWidget() {
		return widget;
	}

	/**
	 * @param widget the AutoComplete widget
	 */
	public void setWidget(DialogObjectConstant widget) {
		this.widget = widget;
	}

	/**
	 * The result columns are the names of the columns used in the result list for the AutoComplete search.
	 *
	 * @return the resultColumns
	 */
	public List<String> getResultColumns() {
		return Collections.unmodifiableList(resultColumns);
	}

	/**
	 * @param resultColumns the resultColumns to set
	 */
	public void setResultColumns(String[] resultColumns) {
		this.resultColumns.clear();
		if (resultColumns != null) {
			this.resultColumns.addAll(Arrays.asList(resultColumns));
		}
	}

	/**
	 * The minimum number of characters to be entered before a search is triggered.
	 * The default is 1 if this is not given from the dialog model.
	 *
	 * @return the minChars
	 */
	public int getMinChars() {
		return minChars;
	}

	/**
	 * @param minChars the minChars to set
	 */
	public void setMinChars(int minChars) {
		this.minChars = minChars;
	}

	/**
	 * The maximum number of result list entries to return from the search.
	 * If 0, the result list is not limited.
	 *
	 * @return the maxResults
	 */
	public int getMaxResults() {
		return maxResults;
	}

	/**
	 * @param maxResults the maxResults to set
	 */
	public void setMaxResults(int maxResults) {
		this.maxResults = maxResults;
	}

}
