/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.util;

import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import javax.faces.model.SelectItem;

import no.esito.util.DateUtil;
import no.g9.client.core.controller.ApplicationController;
import no.g9.client.core.controller.DialogController;
import no.g9.service.JGrapeService;
import no.g9.support.ClientContext;
import no.g9.support.EFindMethod;
import no.g9.support.FindData;

import com.google.common.base.Strings;

/**
 * The Class AutoCompleteHelper is a helper class for the AutoComplete custom widget.
 * It implements the AutoCompleteProvider interface, and uses a database query based
 * on the given search term to update the result list.
 */
@SuppressWarnings("rawtypes")
public final class AutoCompleteHelper implements AutoCompleteProvider {

    @Override
    public void updateResultList(Object searchTerm, List<SelectItem> resultList, AutoCompleteConfig config, DialogController controller) {
        resultList.clear();
        String value = (searchTerm != null) ? searchTerm.toString() : "";
        String labelColumn = config.getWidget().getAttribute().toString();
        Object[] queryAndValues = QueryBuilder.buildAutoCompleteQueryWithParameters(labelColumn, config.getResultColumns(), value, controller);
        Object result = AutoCompleteHelper.getResultForQueryWithParameters(queryAndValues, config.getMaxResults(), controller);
        List<Object[]> objArrList = AutoCompleteHelper.getObjectArrayList(result);
        AutoCompleteHelper.formatDateResults(objArrList, labelColumn, config.getResultColumns(), controller);
        AutoCompleteHelper.convertTypesToString(objArrList, labelColumn, config.getResultColumns(), controller);

        // The value attribute is always located at position 0
        AutoCompleteHelper.updateDisplayList(objArrList, resultList, config.getMaxResults(), 0);
    }

    private static void convertTypesToString(List<Object[]> objArrList, String valueAttribute, List<String> columnAttributes, DialogController controller) {
        RoleNameHelper roleNameHelper = new RoleNameHelper(controller);
        List<Object[]> colNamesAndDataTypes = QueryBuilder.getColumnNamesAndDataTypes(valueAttribute, columnAttributes,
            roleNameHelper);
        int objIndex = 0;
        for (Object[] objects : colNamesAndDataTypes) {
            String type = String.valueOf(objects[1]);
            if (Strings.isNullOrEmpty(type))
                continue;
            if (type.equals(Integer.class.getCanonicalName()) || type.equals(Long.class.getCanonicalName())) {
                for (Object[] object : objArrList) {
                    object[objIndex] = String.valueOf(object[objIndex]);
                }
            }
            objIndex++;
        }
    }

    private static void formatDateResults(List<Object[]> objArrList, String valueAttribute, List<String> columnAttributes, DialogController controller) {

        RoleNameHelper roleNameHelper = new RoleNameHelper(controller);
        List<Object[]> colNamesAndDataTypes = QueryBuilder.getColumnNamesAndDataTypes(valueAttribute, columnAttributes,
            roleNameHelper);
        int i = 0;
        String displayRule = "";
        for (Object[] resultArray : colNamesAndDataTypes) {
            if (!displayRule.equals(""))
                continue;
            if ((resultArray[1].equals("java.util.Date"))) {
                displayRule = resultArray[2].toString();
            }
            i++;
        }
        for (Object[] object : objArrList) {
            if ((!displayRule.equals("")) && (object[i - 1] != null)) {
                String dateStr = DateUtil.dateToString((Date) object[i - 1], displayRule);
                object[i - 1] = dateStr;
            }
        }
    }

    /**
     * Update display list for the given new content.
     *
     * @param localObjList the new content list
     * @param displayList the display list
     * @param noOfRows the no of rows
     */
    private static void updateDisplayList(List<Object[]> localObjList, List<SelectItem> displayList, int noOfRows,
        int valueDisplayPosition) {
        displayList.clear();
        for (Object[] objectVar : localObjList) {
            if (noOfRows > 0 && displayList.size() >= noOfRows) {
                break;
            }
            String title = (objectVar[valueDisplayPosition] != null) ? objectVar[valueDisplayPosition].toString() : "";
            SelectItem selectItem = new SelectItem(objectVar, title);
            displayList.add(selectItem);
        }
    }

    /**
     * Gets the object array list for a Given Object result.
     *
     * @param result the result
     * @return the object array list
     */
	@SuppressWarnings("unchecked")
	private static List<Object[]> getObjectArrayList(Object result) {
		List<Object[]> objArrList = new ArrayList<Object[]>();
		List localList = (List) result;
		for (Object obj : localList) {
			// if the query is a single column query the result will be list of
			// that object like List<Object>
			// To make the result consistent convert to List<Objec[]> because of
			// multi column
			// it will be always List<Object[]>
			if (obj instanceof Object[]) {
				objArrList = (List<Object[]>) result;
			} else {
				Object[] val = new Object[] { obj };
				objArrList.add(val);
			}
		}
		Collections.sort(objArrList, new AutoCompleteComparator());
		return objArrList;
	}

    /**
     * Gets the result for query.
     *
     * @param applicationView the application view
     * @param queryString the query string
     * @return the result for query
     */
    @SuppressWarnings("unchecked")
    private static Object getResultForQueryWithParameters(Object[] queryAndValues, int noOfRows, DialogController controller) {
        FindData findData = new FindData();
        findData.setFindMethod(EFindMethod.QUERY);
        findData.setQuery((String) queryAndValues[0]);
        findData.setQueryParameters((List<Object>) queryAndValues[1]);
        findData.setMaxResults(noOfRows);
        ApplicationController applicationController = controller.getApplicationController();
        JGrapeService serviceProxy = applicationController.getServiceProxy();
        ClientContext ctx = applicationController.getClientContext();
        Object result = serviceProxy.findAll(findData, ctx, null);
        return result;
    }

}

/**
 * The Class AutoCompleteComparator.
 */
class AutoCompleteComparator implements Comparator<Object[]> {

	private Collator col;

	/**
	 * Instantiates a new auto complete comparator.
	 */
	AutoCompleteComparator() {
		col = Collator.getInstance();
		col.setStrength(Collator.PRIMARY);
	}

	/**
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	@Override
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public int compare(Object[] object1, Object[] object2) {
		Comparable v1 = (Comparable) object1[0];
		Comparable v2 = (Comparable) object2[0];
		int result = 0;

		if (v1 == null) { // Test for null
			result = (v2 == null) ? 0 : -1;
		} else if (v2 == null) {
			result = 1;
		} else if (v1 instanceof String) { // Locale compare of Strings
			result = col.compare(v1, v2);
		} else {
			result = v1.compareTo(v2); // "Natural" ordering of comparables.
		}
		return result;
	}

}
