/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.TimeZone;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.ConverterException;
import javax.faces.convert.DateTimeConverter;

import no.esito.log.Logger;
import no.esito.util.DateUtil;
import no.g9.support.FormatHelper;

/**
 * The Class DateConverter is a custom DateTimeConverter which handles G9
 * display rules.
 *
 * <p>
 * The <code>getAsObject()</code> method parses a String into a
 * <code>java.util.Date</code>, according to the following algorithm:
 * </p>
 * <ul>
 * <li>If the specified String is null, return a <code>null</code>. Otherwise,
 * trim leading and trailing whitespace before proceeding.</li>
 * <li>If the specified String - after trimming - has a zero length, return
 * <code>null</code>.</li>
 * <li>If the <code>locale</code> property is not null, use that
 * <code>Locale</code> for managing parsing. Otherwise, use the
 * <code>Locale</code> from the <code>UIViewRoot</code>.</li>
 * <li>If a <code>displayRule</code> has been specified, its syntax must conform
 * the rules specified by <code>java.text.SimpleDateFormat</code>. Such a
 * displayRule will be used to parse, and the <code>type</code>,
 * <code>dateStyle</code>, and <code>timeStyle</code> properties will be
 * ignored.</li>
 * <li>If a <code>displayRule</code> has not been specified, parsing will be
 * based on the <code>type</code> property, which expects a date value, a time
 * value, or both. Any date and time values included will be parsed in
 * accordance to the styles specified by <code>dateStyle</code> and
 * <code>timeStyle</code>, respectively.</li>
 * <li>If a <code>timezone</code> has been specified, it must be passed to the
 * underlying <code>DateFormat</code> instance. Otherwise the "GMT" timezone is
 * used.</li>
 * <li>In all cases, parsing must be non-lenient; the given string must strictly
 * adhere to the parsing format.</li>
 * </ul>
 *
 * <p>
 * The <code>getAsString()</code> method expects a value of type
 * <code>java.util.Date</code> (or a subclass), and creates a formatted String
 * according to the following algorithm:
 * </p>
 * <ul>
 * <li>If the specified value is null, return a zero-length String.</li>
 * <li>If the specified value is a String, return it unmodified.</li>
 * <li>If the <code>locale</code> property is not null, use that
 * <code>Locale</code> for managing formatting. Otherwise, use the
 * <code>Locale</code> from the <code>UIViewRoot</code>.</li>
 * <li>If a <code>timezone</code> has been specified, it must be passed to the
 * underlying <code>DateFormat</code> instance. Otherwise the "GMT" timezone is
 * used.</li>
 * <li>If a <code>displayRule</code> has been specified, its syntax must conform
 * the rules specified by <code>java.text.SimpleDateFormat</code>. Such a
 * displayRule will be used to format, and the <code>dataType</code>,
 * <code>dateStyle</code>, and <code>timeStyle</code> properties will be
 * ignored.</li>
 * <li>If a <code>displayRule</code> has not been specified, formatting will be
 * based on the <code>dataType</code> property, which includes a date value, a
 * time value, or both into the formatted String. Any date and time values
 * included will be formatted in accordance to the styles specified by
 * <code>dateStyle</code> and <code>timeStyle</code>, respectively.</li>
 * </ul>
 */
public class DateConverter extends DateTimeConverter implements G9Converter {

    private ConverterHelper converterHelper;

    private static final Logger log = Logger.getLogger(DateConverter.class);

    /**
     * Create a new G9 date converter.
     * Sets the converter time zone to the current default time zone.
     */
    public DateConverter() {
        super();
        converterHelper = new ConverterHelper();
        converterHelper.setDataType("date");
        converterHelper.setDisplayRule("");
        setTimeZone(TimeZone.getDefault());
    }

    /**
     * Convert the specified string value, which is associated with the
     * specified UIComponent, into a model data object that is appropriate for
     * being stored during the Apply Request Values phase of the request
     * processing life cycle.
     *
     * @see javax.faces.convert.DateTimeConverter#getAsObject(javax.faces.context.FacesContext,
     *      javax.faces.component.UIComponent, java.lang.String)
     */
    @Override
    public Object getAsObject(FacesContext context, UIComponent component, String value) {

        if (context == null || component == null) {
            throw new NullPointerException();
        }

        if (log.isTraceEnabled()) {
            log.trace("Parsing value \"" + value + "\", display rule \"" + getDisplayRule() + "\", data type \""
                    + getDataType() + "\"");
        }

        if (value == null)
            return null;
        value = value.trim();
        if (value.length() < 1)
            return null;

        Locale locale = getLocale(context);
        DateFormat parser = null;
        try {
            if (getDataType().equals("date") || getDataType().equals("timestamp")) {
                parser = parseLocaleFormat(value);
                if (parser == null) {
                    parser = DateUtil.getParseFormat(value, locale);
                    if (parser == null) {
                        parser = getDateFormat(context, locale);
                    }
                }
                if (parser == null) {
                    throw new ParseException("", 0);
                }

            } else {
                parser = getDateFormat(context, locale);
            }
            if (null != getTimeZone()) {
                parser.setTimeZone(getTimeZone());
            }
            return parser.parse(value);
        } catch (Exception e) {
            throw new ConverterException(converterHelper.getConversionErrorMessage());
        }
    }

    /**
     * Parses the locale format.
     *
     * @param value the value.
     * @return the date format.
     */
    private DateFormat parseLocaleFormat(String value) {
        DateFormat localeDateFormat = (DateFormat.getDateInstance(DateFormat.MEDIUM, getLocale()));
        localeDateFormat.setLenient(false);
        try {
            localeDateFormat.parse(value);
        } catch (ParseException e) {
            localeDateFormat = null;
        }
        return localeDateFormat;
    }

    @Override
    public String getAsString(FacesContext context, UIComponent component, Object value) {

        if (context == null || component == null) {
            throw new NullPointerException();
        }

        if (value == null)
            return "";
        if (value instanceof String)
            return (String) value;
        Locale locale = getLocale(context);
        return dateToString(context, value, locale, component);
    }

    /**
     * Converts a date to a string according to this converter.
     *
     * @param context The context
     * @param value The value
     * @param locale The locale
     * @param component The UIComponent
     * @return the string representation of the date
     */
    String dateToString(FacesContext context, Object value, Locale locale, UIComponent component) {
        DateFormat formatter = getDateFormat(context, locale);
        if (null != getTimeZone()) {
            formatter.setTimeZone(getTimeZone());
        }
        return formatter.format(value);
    }

    /**
     * Gets the date format.
     *
     * @param context the context
     * @param locale the locale
     * @return the date format
     */
    DateFormat getDateFormat(FacesContext context, Locale locale) {
        if (getDisplayRule() == null && getDataType() == null) {
            throw new IllegalArgumentException("Either displayRule or datatype must be specified.");
        }
        DateFormat df = null;
        if (getDisplayRule() != null && !getDisplayRule().equals("")) {
            String format = FormatHelper.getDatetimeFormat(getDisplayRule());
            if (format == null) {
                throw new IllegalArgumentException("Unsupported display rule \"" + getDisplayRule() + "\"");
            }
            if (log.isTraceEnabled()) {
                log.trace("Using format \"" + format + "\" for display rule \"" + getDisplayRule() + "\"");
            }
            df = new SimpleDateFormat(format, locale);
        } else if (getDataType().equals("both") || getDataType().equals("timestamp"))
            df = DateFormat.getDateTimeInstance(getStyle(getDateStyle()), getStyle(getTimeStyle()), locale);
        else if (getDataType().equals("date"))
            df = DateFormat.getDateInstance(getStyle(getDateStyle()), locale);
        else if (getDataType().equals("time"))
            df = DateFormat.getTimeInstance(getStyle(getTimeStyle()), locale);
        else
            throw new IllegalArgumentException("Invalid datatype: " + getDataType());
        df.setLenient(false);
        return df;
    }

    /**
     * Gets the locale.
     *
     * @param context the context
     * @return the locale
     */
    private Locale getLocale(FacesContext context) {
        Locale locale = super.getLocale();
        if (locale == null)
            locale = context.getViewRoot().getLocale();
        return locale;
    }

    /**
     * Gets the style.
     *
     * @param name the name
     * @return the style
     */
    private int getStyle(String name) {
        if (name.equals("default"))
            return 2;
        if (name.equals("short"))
            return 3;
        if (name.equals("medium"))
            return 2;
        if (name.equals("long"))
            return 1;
        if (name.equals("full"))
            return 0;
        throw new ConverterException("Invalid style '" + name + "'");
    }

    /**
     * <p>
     * Set the format displayRule to be used when formatting and parsing dates
     * and times. Valid values are those supported by
     * <code>java.text.SimpleDateFormat</code>. An invalid value will cause a
     * {@link ConverterException} when <code>getAsObject()</code> or
     * <code>getAsString()</code> is called.
     * </p>
     *
     * @param displayRule The new format displayRule
     */
    @Override
    public void setDisplayRule(String displayRule) {
        clearInitialState();

        converterHelper.setDisplayRule(displayRule);
        setPattern(FormatHelper.getDatetimeFormat(displayRule));
    }

    @Override
    public void setDataType(String dataType) {
        clearInitialState();

        converterHelper.setDataType(dataType);
    }

    @Override
    public String getDataType() {
        return converterHelper.getDataType();
    }

    @Override
    public String getDisplayRule() {
        return converterHelper.getDisplayRule();
    }

    @Override
    public void setTitle(String title) {
        clearInitialState();

        converterHelper.setTitle(title);
    }

    @Override
    public String getTitle() {
        return converterHelper.getTitle();
    }

    @Override
    public Object saveState(FacesContext context) {

        if (context == null) {
            throw new NullPointerException();
        }
        if (!initialStateMarked()) {
            Object values[] = new Object[2];
            values[0] = converterHelper;
            values[1] = super.saveState(context);
            return (values);
        }
        return null;
    }

    @Override
    public void restoreState(FacesContext context, Object state) {

        if (context == null) {
            throw new NullPointerException();
        }
        if (state != null) {
            Object values[] = (Object[]) state;
            converterHelper = (ConverterHelper) values[0];
            super.restoreState(context, values[1]);
        }
    }

}
