/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import javax.faces.context.FacesContext;
import javax.faces.event.ActionEvent;

import no.esito.jvine.view.faces.FacesMessageUtil;
import no.g9.client.core.controller.DialogInstance;
import no.g9.message.Message;
import no.g9.message.MessageReplyType;
import no.g9.message.MessageType;
import no.g9.message.MessageTypeEnum;
import no.g9.message.ReplySetType;
import no.g9.support.ResourceProvider;

import org.icefaces.ace.event.CloseEvent;


/**
 * The superclass for the JSF backing bean of the application.
 */
public abstract class FacesApplicationBean {

    private FacesApplicationView applicationView;

    private String messageInput;

    private static String applicationBeanName = null;

    /**
     * Create a new application backing bean, and setting the references
     * between the bean and the application view.
     *
     * @param applicationBeanName the name of the JSF application bean
     * @param applicationView the JSF application view
     */
    public FacesApplicationBean(String applicationBeanName, FacesApplicationView applicationView) {
        setApplicationBeanName(applicationBeanName);
        setApplicationView(applicationView);
        applicationView.setApplicationBean(this);
    }

    /**
     * Note that this constructor should not normally be used, it is only to
     * satisfy the Serializable interface of its subclasses.
     */
    protected FacesApplicationBean() {
    	super();
    }

    private synchronized void setApplicationBeanName(String name) {
        FacesApplicationBean.applicationBeanName= name;
    }

    /**
     * @return the JSF application view.
     */
    public FacesApplicationView getApplicationView() {
        return applicationView;
    }

    /**
     * Set the JSF application view.
     *
     * @param applicationView the new JSF application view
     */
    public void setApplicationView(FacesApplicationView applicationView) {
        this.applicationView = applicationView;
    }

    /**
     * Get the JSF message from the message bundle.
     *
     * @param bundleName - the JSF message file name, null if default bundle
     * @param key - the lookup key
     * @return the found message string, or an error string if not found
     */
    public String getFacesMessage(String bundleName, String key) {
        return FacesMessageUtil.getBundleMessage(bundleName, key);
    }

    /**
     * Get the message text for the given key. The default is to look for the
     * message in the generated messages.properties file for the application.
     *
     * This method is used from the generated xhtml markup, to allow
     * for providing other message sources than the default resource bundle.
     *
     * @param key the message key for the message
     * @return the message text for the given key
     */
    public String msg(String key) {
    	return getFacesMessage(null, key);
    }

    /**
     * Get the JSF facelet dynamic include path for the current dialog.
     *
     * @return the dynamic include path for the current dialog
     */
    public String getDialogIncludePath() {
        return buildDialogIncludePath();
    }

    /**
     * @return the number of open dialog boxes in the application
     */
    public int getNumberOfOpenDialogBoxes() {
    	return getApplicationView().getOpenDialogBoxes().size();
    }

    /**
     * Get the JSF facelet dynamic include paths for the dialog boxes.
     *
     * @return the dynamic include paths for the dialog boxes
     */
    public String[] getDialogBoxIncludePaths() {
        return buildDialogBoxIncludePaths();
    }

    /**
     * Get the JSF facelet dynamic include path for the toolbar of the
     * current dialog.
     *
     * @return the dynamic include path for the current dialog toolbar
     */
    protected String getDialogToolbarIncludePath() {
        return buildDialogToolbarIncludePath();
    }

    /**
     * Build the JSF facelet dynamic include path for the current dialog.
     *
     * @return the dynamic include path
     */
    protected abstract String buildDialogIncludePath();

    /**
     * Build the JSF facelet dynamic include paths for the dialog boxes.
     *
     * @return the dynamic include paths
     */
    protected abstract String[] buildDialogBoxIncludePaths();

    /**
     * Build the JSF facelet dynamic include path for the toolbar of the
     * current dialog.
     *
     * @return the dynamic include path
     */
    protected abstract String buildDialogToolbarIncludePath();

    /**
     * The event handler for the message box Close button.
     *
     * @param event the JSF event
     */
    public void messageBox_CloseClicked(CloseEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_CANCEL);
    }

    /**
     * The event handler for the message box OK button.
     *
     * @param event the JSF event
     */
    public void messageBox_OkClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_OK);
    }

    /**
     * The event handler for the message box Yes button.
     *
     * @param event the JSF event
     */
    public void messageBox_YesClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_YES);
    }

    /**
     * The event handler for the message box No button.
     *
     * @param event the JSF event
     */
    public void messageBox_NoClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_NO);
    }

    /**
     * The event handler for the message box Cancel button.
     *
     * @param event the JSF event
     */
    public void messageBox_CancelClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_CANCEL);
    }

    /**
     * The event handler for the message box Abort button.
     *
     * @param event the JSF event
     */
    public void messageBox_AbortClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_ABORT);
    }

    /**
     * The event handler for the message box Retry button.
     *
     * @param event the JSF event
     */
    public void messageBox_RetryClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_RETRY);
    }

    /**
     * The event handler for the message box Ignore button.
     *
     * @param event the JSF event
     */
    public void messageBox_IgnoreClicked(ActionEvent event) {
        getApplicationView().handleMessageReply(MessageReplyType.REPLY_IGNORE);
    }

    /**
     * @return true if a message should be shown, false otherwise
     */
    public boolean getShowMessage() {
        return getApplicationView().getShowMessage();
    }

    /**
     * Dummy setter for JSF.
     *
     * @param on ignored
     */
    public void setShowMessage(boolean on) {
        // Dummy
    }

    /**
     * Get the message box title. If the message has a title, it will be used.
     * If not, the application title is used.
     *
     * @return the title for the message box
     */
    public String getMessageBoxTitle() {
        String appName = getApplicationView().getApplicationController().getApplicationName();
        String title = getFacesMessage(null, appName.toLowerCase() + ".title");
        if (getShowMessage() && !Message.isEmpty(getApplicationView().getMessage().getTitle())) {
            title = getApplicationView().getMessage().getTitle();
        }
        return title;
    }

    /**
     * @return the URL for the message box icon
     */
    public String getMessageBoxIcon() {
        String appName = getApplicationView().getApplicationController().getApplicationName();
        MessageType type = MessageTypeEnum.NONE;
        if (getShowMessage()) {
            type = getApplicationView().getMessage().getMsgType();
        }
        return getFacesMessage(null, appName.toLowerCase() + "." + type + ".image");
    }

    /**
     * @return the Alt text for the message box icon
     */
    public String getMessageBoxAltText() {
        String appName = getApplicationView().getApplicationController().getApplicationName();
        MessageType type = MessageTypeEnum.NONE;
        if (getShowMessage()) {
            type = getApplicationView().getMessage().getMsgType();
        }
        return getFacesMessage(null, appName.toLowerCase() + "." + type + ".alt");
    }


    /**
     * @return the message text if there is a current message, null otherwise
     */
    public String getMessageText() {
        if (getShowMessage()) {
            return getApplicationView().getMessage().getMessageText();
        }
        return null;
    }

    /**
     * @return true if the message input field is to be shown
     */
    public boolean getShowMessageInput() {
        if (getShowMessage()) {
            if (ReplySetType.REPLSET_STRING.equals(getApplicationView().getMessage().getValidReplies())) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return true if the message ok button is to be shown
     */
    public boolean getShowMessageOkButton() {
        return showMessageReply(MessageReplyType.REPLY_OK);
    }

    /**
     * @return true if the message yes button is to be shown
     */
    public boolean getShowMessageYesButton() {
        return showMessageReply(MessageReplyType.REPLY_YES);
    }

    /**
     * @return true if the message no button is to be shown
     */
    public boolean getShowMessageNoButton() {
        return showMessageReply(MessageReplyType.REPLY_NO);
    }

    /**
     * @return true if the message cancel button is to be shown
     */
    public boolean getShowMessageCancelButton() {
        return showMessageReply(MessageReplyType.REPLY_CANCEL);
    }

    /**
     * @return true if the message abort button is to be shown
     */
    public boolean getShowMessageAbortButton() {
        return showMessageReply(MessageReplyType.REPLY_ABORT);
    }

    /**
     * @return true if the message retry button is to be shown
     */
    public boolean getShowMessageRetryButton() {
        return showMessageReply(MessageReplyType.REPLY_RETRY);
    }

    /**
     * @return true if the message ignore button is to be shown
     */
    public boolean getShowMessageIgnoreButton() {
        return showMessageReply(MessageReplyType.REPLY_IGNORE);
    }

    private boolean showMessageReply(MessageReplyType reply) {
        if (getShowMessage()) {
            MessageReplyType validReplies[] = getApplicationView().getMessage().getValidReplies().toReplySet();
            for (MessageReplyType r : validReplies) {
                if (r == reply) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @return the message input string
     */
    public String getMessageInput() {
        return messageInput;
    }

    /**
     * Set the message input string (the reply string).
     *
     * @param messageInput the new message input string
     */
    public void setMessageInput(String messageInput) {
        this.messageInput = messageInput;
    }

    /**
     * @return the JSF ID of the message box button to receive initial focus
     */
    public String getMessageBoxFocusButton() {
        MessageReplyType defaultReply = MessageReplyType.REPLY_OK;
        if (getShowMessage()) {
            MessageReplyType validReplies[] = getApplicationView().getMessage().getValidReplies().toReplySet();
            if (validReplies.length > 0) {
                defaultReply = validReplies[0];
            }
        }
        return getMessageBoxButtonId(defaultReply);
    }

    /**
     * Get the JSF ID of the message box button for the given reply type.
     *
     * @param reply the reply type corresponding to the button
     * @return the JSF ID of the button
     */
    protected String getMessageBoxButtonId(MessageReplyType reply) {
        final String PREFIX = "messageBox_";
        if (reply == MessageReplyType.REPLY_ABORT) {
            return PREFIX + "abort";
        }
        if (reply == MessageReplyType.REPLY_CANCEL) {
            return PREFIX + "cancel";
        }
        if (reply == MessageReplyType.REPLY_IGNORE) {
            return PREFIX + "ignore";
        }
        if (reply == MessageReplyType.REPLY_NO) {
            return PREFIX + "no";
        }
        if (reply == MessageReplyType.REPLY_RETRY) {
            return PREFIX + "retry";
        }
        if (reply == MessageReplyType.REPLY_YES) {
            return PREFIX + "yes";
        }
        return PREFIX + "ok";
    }

    /**
     * Get the title for the given dialog instance.
     *
     * @param instance the dialog instance in question
     * @return the title for the instance
     */
    public String getDialogTitle(DialogInstance instance) {
        return getApplicationView().getDialogTitle(instance);
    }

    /**
     * Get the named JSF managed bean from the JSF context.
     *
     * @param <T> the bean class
     * @param beanName the name of the managed bean
     * @return the bean
     */
    @SuppressWarnings("unchecked")
    public static <T> T getFacesBean(String beanName) {
        FacesContext context = FacesContext.getCurrentInstance();
        return (T) context.getApplication().evaluateExpressionGet(context, "#{" + beanName + "}", Object.class);
    }

    /**
     * Find the current instance of the FacesApplicationBean
     * for the JSF context of this application. The name of
     * the application bean is expected to be set from the
     * first instance of the application bean, and this method
     * throws an IllegalStateException if called before the
     * name is set.
     *
     * @return the current application bean instance
     */
    public static FacesApplicationBean getCurrentApplicationBean() {
        String beanName = FacesApplicationBean.applicationBeanName;
        if (beanName == null) {
            throw new IllegalStateException("");
        }
        FacesApplicationBean bean = FacesApplicationBean.getFacesBean(beanName);
        return bean;
    }

    /**
     * Sets an external resource provider. This resource provider
     * is checked first, before the generated <code>ResourceBundle</code>s
     * are used when looking for resource texts.
     *
     * @param resourceProvider New resource provider (or null, to erase)
     */
	public static void setExternalResourceProvider(ResourceProvider<String> resourceProvider) {
    	FacesMessageUtil.setExternalResourceProvider(resourceProvider);
    }



}
