/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;

import no.esito.jvine.action.GuiTask;
import no.esito.jvine.view.AbstractDialogView;
import no.esito.jvine.view.ViewModelImpl;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.controller.DialogObjectType;
import no.g9.client.core.util.DialogObjectConstantHelper;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.faces.tree.FacesTreeNode;
import no.g9.client.core.view.menu.Menu;
import no.g9.client.core.view.table.TableModel;
import no.g9.client.core.view.table.TableModel.SelectionModel;
import no.g9.os.RoleConstant;
import no.g9.support.ActionType;

import org.icefaces.ace.model.table.RowState;
import org.icefaces.ace.model.table.RowStateMap;
import org.icefaces.ace.model.tree.NodeState;
import org.icefaces.ace.model.tree.NodeStateMap;
import org.icefaces.util.JavaScriptRunner;

/**
 * Common superclass for class for JSF views JSF views.
 *
 */
public abstract class FacesDialogView extends AbstractDialogView {

    private List<Menu> dialogMenu = new ArrayList<Menu>();

    private Set<DialogObjectConstant> ignoreValuechangeFields =
        new HashSet<DialogObjectConstant>();

    private Map<DialogObjectConstant, Integer> currentNotebookIndex = new HashMap<DialogObjectConstant, Integer>();

    private Map<DialogObjectConstant, Integer> oldNotebookIndex = new HashMap<DialogObjectConstant, Integer>();

    private Class<Enum<?>> dialogObjectEnum;

	private final Map<RoleConstant, RowStateMap> rowStateMaps = new HashMap<RoleConstant, RowStateMap>();

	private final Map<DialogObjectConstant, NodeStateMap> nodeStateMaps = new HashMap<>();

    /**
     * Create a new view for the given dialog.
     *
     * @param dialog the dialog which owns this view.
     * @param dialogObjectEnum the enum for all dialog objects in this dialog
     */
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public FacesDialogView(DialogConstant dialog, Class dialogObjectEnum) {
        super(dialog);
        this.dialogObjectEnum = dialogObjectEnum;
    }

    @Override
    public void setProperty(DialogObjectConstant target, String key, Object value) {
    	setAttribute(getFacesId(target), key, value);
    }

    /**
     * Return the JSF component with the given id. The method searches the JSF
     * component hierarchy starting at the given UIComponent, or from the top if
     * none is given.
     *
     * @param comp where to start looking for the component, or <code>null</code>.
     * @param key the JSF id of the component
     * @return the component or <code>null</code> if not found.
     * @see #getFacesId(DialogObjectConstant)
     */
    protected UIComponent getFacesComponent(UIComponent comp, String key) {
        FacesContext context = FacesContext.getCurrentInstance();
        UIComponent currentComp = comp;
        if (currentComp == null) {
            currentComp = context.getViewRoot();
        }
        if (currentComp.getId() != null && currentComp.getId().equals(key)) {
            return currentComp;
        }
        for (Object child : currentComp.getChildren()) {
            final UIComponent foundComp = getFacesComponent((UIComponent) child, key);
            if (foundComp != null) {
                return foundComp;
            }

        }
        if (currentComp.getChildren() != null && currentComp.getChildren().size() == 0) {
            Iterator<UIComponent> it = currentComp.getFacetsAndChildren();
            while (it.hasNext()) {
                final UIComponent foundComp = getFacesComponent(it.next(), key);
                if (foundComp != null) {
                    return foundComp;
                }
            }

        }
        return null;
    }

    /**
     * Gets the string used by JSF to identify the component.
     * @param target the constant denoting the component
     * @return the JSF id of the component.
     */
    protected String getFacesId(DialogObjectConstant target) {
        String id = target.getInternalName();
        String instanceName = getApplicationView().getDialogInstanceName(getDialogInstance());
        if (instanceName != null && !id.startsWith(instanceName)) { // Dialog instances have a number suffix
        	int ix = getApplicationView().getDialogName(getDialog()).length();
        	id = instanceName + id.substring(ix);
        }
        return id;
    }

    /**
     * Get the row state map for the given role.
     *
     * @param role the role to get the row state map for
     * @return the rowStateMap
     */
    protected RowStateMap getRowStateMap(RoleConstant role) {
        RowStateMap rowStateMap = rowStateMaps.get(role);
        if (rowStateMap == null) {
            rowStateMap = new RowStateMap();
            rowStateMaps.put(role, rowStateMap);
        }
        return rowStateMap;
    }

    /**
     * Set the row state map for the given role.
     *
     * @param role  the role to set the row state map for
     * @param rowStateMap the rowStateMap to set
     */
    protected void setRowStateMap(RoleConstant role, RowStateMap rowStateMap) {
        rowStateMaps.put(role, rowStateMap);
    }

    /**
     * Update the row states for the given role.
     *
     * @param role the owning role of the table
     */
    protected synchronized void updateRowState(RoleConstant role) {
        TableModel<?> tableModel = getViewModel().getTableModel(role);
        RowStateMap stateMap = getRowStateMap(role);
        stateMap.setAllSelected(false);
        SelectionModel selectionModel = tableModel.getSelectionModel();
        if (selectionModel == SelectionModel.NO_SELECT) {
        	stateMap.setAllSelectable(false);
        }
        else {
        	for (ListRow row : tableModel.getSelected()) {
        		RowState state = stateMap.get(row);
        		if (state != null) {
        			state.setSelected(true);
        		}
        	}
        }
    }

    /**
     * Get the node state map for the given dialog constant.
     *
     * @param diaConst the dialog constant to get the node state map for
     * @return the nodeStateMap
     */
    protected NodeStateMap getNodeStateMap(DialogObjectConstant diaConst) {
        NodeStateMap nodeStateMap = nodeStateMaps.get(diaConst);
        if (nodeStateMap == null) {
            nodeStateMap = new NodeStateMap();
            nodeStateMaps.put(diaConst, nodeStateMap);
        }
        return nodeStateMap;
    }

    /**
     * Set the node state map for the given dialog constant.
     *
     * @param diaConst  the dialog constant to set the node state map for
     * @param nodeStateMap the nodeStateMap to set
     */
    protected void setNodeStateMap(DialogObjectConstant diaConst, NodeStateMap nodeStateMap) {
        nodeStateMaps.put(diaConst, nodeStateMap);
    }

    private void setAttribute(final String expr, final String key, final Object value) {
        guiInvoke(new GuiTask(key) {
            @Override
            public void run() {
                UIComponent comp = getFacesComponent(null, expr);
                if (comp != null) {
                    comp.getAttributes().put(key, value);
                }
            }
        });
    }

    /**
     * Set the "value" attribute for the given JSF component.
     *
     * @param target the dialog object represented by the JSF component.
     * @param value the new value to set.
     * @deprecated use setProperty with "value" as key instead.
     */
    @Deprecated
    public void setValue(DialogObjectConstant target, String value) {
    	setAttribute(getFacesId(target), "value", value);
    }

    @Override
    protected void focus(final DialogObjectConstant target) {
        guiInvoke(new GuiTask("focus") {
            @Override
            public void run() {
            	String facesId = getFacesId(target);
            	if (DialogObjectConstantHelper.isAutoCompleteEntry(target) || DialogObjectConstantHelper.isDateTimeEntry(target)) {
            	    facesId += "_input";
            	}
                JavaScriptRunner.runScript(FacesContext.getCurrentInstance(), "ice.applyFocus('iceform:" + facesId + "');");
            }
        });
    }

    /**
     * @return the set of fields where value changed events is to be ignored.
     */
    protected Set<DialogObjectConstant> getIgnoreValuechangeFields() {
        return ignoreValuechangeFields;
    }

    /**
     * Add the given menu to the dialog menu.
     *
     * @param menu - the menu to add
     */
    protected void addMenu(Menu menu) {
        dialogMenu.add(menu);
    }

    /**
     * Sets the dialog instance number on the menus.
     *
     * @param menus the list of menus
     * @return the list of modified menus
     */
    protected List<Menu> setDialogInstanceNumberOnMenus(List<Menu> menus) {
        for (Menu menu : menus) {
            menu.setDialogInstance(getDialogInstanceNumber());
        }
        return menus;
    }

    /**
     * Add the given menus to the dialog menu.
     *
     * @param menus - the menus to add
     */
    public void addMenus(List<Menu> menus) {
        for (Menu menu : menus) {
            addMenu(menu);
        }
    }

    /**
     * @return the dialog menu.
     */
    List<Menu> getDialogMenu() {
        return dialogMenu;
    }

    /**
     * Overridden just to add local package visibility.
     */
    @Override
    protected void setOpen(boolean isOpen) {
        super.setOpen(isOpen);
    }

    /**
     * Overridden just to add local package visibility.
     */
    @Override
    protected boolean handleListSelection(RoleConstant listRole,
            boolean selected, int rowNo, ListRow emptyRow) {
        return super.handleListSelection(listRole, selected, rowNo, emptyRow);
    }
    
    /**
     * Overridden just to add local package visibility.
     */
    @Override
    protected void setListSelectionRowNum(int rowNum) {
        super.setListSelectionRowNum(rowNum);
    }

    /**
     * Dispatch the event without a listRow.
     *
     * @param trigger the cause of the event.
     * @param event the type of event.
     * @param method the event method name.
     */
    protected void dispatchEvent(DialogObjectConstant trigger, Object event, String method) {
        dispatchEvent(null, trigger, event, method);
    }

    /**
     * Overridden just to add local package visibility.
     */
    @Override
    public void dispatchEvent(ListRow listRow, DialogObjectConstant trigger, Object event, String method) {
        super.dispatchEvent(listRow, trigger, event, method);
    }

    /**
     * Override the default implementation, returning a FacesApplicationView.
     *
     * @return the JSF application view.
     */
    @Override
    public FacesApplicationView getApplicationView() {
        return (FacesApplicationView) super.getApplicationView();
    }

    @Override
    public void performAction(ActionType action, DialogObjectConstant target) {
        super.performAction(action, target);

        switch(action) {

        case HIDE: /* fall through */
        case SHOW:
            // Make sure the effect sets the "visible" property for hide/show
            ViewModelImpl model = (ViewModelImpl) getViewModel();
            Object effectBean = model.getPropertyManager().getEffect(target);
            if (effectBean != null) {
                // TODO: fix this the JSF2 way
                // effectBean.setSubmit(true);
                // effectBean.setTransitory(false);
            }
            break;

        case EXPAND:
            expandNodeTypes(target, true);
            break;

        case COLLAPSE:
            expandNodeTypes(target, false);
            break;
            
        default: // no default action
        	break;
        }
    }

    /**
     * Expand or collapse all nodes in a tree view or all nodes of target type.
     *
     * @param target The dialog object constant for the tree node or tree view
     * @param expand Whether to expand or collapse the node
     */
    private void expandNodeTypes(DialogObjectConstant target, boolean expand) {
        DialogObjectType objectType = target.getType();

        if(objectType.equals(DialogObjectType.TreeNode)) {
            DialogObjectConstant parent = (DialogObjectConstant) target.getParent();

            while(!parent.getType().equals(DialogObjectType.TreeView)) {
                parent = (DialogObjectConstant) parent.getParent();
            }

            NodeStateMap stateMap = nodeStateMaps.get(parent);

            for (Map.Entry<Object, NodeState> entry : stateMap.entrySet()) {
                if(((FacesTreeNode)entry.getKey()).getNodeConst().equals(target)) {
                    entry.getValue().setExpanded(expand);
                }
            }

        } else if (objectType.equals(DialogObjectType.TreeView)) {
            nodeStateMaps.get(target).setAllExpanded(expand);
        }
    }

    /**
     * Creates a bean reference that is a concatenation of the
     * <code>beanName</code> and the <code>dialogIntanceNumber</code>.
     *
     * @param beanName
     *            the bean name
     * @return the string the correct bean ref for this instance.
     */
    public String createBeanRef(String beanName) {
        String retVal = beanName;
        if (getDialogInstanceNumber() > 1) {
            retVal = retVal + getDialogInstanceNumber();
        }
        return retVal;
    }

    /**
     * Creates a bean reference for this dialog.
     * It is a concatenation of the default bean name for this dialog,
     * and the <code>dialogIntanceNumber</code>.
     *
     * @return the string the correct bean ref for this instance.
     */
    public String createBeanRef() {
        return createBeanRef(getDialog().getG9Name().toLowerCase());
    }

    /**
     * Validate the new value for the given field.
     *
     * @param listRow the ListRow object, if validating in a list
     * @param field the field to validate
     * @param newValue the new value for the field
     * @return true if the validation was ok
     */
    boolean validateField(ListRow listRow, DialogObjectConstant field,
            Object newValue) {
        return getApplicationView().getDialogController(getDialog()).validateField(listRow, field, newValue);
    }

    /**
     * Set the index of the current Notebook page to show. To be used for
     * Notebooks with tabs and accordions.
     *
     * @param notebook the Notebook.
     * @param index the new current page index for the Notebook.
     */
    protected void setCurrentNotebookIndex(DialogObjectConstant notebook, int index) {
        int oldIndex = getCurrentNotebookIndex(notebook);
        if (index != oldIndex) {
            setOldNotebookIndex(notebook, oldIndex);
            currentNotebookIndex.put(notebook, Integer.valueOf(index));
        }
    }

    /**
     * Get the current page index for the given Notebook.
     *
     * @param notebook the Notebook in question.
     * @return the current page index for the Notebook.
     */
    protected int getCurrentNotebookIndex(DialogObjectConstant notebook) {
        Integer index = currentNotebookIndex.get(notebook);
        if (index == null) {
            return 0;
        }
        return index.intValue();
    }

    /**
     * Set the index of the old (previous) Notebook page shown.
     * This is currently only used for accordions, to be able to support the Hidden event.
     *
     * @param notebook the Notebook.
     * @param index the new old page index for the Notebook.
     */
    protected void setOldNotebookIndex(DialogObjectConstant notebook, int index) {
        oldNotebookIndex.put(notebook, Integer.valueOf(index));
    }

    /**
     * Get the old page index for the given Notebook.
     * This is currently only used for accordions, to be able to support the Hidden event.
     *
     * @param notebook the Notebook in question.
     * @return the old page index for the Notebook.
     */
    protected int getOldNotebookIndex(DialogObjectConstant notebook) {
        Integer index = oldNotebookIndex.get(notebook);
        if (index == null) {
            return 0;
        }
        return index.intValue();
    }

    /**
     * Get the DialogObjectConstant for the dialog object with the given name (from the dialog model).
     *
     * @param name the model name of the dialog object
     * @return the dialog object constant, or null if not found
     */
    public DialogObjectConstant getDialogObjectFromName(String name) {
        Enum<?>[] enumConstants = dialogObjectEnum.getEnumConstants();
        for (Enum<?> enum1 : enumConstants) {
            DialogObjectConstant tmp = (DialogObjectConstant) enum1;
            if (name.equals(tmp.getG9Name())) {
                return tmp;
            }
        }
        return null;
    }

    /**
     * Get the DialogObjectConstant for the dialog object with the given internal name (JSF ID).
     *
     * @param name the internal name of the dialog object
     * @return the dialog object constant, or null if not found
     */
    public DialogObjectConstant getDialogObjectFromInternalName(String name) {
        Enum<?>[] enumConstants = dialogObjectEnum.getEnumConstants();
        for (Enum<?> enum1 : enumConstants) {
            DialogObjectConstant tmp = (DialogObjectConstant) enum1;
            if (name.equals(tmp.getInternalName())) {
                return tmp;
            }
        }
        return null;
    }

}
