/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.esito.jvine.view.faces;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.el.MethodExpression;
import javax.faces.context.FacesContext;
import javax.faces.event.ActionEvent;
import javax.faces.event.ActionListener;
import javax.faces.event.MethodExpressionActionListener;
import javax.servlet.http.Cookie;

import no.esito.jvine.controller.DialogInstanceKey;
import no.esito.log.Logger;
import no.g9.client.core.communication.SystemMessage;
import no.g9.client.core.controller.DialogConstant;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.faces.FacesApplicationView;
import no.g9.client.core.view.faces.FacesDialogView;
import no.g9.client.core.view.faces.FacesMenuItem;
import no.g9.client.core.view.faces.G9ResourceServlet;
import no.g9.client.core.view.menu.Menu;
import no.g9.client.core.view.menu.Menu.MENU_TYPE;
import no.g9.client.core.view.menu.MenuBase;
import no.g9.client.core.view.menu.MenuItem;
import no.g9.exception.G9BaseException;

import org.icefaces.ace.component.ajax.AjaxBehavior;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.util.StringUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * A helper class for various menu related operations.
 */
public class MenuHelper {

    /** The title part of the bundle key */
    public final static String TITLE_KEY = "title";

    /** The mnemonic part of the bundle key */
    public final static String MNEMONIC_KEY = "mnemonic";

    /** The accelerator part of the bundle key */
    public final static String ACCELERATOR_KEY = "accelerator";

    /** The dialog prefix of the bundle key */
    public final static String DIALOG_PREFIX = "dialog";

    private static Logger logger = Logger.getLogger(MenuHelper.class);

    /**
     * Get the style class for the given menu item or menu.
     *
     * @param menuBase the menu item or menu
     * @return the style class
     */
    public static String getStyleClass(MenuBase menuBase) {
        return menuBase.getStyle();
    }

    /**
     * Get the icon style class for the given menu item or menu.
     *
     * @param menuBase the menu item or menu
     * @param view the dialog view
     * @return the icon style class, or null if no icon class can be created
     */
    public static String getMenuIconClass(MenuBase menuBase, FacesDialogView view) {
    	if (menuBase == null || (menuBase.getImage() == null && menuBase.getDisabledImage() == null)) {
    		return null;
    	}
    	String imgExpr = menuBase.getImage().getName();
    	if (!view.isEnabled(view.getDialogObjectFromName(menuBase.getId())) && menuBase.getDisabledImage() != null) {
    		imgExpr = menuBase.getDisabledImage().getName();
    	}
    	return "gvaMenuImage " + imgExpr;
    }

    /**
     * Add action listeners for all events for the given menu item.
     *
     * @param facesItem the faces menu item to add action listeners to
     * @param item the menu item
     * @param view the dialog which owns the menu item or menu
     */
    public static void addActionListeners(FacesMenuItem facesItem, MenuItem item, FacesDialogView view) {
        for (String event : item.getMethods().keySet()) {
            facesItem.addActionListener(MenuHelper.createActionListener(MenuHelper.getActionExpression(event, item, view)));
        }
    }

    /**
     * Add an "action" ajax behavior to the given menu item.
     *
     * @param menuItem the menu item to add an ajax behavior to
     */
    public static void addAjaxBehaviorTo(FacesMenuItem menuItem) { 
    	AjaxBehavior ajaxBehavior = new AjaxBehavior(); 
    	ajaxBehavior.setExecute("@form"); 
    	ajaxBehavior.setRender("@form"); 
    	menuItem.addClientBehavior("action", ajaxBehavior); 
    }

    /**
     * Get the message used to trigger the event method from an external application.
     *
     * @param item the menu item
     * @return the message
     */
    public static SystemMessage getExternalMessage(MenuItem item) {
        SystemMessage message = null;
        for (String event : item.getMethods().keySet()) {
            for (String method : item.getMethods().get(event)) {
                message = new SystemMessage(item.getApplicationId(), SystemMessage.MENU_PORT, method);
            }
        }
        return message;
    }

    /**
     * Create a new action listener for the given menu item.
     *
     * @param actionExpression the el expression for the action listener
     *
     * @return a JSF action listener
     */
    public static ActionListener createActionListener(String actionExpression) {
        FacesContext context = FacesContext.getCurrentInstance();
        MethodExpression methodExpression = context.getApplication().getExpressionFactory()
                .createMethodExpression(context.getELContext(), actionExpression, null, new Class[] { ActionEvent.class });
        return new MethodExpressionActionListener(methodExpression);
    }

    /**
     * Get the insert point for new menus for the given list of menus.
     *
     * @param menus - the list of menus to check.
     *
     * @return the insert point, or null if insert point is at the end.
     */
    public static Menu.MENU_TYPE getInsertPoint(List<Menu> menus) {
        for (Menu menu : menus) {
            switch (menu.getType()) {
            case WINDOWMENU:
                return MENU_TYPE.WINDOWMENU;
            case HELPMENU:
                return MENU_TYPE.HELPMENU;
            default:
                // Empty
            }
        }
        return null;
    }

    private static String getActionExpression(String event, MenuItem item, FacesDialogView view) {
        String methodRef = StringUtils.uncapitalize(item.getId()) + "_" + event;
        return "#{" + view.createBeanRef() + "." + methodRef + "}";
    }

    /**
     * Get the title for the given dialog item.
     *
     * @param item the dialog item
     * @param dialog the dialog which contains the dialog item
     * @param bundleName the name of the message bundle for the dialog
     * @return the title
     */
    public static String getTitle(DialogObjectConstant item, DialogConstant dialog, String bundleName) {
        String key = MenuHelper.getBundleKey(item, dialog, MenuHelper.TITLE_KEY);
        return FacesMessageUtil.getBundleMessage(bundleName, key);
    }

    /**
     * Get the mnemonic for the given dialog item.
     *
     * @param item the dialog item
     * @param dialog the dialog which contains the dialog item
     * @param bundleName the name of the message bundle for the dialog
     * @return the mnemonic
     */
    public static String getMnemonic(DialogObjectConstant item, DialogConstant dialog, String bundleName) {
        String key = MenuHelper.getBundleKey(item, dialog, MenuHelper.MNEMONIC_KEY);
        return FacesMessageUtil.getBundleMessage(bundleName, key);
    }

    /**
     * Get the accelerator for the given dialog item.
     *
     * @param item the dialog item
     * @param dialog the dialog which contains the dialog item
     * @param bundleName the name of the message bundle for the dialog
     * @return the accelerator
     */
    public static String getAccelerator(DialogObjectConstant item, DialogConstant dialog, String bundleName) {
        String key = MenuHelper.getBundleKey(item, dialog, MenuHelper.ACCELERATOR_KEY);
        return FacesMessageUtil.getBundleMessage(bundleName, key);
    }

    /**
     * Get the message bundle key for the given dialog item.
     *
     * @param item the dialog item
     * @param dialog the dialog which contains the dialog item
     * @param suffix the key suffix, for example "title"
     * @return the bundle key
     */
    public static String getBundleKey(DialogObjectConstant item, DialogConstant dialog, String suffix) {
        return MenuHelper.DIALOG_PREFIX + "." + dialog.getG9Name() + "." + item.getG9Name() + "." + suffix;
    }

    /**
     * Get the dialog view for the given menu or menu item.
     * The dialog view is found by a Spring lookup. If the Spring
     * bean isn't found, null is returned.
     *
     * @param item the menu or menu item
     * @param appView the application view
     * @return the dialog view, or null if the view was not found
     */
    public static FacesDialogView getDialogView(MenuBase item, FacesApplicationView appView) {
        try {
            DialogConstant dialog = appView.getApplicationController().getDialogConst(item.getDialogId());
            DialogInstance instance = new DialogInstanceKey(dialog, MenuHelper.getDialogInstance(item));
            return appView.getDialogView(instance);
        }
        catch (NoSuchBeanDefinitionException e) {
            return null;
        }
    }

    /**
     * Get a list of menus for the given application URL. The menus returned will normally be
     * the menus defined for the application window of the given application.
     *
     * @param applicationInfo contains the URL for the application to get the menus from
     * @return the list of menus
     */
    public static List<Menu> getApplicationMenus(ApplicationInfo applicationInfo) {
        try {
            String applicationUrl = applicationInfo.url;
            InputStream response = new URL(MenuHelper.getMenuResourceUrl(applicationUrl)).openStream();
            if ( response == null) {
                throw new G9BaseException("No response from " + applicationUrl);
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Response from " + applicationUrl + ", available bytes: " + response.available());
            }
            List<Menu> menus = MenuHelper.parseMenuResponse(response);
            response.close();
            MenuHelper.tagApplicationId(menus, applicationInfo.name);
            return menus;
        }
        catch (IOException e) {
            throw new G9BaseException("IOException for application " + applicationInfo, e);
        }
    }

    /**
     * Gets the url of the system setup.
     * @return The setup url
     */
    public static String getSetupUrl(){
        Cookie setupCookie=(Cookie) FacesContext.getCurrentInstance().getExternalContext().getRequestCookieMap().get("setupUrl");
        if(setupCookie!=null){
            return setupCookie.getValue();
        }
        throw new G9BaseException("G9 exeption. Missing system setup");
    }

    /**
     * Create an application menu for the given application name.
     * The menu will be of type USERMENU.
     *
     * @param applicationName the name of the application, used for menu ID
     * @param applicationTitle the title for the application
     * @param dialog the dialog which will own the new menu
     * @return the new menu
     */
    public static Menu createApplicationMenu(String applicationName, String applicationTitle, DialogConstant dialog) {
        String applicationId = applicationName.replace(' ', '_');
        Menu appMenu = new Menu(applicationId, dialog.getInternalName(), applicationTitle, null, null, null, null, null, MENU_TYPE.USERMENU);
        return appMenu;
    }

    /**
     * Get a map containing info for all application partitions.
     * The list of applications is read from a setup JSON at the given URL.
     * The map key is the application name.
     *
     * @param setupUrl
     * @return a map with all application partitions
     */
    public static Map<String, ApplicationInfo> getApplications(String setupUrl) {
        Map<String, ApplicationInfo> apps = new LinkedHashMap<String, ApplicationInfo>();
        List<String> applicationUrls = getApplicationUrls(setupUrl);
        for (String url : applicationUrls) {
            ApplicationInfo info = null;
            try {
                info = getApplicationInfo(url);
            }
            catch (G9BaseException e) {
                logger.info(e.getMessage(), e);
                info = new ApplicationInfo(url);
            }
            apps.put(info.name, info);
        }
        return apps;
    }

    /**
     * Get the name and title of the application at the given URL.
     * The URL is suffixed with "resources/application.json", and the
     * JSON is parsed to get the application name/title.
     *
     * @param applicationUrl the URL for the application
     * @return the application name and title
     * @throws G9BaseException if the application name can not be found from the URL
     */
    private static ApplicationInfo getApplicationInfo(String applicationUrl) {
        try {
            if (!applicationUrl.endsWith("/")) {
                applicationUrl = applicationUrl + "/";
            }
            String url = applicationUrl + "resources/application.json";
            InputStream response = new URL(url).openStream();
            if ( response == null) {
                throw new G9BaseException("No response from " + url);
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Response from " + url + ", available bytes: " + response.available());
            }
            ApplicationInfo info = parseApplicationResponse(response, applicationUrl);
            response.close();
            return info;
        }
        catch (IOException e) {
            throw new G9BaseException("IOException for setup " + applicationUrl, e);
        }
    }

    /**
     * Get the list of all application partition URLs from the given setup URL.
     * The setup URL should point to a JSON which lists all applications.
     *
     * @param setupUrl the URL to the setup JSON
     * @return a list of all application partition URLs
     */
    private static List<String> getApplicationUrls(String setupUrl) {
        try {
            InputStream response = new URL(setupUrl).openStream();
            if ( response == null) {
                throw new G9BaseException("No response from " + setupUrl);
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Response from " + setupUrl + ", available bytes: " + response.available());
            }
            List<String> applications = parseSetupResponse(response);
            response.close();
            return applications;
        }
        catch (IOException e) {
            throw new G9BaseException("IOException for setup " + setupUrl, e);
        }
    }

    /**
     * Read the application menu from the given response stream.
     *
     * @param response the response stream with the menu object
     * @return the list of menus
     */
    @SuppressWarnings("unchecked")
    private static List<Menu> parseMenuResponse(InputStream response) {
        try {
            ObjectInputStream ois = new ObjectInputStream(response);
            Object o = ois.readObject();
            return (List<Menu>) o;
        } catch (IOException e) {
            throw new G9BaseException("IOException when reading application menu objects", e);
        } catch (ClassNotFoundException e) {
            throw new G9BaseException("Exception when parsing application menu objects", e);
        }
    }

    /**
     * Read the list of application partition URLs from the given JSON response stream.
     * The application URLs are contained in an array named "parts", with the key "url" for
     * each URL.
     *
     * If the application is marked as "external" in the JSON stream, it is not added
     * to the returned list, as this type of application will not have an
     * application.json file available.
     *
     * @param response the response stream with the JSON array of application partitions
     * @return a list of the application partitions
     */
    private static List<String> parseSetupResponse(InputStream response) {
        try {
            List<String> urls = new ArrayList<String>();
            ObjectMapper mapper = new ObjectMapper();
            JsonNode tree = mapper.readTree(response);
            Iterator<JsonNode> partsIterator = tree.findValue("parts").elements();
            while (partsIterator.hasNext()) {
                JsonNode part = partsIterator.next();
                if (part.findValue("external") == null) {
                    urls.add(part.findValue("url").asText());
                }
            }
            return urls;
        } catch (IOException e) {
            throw new G9BaseException("IOException when reading application menu objects", e);
        }
    }

    /**
     * Get the application name and title from the given JSON response stream. The key "name" is used
     * to get the application name from the JSON object, and for the title the key "title" is used.
     *
     * @param response the response stream with the JSON object containing the application name
     * @param applicationUrl the URL of the application, used when creating the info object
     * @return the name and title of the application
     */
    private static ApplicationInfo parseApplicationResponse(InputStream response, String applicationUrl) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            JsonNode tree = mapper.readTree(response);
            ApplicationInfo info = new ApplicationInfo(tree.findValue("name").asText(), tree.findValue("title").asText(),
                    applicationUrl);
            return info;
        } catch (IOException e) {
            throw new G9BaseException("IOException when reading application menu objects", e);
        }
    }


    /**
     * Append the parameters etc. to the given application URL, to be able to
     * retrieve the application menus.
     *
     * @param applicationUrl the URL of the application to fetch menus from
     * @return the full URL to be used
     */
    private static String getMenuResourceUrl(String applicationUrl) {
        if (!applicationUrl.endsWith("/")) {
            applicationUrl = applicationUrl + "/";
        }
        return applicationUrl + G9ResourceServlet.URL_SUFFIX + "?" + G9ResourceServlet.TYPE_PARAM + "="
                + G9ResourceServlet.APPLICATION_MENU_TYPE;
    }

    /**
     * Get the dialog instance number for the given menu or menu item.
     * The instance number is found in the topmost menu.
     *
     * @param item the menu or menu item to get the instance number from
     * @return the instance number
     */
    private static int getDialogInstance(MenuBase item) {
        Menu parent;
        if (item instanceof Menu) {
            parent = (Menu) item;
        } else {
            parent = item.getParent();
        }
        while (parent.getParent() != null) {
            parent = parent.getParent();
        }
        return parent.getDialogInstance();
    }

    /**
     * Set the given application ID on all menu items.
     *
     * @param menus the list of menus to tag
     * @param applicationId the application ID
     */
    private static void tagApplicationId(List<Menu> menus, String applicationId) {
        for (Menu menu : menus) {
            tagApplicationId(menu, applicationId);
        }
    }

    /**
     * Set the given application ID on all menu items.
     *
     * @param menu the menu to tag
     * @param applicationId the application ID
     */
    private static void tagApplicationId(Menu menu, String applicationId) {
        for (MenuBase item : menu.getChildren()) {
            if (item instanceof MenuItem) {
                item.setApplicationId(applicationId);
            }
            else if (item instanceof Menu) {
                MenuHelper.tagApplicationId((Menu)item, applicationId);
            }
        }
    }

}
