/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import javax.faces.application.FacesMessage;
import javax.faces.component.UIComponent;

import no.esito.jvine.view.ViewModelImpl;
import no.esito.jvine.view.faces.FacesMessageUtil;
import no.g9.client.core.controller.DialogInstance;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;

/**
 *
 */
public class ConverterHelper implements G9Converter {

    private String dataType;
    private String displayRule;
    private String title;

    @Override
    public void setDataType(String dataType) {
        this.dataType = dataType;
    }

    @Override
    public String getDataType() {
        return dataType;
    }

    @Override
    public void setDisplayRule(String displayRule) {
        this.displayRule = displayRule;
    }

    @Override
    public String getDisplayRule() {
        return displayRule;
    }

    @Override
    public void setTitle(String title) {
        this.title = title;
    }

    @Override
    public String getTitle() {
        return title;
    }

    /**
     * @param component the JSF component
     */
    void addConversionErrorMessage(UIComponent component) {
        FacesMessageUtil.addContextMessage(component, FacesMessage.SEVERITY_ERROR, FacesMessageUtil.CONVERSION_ERROR, " "
                + getTitle());
    }

    /**
     * @param component the JSF component
     */
    void addIllegalParameterMessage(UIComponent component) {
        FacesMessageUtil.addContextMessage(component, FacesMessage.SEVERITY_FATAL, FacesMessageUtil.ILLEGAL_PARAMETER, " "
                + getTitle());
    }

    /**
     * @return a <code>FacesMessage</code> for a conversion error
     */
    FacesMessage getConversionErrorMessage() {
        return ConverterHelper.getConversionErrorMessage(getTitle());
    }

    /**
     * @param title the component title
     * @return a <code>FacesMessage</code> for a conversion error
     */
    public static FacesMessage getConversionErrorMessage(String title) {
        Message message = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.CF_FRAMEWORK_FIELD_CONVERSION_FAILED, title);
        return new FacesMessage(FacesMessage.SEVERITY_ERROR, message.getTitle(), message.getMessageText());        
    }

    /**
     * @param component the JSF component
     * @return a <code>FacesMessage</code> for a conversion error
     */
    public static FacesMessage getConversionErrorMessage(UIComponent component) {
        String title = ConverterHelper.getTitle(component);
        return ConverterHelper.getConversionErrorMessage(title);
    }
    
    /**
     * Get the value of the given component from the view model of the current dialog.
     *
     * @param component the JSF component which identifies the view model field
     * @return the view model value of the field found for the JSF component
     */
    Object getComponentValue(UIComponent component) {
        ViewModelImpl viewModel = ConverterHelper.getCurrentViewModel();
        DialogObjectConstant field = ConverterHelper.getDialogObjectConstant(component);
        return viewModel.getField(field);
    }

    /**
     * Get the dialog object constant for the given JSF component.
     *
     * @param component the JSF component
     * @return the dialog object constant
     */
    public static DialogObjectConstant getDialogObjectConstant(UIComponent component) {
        if (component == null) {
            throw new IllegalArgumentException("Component may not be null");
        }
        ViewModelImpl viewModel = ConverterHelper.getCurrentViewModel();
        return viewModel.getFieldDialogObjectConstant(ConverterHelper.getInternalName(component));
    }

    private static ViewModelImpl getCurrentViewModel() {
        FacesApplicationView appView = ConverterHelper.getApplicationView();
        DialogInstance currentDialog = appView.getCurrentDialog(false);
        ViewModelImpl viewModel = (ViewModelImpl) appView.getViewModel(currentDialog);
        return viewModel;
    }

    private static FacesDialogView getCurrentDialogView() {
        FacesApplicationView appView = ConverterHelper.getApplicationView();
        DialogInstance currentDialog = appView.getCurrentDialog(false);
        return appView.getDialogView(currentDialog);
    }

    private static FacesApplicationView getApplicationView() {
        return FacesApplicationBean.getCurrentApplicationBean().getApplicationView();
    }

    private static String getTitle(UIComponent component) {
        FacesDialogView dialogView = ConverterHelper.getCurrentDialogView();
        DialogObjectConstant field = ConverterHelper.getDialogObjectConstant(component);
        if (field != null && dialogView != null) {
        	return dialogView.getTitle(field);
        }
        return ConverterHelper.getInternalName(component);
    }

    /**
     * Get the internal name (for the DialogObjectConstant) of the view model field
     * for the given JSF component.
     *
     * @param component the JSF component which identifies the view model field
     * @return the internal name
     */
    private static String getInternalName(UIComponent component) {
        return component.getId();
    }

}
