/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.table.DefaultTableModel;
import no.g9.client.core.view.table.ListRowComparator;
import no.g9.client.core.view.table.ListRowComparator.Sorting;
import no.g9.client.core.view.table.TableModel;
import no.g9.os.RoleConstant;

/**
 * A Map implementation used to allow access to Table comparator properties
 * from JSF markup. Expressions of the form "${bean.sortPriority[ID]}" is
 * routed through this Map.
 * 
 * NB! The Map implementation is not complete. It is only intended as a
 * property interface from JSF EL expression, and the only supported methods
 * are <code>get()</code> and <code>put()</code>.
 */
public final class FacesComparatorMap implements Map<Object, Object> {

    private static final Logger log = Logger.getLogger(FacesComparatorMap.class);

    private FacesDialogBean jsfBean;

    private ComparatorPropertyType comparatorPropertyType;

    private Map<String, DialogObjectConstant> enumMap = new HashMap<>();

    /**
     * Create a new Map for the given comparator property type.
     * 
     * @param enumType - the DialogObjectConstant type for this Map
     * @param jsfBean - the JSF bean to get the table model from
     * @param comparatorPropertyType - the comparator property type for this Map
     */
    public FacesComparatorMap(Class<Enum<?>> enumType, FacesDialogBean jsfBean, ComparatorPropertyType comparatorPropertyType) {
        this.jsfBean = jsfBean;
        this.comparatorPropertyType = comparatorPropertyType;

        Enum<?>[] enumConstants = enumType.getEnumConstants();
        for (Enum<?> enum1 : enumConstants) {
            if (enum1 instanceof DialogObjectConstant) {
                DialogObjectConstant tmp = (DialogObjectConstant) enum1;
                enumMap.put(tmp.getInternalName(), tmp);
            }
        }
    }

    @Override
    public Object get(Object key) {
        Object value;
        String internalName = (String) key;
        DialogObjectConstant column = enumMap.get(internalName);
        ListRowComparator<?> comparator = getComparator(column);
        if (comparatorPropertyType == ComparatorPropertyType.PRIORITY) {
            if (comparator.getSorting() == Sorting.NO_SORT) {
                value = 0;
            }
            else {
                TableModel<?> tableModel = jsfBean.getViewModel().getTableModel(getTableRole(column));
                List<?> comparators = ((DefaultTableModel<?>) tableModel).getListRowComparator();
                int index = comparators.indexOf(comparator);
                value = index+1;
            }
        }
        else {
            value = comparator.getSorting() == Sorting.ASCENDING;
        }
        log.trace("Get comparator property " + comparatorPropertyType + " for " + key + ": " + value);
        return value;
    }

    @Override
    @SuppressWarnings({ "unchecked", "rawtypes" })
    public Object put(Object key, Object value) {
        Object oldVal = get(key);
        String internalName = (String)key;
        DialogObjectConstant column = enumMap.get(internalName);
        ListRowComparator<?> comparator = getComparator(column);
        if (comparatorPropertyType == ComparatorPropertyType.PRIORITY) {
            TableModel<?> tableModel = jsfBean.getViewModel().getTableModel(getTableRole(column));
            List comparators = ((DefaultTableModel<?>) tableModel).getListRowComparator();
            comparators.remove(comparator);
            int priority = getPriority(value);
            if (priority == 0) {
                comparator.setSorting(Sorting.NO_SORT);
                comparators.add(comparator);
            }
            else {
                comparators.add(priority-1, comparator);
            }
            tableModel.setListRowComparator(comparators);
        }
        else {
            boolean ascending = getAscending(value);
            comparator.setSorting(ascending ? Sorting.ASCENDING : Sorting.DESCENDING);
        }
        log.trace("Set comparator property " + comparatorPropertyType + " for " + key + " to: " + value);
        return oldVal;
    }

    private int getPriority(Object value) {
        if (value == null) {
            return 0;
        }
        return (int) value;
    }

    private boolean getAscending(Object value) {
        if (value == null) {
            return false;
        }
        return (boolean) value;
    }

    private ListRowComparator<?> getComparator(DialogObjectConstant column) {
        if (getTableRole(column) != null) {
            TableModel<?> tableModel = jsfBean.getViewModel().getTableModel(getTableRole(column));
            for (ListRowComparator<?> comparator : ((DefaultTableModel<?>) tableModel).getListRowComparator()) {
                if (comparator.getColumn().equals(column)) {
                    return comparator;
                }
            }
        }
        return null;
    }

    private static RoleConstant getTableRole(DialogObjectConstant column) {
        DialogObjectConstant table = (DialogObjectConstant) column.getParent();
        if (table != null) {
            return table.getRole();
        }
        return null;
    }

    @Override
    public int size() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean isEmpty() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsKey(Object key) {
        throw new UnsupportedOperationException();
    }


    @Override
    public String remove(Object key) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void clear() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Set<Object> keySet() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Collection<Object> values() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Set<java.util.Map.Entry<Object, Object>> entrySet() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsValue(Object value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void putAll(Map<? extends Object, ? extends Object> m) {
        throw new UnsupportedOperationException();
    }

    public enum ComparatorPropertyType {
        PRIORITY,
        ASCENDING
    }

}
