/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import no.esito.jvine.action.ThreadManager;
import no.esito.jvine.view.AbstractApplicationView;
import no.esito.log.Logger;
import no.esito.util.BeanID;
import no.g9.client.core.message.BlockingMessageCallback;
import no.g9.client.core.message.InteractionThreadPolicy;
import no.g9.client.core.message.JVineDispatcherContext;
import no.g9.client.core.message.MessageCallback;
import no.g9.message.DispatcherContext;
import no.g9.message.Message;
import no.g9.message.MessageInteractor;
import no.g9.message.MessageReply;
import no.g9.message.MessageTypeEnum;

/**
 * Default message interactor for the JSF target.
 */
@BeanID("messageInteractor")
public class FacesInteractor implements MessageInteractor {


	private final static Logger log = Logger.getLogger(FacesInteractor.class);

	private InteractionThreadPolicy getPolicy(DispatcherContext dispatcherContext) {
	    if (dispatcherContext instanceof JVineDispatcherContext) {
            JVineDispatcherContext jContext = (JVineDispatcherContext) dispatcherContext;
            return jContext.getInteractionThreadPolicy();
        }

	    return InteractionThreadPolicy.DEFAULT_POLICY;
	}

	@Override
	public MessageReply interact(Message message, DispatcherContext dispatcherContext) {

	    if (message.getMsgType() == MessageTypeEnum.INLINE) return null;


	    if (!policyCheck(getPolicy(dispatcherContext))) {
            throwThreadPolicyException(dispatcherContext);
        }

		log.info("Interacting message: [" + message + "], context: [" + dispatcherContext + "]");

		if (!(dispatcherContext instanceof JVineDispatcherContext)) {
			String msg = "Expected JVineMessageContext but got "
			+ (dispatcherContext != null ?
					dispatcherContext.getClass() : dispatcherContext);
			throw new IllegalArgumentException(msg);
		}

		JVineDispatcherContext jMsgContext = (JVineDispatcherContext) dispatcherContext;
		AbstractApplicationView appView = (AbstractApplicationView) jMsgContext.getApplicationView();
		appView.setMessage(message);
		MessageCallback msgCallback = jMsgContext.getMessageCallback();
        appView.setMessageCallback(msgCallback);
        Message answer = null;
        if (msgCallback instanceof BlockingMessageCallback) {

            appView.sendPushMessage();
            answer = ((BlockingMessageCallback) msgCallback).getMessage();
            appView.sendPopMessage();
        }

		log.debug("Got answer " + answer);

		return answer != null ? answer.getReply() : null;
	}

    private void throwThreadPolicyException(DispatcherContext dispatcherContext) {

        InteractionThreadPolicy policy = getPolicy(dispatcherContext);
        String exMsg = "Interaction on wrong thread."
                + " Current Thread is : "
                + Thread.currentThread().getName() + ". Thread policy is: "
                + policy + ", "
                + policy.getPolicyDescription();
        throw new IllegalStateException(exMsg);
    }

	private boolean policyCheck(InteractionThreadPolicy policy) {
	    boolean currentIsWorker = ThreadManager.isWorkerThread();
	    switch (policy) {
	    case ASYNC_GUI_POLICY:
	        return !currentIsWorker;
	    case ASYNC_WORKER_POLICY:
	        return currentIsWorker;
	    case DEFAULT_POLICY:
	        return true;
	    }

	    // This statement is not reached, but in order for the
	    // compiler to give a warning if all enum-cases are not
	    // covered in the switch above, we cannot use a default switch.
	    return true;
	}

}
