/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import no.esito.jvine.view.ViewModelImpl;
import no.esito.log.Logger;
import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.Property;
import no.g9.client.core.view.PropertyManager;

/**
 * A Map implementation used to allow access to PropertyManager properties
 * from JSF markup. Expressions of the form "${bean.propertyType[ID]}" is
 * routed through this Map.
 * 
 * NB! The Map implementation is not complete. It is only intended as a
 * property interface from JSF EL expression, and the only supported methods
 * are <code>get()</code> and <code>put()</code>.
 * 
 * @param <T> the type of the values in the Map 
 */
@SuppressWarnings({"unchecked", "rawtypes"})
public final class FacesPropertyMap<T> implements Map<Object, T> {

    private static final Logger log = Logger.getLogger(FacesPropertyMap.class);

    private FacesDialogBean jsfBean;

    private Property<T> propertyType;

    private Map<String, DialogObjectConstant> enumMap = new HashMap<String, DialogObjectConstant>();

    /**
     * Create a new Map for the given property type.
     * 
     * @param enumType - the DialogObjectConstant type for this Map
     * @param jsfBean - the JSF bean to get the property manager from
     * @param propertyType - the property type for this Map
     */
    FacesPropertyMap(Class<Enum<?>> enumType, FacesDialogBean jsfBean, Property<T> propertyType) {
    	this.jsfBean = jsfBean;
        this.propertyType = propertyType;

        Enum<?>[] enumConstants = enumType.getEnumConstants();
        for (Enum<?> enum1 : enumConstants) {
            if (enum1 instanceof DialogObjectConstant) {
                DialogObjectConstant tmp = (DialogObjectConstant) enum1;
                enumMap.put(tmp.getInternalName(), tmp);
            }
        }
    }

    private PropertyManager getPropertyManager() {
    	ViewModelImpl viewModel = (ViewModelImpl) jsfBean.getDialogView().getViewModel();
    	return viewModel.getPropertyManager();
    }

    @Override
    public T get(Object key) {
        String internalName = (String) key;
        T prop = getPropertyManager().getProperty(enumMap.get(internalName),
                propertyType);
        log.trace("Get faces property " + propertyType + " for " + key
                + ": " + prop);
        return prop;
    }

    @Override
    public T put(Object key, T value) {
        T oldValue = get(key);
        String internalName = (String) key;
        getPropertyManager().setProperty(enumMap.get(internalName),
                propertyType, value);
        log.trace("Set faces property " + propertyType + " for " + key
                + " to: " + value);
        return oldValue;
    }
    
    @Override
    public void clear() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsKey(Object key) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsValue(Object value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Set<Map.Entry<Object, T>> entrySet() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean isEmpty() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Set keySet() {
        throw new UnsupportedOperationException();
    }

    @Override
    public void putAll(Map m) {
        throw new UnsupportedOperationException();
    }

    @Override
    public T remove(Object key) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int size() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Collection values() {
        throw new UnsupportedOperationException();
    }

}
