/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.util.List;

import org.icefaces.ace.component.submenu.Submenu;

/**
 * Extending the Submenu class with mnemonic and accelerator support.
 */
public class FacesSubmenu extends Submenu {

    /** The Mnemonic key for this menu (in Alt+combination) */
    private String mnemonic;

    /** The accelerator key for this menu */
    private String accelerator;

    /**
     * Returns the preferred shortcut key. If both mnemonic and accelerator
     * is set, the accelerator is returned.
     *
     * @return shortcut key
     */
    public String getShortcutKey() {
        String key = "";

        if (hasAccelerator()) {
            key = getAccelerator();
        }
        else if (hasMnemonic()) {
            key = "Alt+" + getMnemonic();
        }

        return key;
    }

    /**
     * @return the mnemonic
     */
    public String getMnemonic() {
        return this.mnemonic;
    }

    /**
     * @param mnemonic the mnemonic to set
     */
    public void setMnemonic(final String mnemonic) {
        this.mnemonic = renderMnemonic(mnemonic);
    }

    /**
     * @return true if the menu item has a mnemonic, false otherwise.
     */
    public boolean hasMnemonic() {
        if (getMnemonic() != null && getMnemonic().length() > 0) {
            return true;
        }
        return false;
    }

    /**
     * @return the accelerator
     */
    public String getAccelerator() {
        return this.accelerator;
    }

    /**
     * @param accelerator the accelerator to set
     */
    public void setAccelerator(final String accelerator) {
        this.accelerator = accelerator;
    }

    /**
     * @return true if the menu item has an accelerator, false otherwise.
     */
    public boolean hasAccelerator() {
        if (getAccelerator() != null && getAccelerator().length() > 0) {
            return true;
        }
        return false;
    }

    private String renderMnemonic(String keys) {
        if (keys == null) {
            return null;
        }

        keys = keys.replaceAll("(?i)ALT", "");
        keys = keys.replaceAll("[+]", "");

        return keys.trim();
    }

    /**
     * Add menu items to this menu.
     * All menu items in the given list are added to this menu,
     * unless the menu item is already present in the menu.
     * 
     * @param menuItems the list of menu items to add
     */
    public void setMenuItems(List<FacesMenuItem> menuItems) {
        for (FacesMenuItem menuItem : menuItems) {
            if (!getChildren().contains(menuItem)) {
                getChildren().add(menuItem);
                menuItem.setParent(this);
            }
        }
    }
}
