/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.client.core.view.Resource;
import no.g9.client.core.view.ResourceStore;
import no.g9.service.G9Spring;

/**
 * A servlet which returns the requested resource based on the given resource
 * id. The resource and its meta info is found in the jVine resource cache.
 *
 * To get a resource, set the value of the "id" parameter to the resource id.
 * If the resource should not be purged from the resource store after use, set the
 * "purge" parameter to "false". The default is to purge the resource.
 *
 * To get the application menu, set the value of the "type" parameter to
 * "application-menu". Using the "type" parameter is optional, and the default
 * value of the "type" parameter is "resource".
 */
public class G9ResourceServlet extends HttpServlet {

    /** The URL suffix which is used in the pattern for this servlet. */
    public static final String URL_SUFFIX = "gva-resource";

    /** The "id" parameter */
    public static final String ID_PARAM = "id";

    /** The "purge" parameter */
    public static final String PURGE_PARAM = "purge";

    /** The "type" parameter */
    public static final String TYPE_PARAM = "type";

    /** The parameter value for the "type" parameter, to be used for fetching the application menu. */
    public static final String APPLICATION_MENU_TYPE = "application-menu";

    /** The parameter value for the "type" parameter, to be used for fetching resources (optional). */
    public static final String RESOURCE_TYPE = "resource";
    
    /** The name of the Spring bean which returns the application menu. */
    public static final String APPLICATION_MENU_BEAN = "gvaApplicationMenu";

    /** The content type used for serialized Java objects. */
    public static final String OBJECT_CONTENT_TYPE = "application/x-java-serialized-object";

    private static Logger logger = Logger.getLogger(G9ResourceServlet.class);

    @Override
    public void doGet(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException {

        // Get the 'type' parameter
        String resourceType = request.getParameter(G9ResourceServlet.TYPE_PARAM);
        if (G9ResourceServlet.APPLICATION_MENU_TYPE.equals(resourceType)) {
            getApplicationMenu(response);
            return;
        } else if (resourceType != null) {
            throw new ServletException("Invalid resource type parameter.");
        } else {

            // Get the 'id' parameter
            String resourceId = request.getParameter(G9ResourceServlet.ID_PARAM);
            if (resourceId == null || resourceId.equals("")) {
                throw new ServletException("Invalid or non-existent resource id parameter.");
            }

            // Get the resource
            Resource resource = null;
            ResourceStore resourceStore = ServiceLoader.getService(no.g9.client.core.view.ResourceStore.class);
            if (resourceStore != null) {
                resource = resourceStore.getResource(resourceId);
            }
            if (resource == null) {
                throw new ServletException("Non-existent resource for the given id parameter.");
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Opening resource with id " + resourceId + " of type " + resource.getMimeType() + ", named "
                        + resource.getResourceName());
            }

            // Default is to purge the resource after use
            boolean purge = true;
            String purgeStr = request.getParameter(G9ResourceServlet.PURGE_PARAM);
            if (purgeStr != null && purgeStr.toUpperCase().equals("FALSE")) {
                purge = false;
            }

            ServletOutputStream stream = null;
            ByteArrayInputStream buf = null;

            String fileName = resource.getResourceName();
            String mimeType = resource.getMimeType();
            byte[] data = resource.getResource();
            try {
                stream = response.getOutputStream();

                // Set response headers
                response.setContentType(mimeType);
                response.addHeader("Content-Disposition", "attachment; filename=" + fileName);
                response.setContentLength(data.length);

                buf = new ByteArrayInputStream(data);
                int readBytes = 0;

                // Write the data to the ServletOutputStream
                while ((readBytes = buf.read()) != -1) {
                    stream.write(readBytes);
                }
            } catch (IOException ioe) {
                throw new ServletException(ioe.getMessage());
            } finally {
                if (stream != null) {
                    stream.close();
                }
                if (purge && resourceStore != null) {
                    if (logger.isTraceEnabled()) {
                        logger.trace("Purging resource with id " + resourceId);
                    }
                    resourceStore.removeResource(resourceId);
                }
            }
        }
    }

    @Override
    public void doPost(HttpServletRequest request, HttpServletResponse response)
            throws ServletException, IOException {

        doGet(request, response);
    }

    private void getApplicationMenu(HttpServletResponse response)
            throws ServletException, IOException {
        Object menus = G9Spring.getBean(G9ResourceServlet.APPLICATION_MENU_BEAN);
        if (menus == null) {
            throw new ServletException("Resource type application-menu, bean not found.");
        }

        ObjectOutputStream stream = null;
        try {
            stream = new ObjectOutputStream(response.getOutputStream());
            stream.writeObject(menus);

            // Set response headers
            response.setContentType(G9ResourceServlet.OBJECT_CONTENT_TYPE);

        } catch (IOException ioe) {
            throw new ServletException(ioe.getMessage());
        } finally {
            if (stream != null) {
                stream.close();
            }
            response.getOutputStream().close();
        }
    }

}