/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces.tree;

import java.util.ArrayList;
import java.util.List;

import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.ViewModel;

import org.icefaces.ace.model.tree.LazyNodeDataModel;
import org.icefaces.ace.model.tree.NodeStateMap;

/**
 * A lazily loaded tree model. Subclasses must implement abstract method <code>
 * loadChildrenForNode(T node)</code>.
 *
 * @param <T> the generated FacesTreeNode class
 * @param <L> the generated ListRow class
 */
public abstract class FacesLazyTreeModel <T extends FacesTreeNode, L extends ListRow> extends FacesDefaultTreeModel<T,L> {
		
	private LazyNodeDataModel<T> lazyModel = new LazyModel();
	
	/**
	 * Create a new lazy tree model, setting the viewmodel and a node state map.
	 * 
	 * @param viewModel the dialogs viewmodel
	 * @param nodeStateMap a node state map for the nodes
	 */
	public FacesLazyTreeModel(ViewModel viewModel, NodeStateMap nodeStateMap) {
		super(viewModel, nodeStateMap);
		treeData.deleteObservers();
	}

	@Override
	public Object getDataModel() {
		return lazyModel;
	}
	
	/**
	 * Called by the system to populate a node with children.
	 * 
	 * @param node the node for which to load its children
	 * @return a list of child nodes
	 */
	public abstract List<T> loadChildrenForNode(T node);
	

	class LazyModel extends LazyNodeDataModel<T> {

		@Override
		public List<T> loadChildrenForNode(T node) {
			
			List<T> srcList = FacesLazyTreeModel.this.loadChildrenForNode(node);
			List<T> dstList = new ArrayList<>();
			
			if(node == null) {
				FacesLazyTreeModel.this.setRootNodes(srcList);
			}
		
			filterTree(srcList, dstList);
			sortTree(dstList);
			
			return dstList;
		}
		
	}

}
