/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.client.core.view.faces.tree;

import java.util.Observer;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;

import no.g9.client.core.controller.DialogObjectConstant;
import no.g9.client.core.view.ListRow;
import no.g9.client.core.view.tree.TreeNode;


/**
 * Represents a tree node in a tree. This implementation extends <code>DefaultMutableTreeNode</code>,
 * adding support for notifying subscribing classes.
 *
 */
@SuppressWarnings("serial")
public class FacesTreeNode extends DefaultMutableTreeNode implements TreeNode {

	private DialogObjectConstant nodeConst;
    private NodeObservable obs = new NodeObservable();
    private Object domainObject;
    
	/**
	 * Create a new node with given <code>ListRow</code> and dialog object constant representing the node. 
	 * 
	 * @param userData the ListRow holding the node's data.
	 * @param nodeConst the dialog object constant representing the node.
	 */
	public <L extends ListRow> FacesTreeNode(L userData, DialogObjectConstant nodeConst) {
		super(userData);
		this.nodeConst = nodeConst;
	}

    /**
     * Returns the domain object associated with the node.
     * 
     * @return the domain object
     */
    public Object getDomainObject() {
		return domainObject;
	}

	/**
	 * Set the domain object associated with the node.
	 * 
	 * @param domainObject the domain object
	 */
	public void setDomainObject(Object domainObject) {
		this.domainObject = domainObject;
	}
	
	/**
	 * Adds an observer to notify about changes to the node
	 * 
	 * @param o the observer
	 */
	public void addObserver(Observer o) {
		obs.addObserver(o); 
	}

	  ////////////////////////////
	 ////     TreeNode      /////
	////////////////////////////
	
    /**
     * Returns the text representation of the node
     * 
     * @return a string representing the node
     */
    public String getType() {
    	return nodeConst.toString(); 
    }

    @Override
	public DialogObjectConstant getNodeConst() {
    	return nodeConst;
    }	

	@Override
	@SuppressWarnings("unchecked")
	public <L extends ListRow> L getListRow() {
		return (L) getUserObject();
	}
	
	  ////////////////////////////
	 ////   MutableTreeNode /////
	////////////////////////////
	
	@Override
	public void insert(MutableTreeNode child, int index) {
		super.insert(child, index);
		obs.notifyObservers();
	}

	@Override
	public void remove(int index) {
		super.remove(index);
		obs.notifyObservers();
	}

	@Override
	public void remove(MutableTreeNode node) {
		super.remove(node);
		obs.notifyObservers();
	}

	@Override
	public void setUserObject(Object object) {
		super.setUserObject(object);
		obs.notifyObservers();
	}

	@Override
	public void removeFromParent() {
		super.removeFromParent();
		obs.notifyObservers();
	}

	@Override
	public void setParent(MutableTreeNode newParent) {
		super.setParent(newParent);
		obs.notifyObservers();
	}
	
	  /////////////////////////////////////
	 ////   DefaultMutableTreeNode   /////
	/////////////////////////////////////
	
	@Override
	public void add(MutableTreeNode newChild) {
		super.add(newChild);
		obs.notifyObservers();
	}
	
    @Override
	public void removeAllChildren() {
    	super.removeAllChildren();
    	obs.notifyObservers();
    }
    
	@Override
	public Object clone() {
		FacesTreeNode newNode = (FacesTreeNode) super.clone();
		newNode.obs.deleteObservers();	
		return newNode;
	}
	
}
