/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.SharedSessionContract;
import org.hibernate.StaleObjectStateException;
import org.hibernate.StaleStateException;
import org.hibernate.Transaction;
import org.hibernate.collection.spi.PersistentCollection;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Example;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.Restrictions;
import org.hibernate.proxy.HibernateProxy;
import org.hibernate.sql.JoinType;

import no.esito.log.Logger;
import no.g9.domain.DomainUtil;
import no.g9.exception.G9DataAccessException;
import no.g9.exception.G9ServiceException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.support.EFindMethod;
import no.g9.support.FindData;
import no.g9.support.FindOrder;
import no.g9.support.TypeTool;

/**
 * Common methods for a Hibernate implementation of the Session interface.
 */
public abstract class AbstractHibernateSession implements Session {

	private Logger logger = no.esito.log.Logger.getLogger(AbstractHibernateSession.class);

	@Override
    public Object get(FindData findData) throws G9DataAccessException {
        if (findData.getFindMethod().equals(EFindMethod.QUERY)) {
            return query(findData, false);
        } else if (findData.getFindMethod().equals(EFindMethod.EXAMPLE)) {
            return qbe(findData, false);
        } else if (findData.getFindMethod().equals(EFindMethod.CRITERIA)) {
            return qbc(findData, false);
        } else if (findData.getFindMethod().equals(EFindMethod.DEFAULT)) {
            return simpleGet(findData);
        }
        return null;
    }

    @Override
    public List<?> getAll(FindData findData) throws G9DataAccessException {
        if (findData.getFindMethod().equals(EFindMethod.QUERY)) {
            return (List<?>) query(findData, true);
        } else if (findData.getFindMethod().equals(EFindMethod.EXAMPLE)) {
            return (List<?>) qbe(findData, true);
        } else if (findData.getFindMethod().equals(EFindMethod.CRITERIA)
                || findData.getFindMethod().equals(EFindMethod.DEFAULT)) {
            return (List<?>) qbc(findData, true);
        }
        return null;
    }

    @Override
    public Serializable insert(Object object) throws G9DataAccessException {
        Serializable id = null;
        try {
            id = insertImpl(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return id;
    }

    @Override
    public void update(Object object) throws G9DataAccessException {
        try {
            updateImpl(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public Object merge(Object object) throws G9DataAccessException {
        try {
            return mergeImpl(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void delete(Object object) throws G9DataAccessException {
        try {
            deleteImpl(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public boolean isInitialized(Object object) {
        if (object instanceof HibernateProxy) {
            return !((HibernateProxy) object).getHibernateLazyInitializer().isUninitialized();
        }
        else if (object instanceof PersistentCollection) {
            return ((PersistentCollection) object).wasInitialized();
        }
        return true;
    }

    @Override
    public void initialize(Object object) throws G9DataAccessException {
        try {
            org.hibernate.Hibernate.initialize(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void refresh(Object object) throws G9DataAccessException {
        try {
            refreshImpl(object);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void close() throws G9DataAccessException {
        try {
            closeImpl();
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void flush() throws G9DataAccessException {
        try {
            flushImpl();
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void beginTransaction() throws G9DataAccessException {
        try {
            beginTransactionImpl();
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void commitTransaction() throws G9DataAccessException {
        try {
        	getSession().getTransaction().commit();
        } catch (StaleStateException e) {
            Object[] args = null;
            if (e instanceof StaleObjectStateException) {
                args = new Object[1];
                args[0] = ((StaleObjectStateException) e).getEntityName();
            }
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.SM_UPDATED_BY_ANOTHER_USER, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9ServiceException(msg);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    @Override
    public void rollbackTransaction() {
        try {
        	getSession().getTransaction().rollback();
        } catch (Exception e) {
        	logger.warn("Caught exception while trying to rollback transaction", e);
        }
    }

    @Override
    public boolean canRollbackTransaction() {
    	try {
    		return getSession().getTransaction().getStatus().canRollback();
        } catch (Exception e) {
        	logger.warn("Caught exception while checking if transaction can be rolled back", e);
        	return false;
        }
    }

    @Override
    public boolean hasActiveTransaction() throws G9DataAccessException {
        try {
            return hasActiveTransactionImpl();
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
    }

    abstract public SharedSessionContract getSession();

    /**
     * Build and execute a Hibernate query from the given FindData.
     *
     * @param findData - the query is based on the given FindData.
     * @param all - false if only the first result matching the query is returned.
     * @return the instances matching the query.
     * @throws G9DataAccessException (no doc)
     */
    protected Object query(FindData findData, boolean all)
            throws G9DataAccessException {
        Object result = null;
        try {
            Query query = buildQuery(findData.getQuery(), findData.getQueryParameters(), findData.getUseJpaPositionalParameters());
            applyFindData(query, findData);
            if (all) {
                result = query.list();
            } else {
                result = query.setMaxResults(1).uniqueResult();
            }
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return result;
    }

    private Query buildQuery(String queryString, List<Object> parameters, boolean jpaStyle) {
        Query query = buildQueryImpl(queryString);
        if (parameters != null) {
            Iterator<Object> iterator = parameters.iterator();
            for (int i = 0; iterator.hasNext(); ++i) {
            	if (jpaStyle) {
            		query.setParameter(String.valueOf(i+1), iterator.next());
            	}
            	else {
            		query.setParameter(i, iterator.next());
            	}
            }
        }
        return query;
    }

    /**
     * Build a HQL query string for the given class and list of attribute names.
     * The query string uses positional parameters (if any).
     *
     * @param clazz - the class for the query.
     * @param keyAttributeNames - list containing the name of the key attributes.
     * @return the HQL query string.
     */
    @Override
    public String buildQueryString(Class<?> clazz,
            List<String> keyAttributeNames) {
        StringBuffer query = new StringBuffer("from ");
        query.append(clazz.getName());
        if (keyAttributeNames != null) {
            query.append(" q where");
            Iterator<String> iterator = keyAttributeNames.iterator();
            while (iterator.hasNext()) {
                String attributeName = iterator.next();
                query.append(" q.");
                query.append(attributeName);
                query.append(" = ?");
                if (iterator.hasNext()) {
                    query.append(" and");
                }
            }
        }
        return query.toString();
    }

    @Override
    public List<Object> buildQueryParameterValueList(Object key,
            List<String> keyAttributeNames) {
        List<Object> params = null;
        if (keyAttributeNames != null) {
            params = new LinkedList<Object>();
            Class<?> clazz = DomainUtil.getDomainClass(key);
            Iterator<String> iterator = keyAttributeNames.iterator();
            String attributeName = null;
            try {
                while (iterator.hasNext()) {
                    attributeName = iterator.next();
                    Method method = clazz.getMethod(TypeTool
                            .asBeanGetter(attributeName), (Class[]) null);
                    // Work-around for bug #4533479 in java
                    method.setAccessible(true);
                    params.add(method.invoke(key, (Object[]) null));
                }
            } catch (Exception e) {
                Object[] args = { attributeName };
                Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_QUERY_PARAM_ERROR, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9DataAccessException(e, msg);
            }
        }
        return params;
    }

    @Override
    public List<Object> buildCriterionList(Object key,
            List<String> keyAttributeNames) {
        List<Object> criterions = new LinkedList<Object>();
        if (keyAttributeNames != null) {
            List<Object> values = buildQueryParameterValueList(key,
                    keyAttributeNames);
            Iterator<String> nameIter = keyAttributeNames.iterator();
            Iterator<Object> valueIter = values.iterator();
            while (nameIter.hasNext()) {
                criterions.add(Restrictions.eq(nameIter.next(), valueIter
                        .next()));
            }
        }
        return criterions;
    }

    /**
     * Create criteria projections from findData.
     *
     * @param criteria - the criteria to set the projection on.
     * @param findData - the input FindData.
     * @return the updated criteria.
     */
    protected Criteria applyFindDataProjection(Criteria criteria, FindData findData) {
        if (findData != null) {
            Object projection = findData.getCriteriaProjection();
            if (projection != null && projection instanceof Projection) {
                criteria.setProjection((Projection) projection);
            }
        }
        return criteria;
    }

    /**
     * Create criteria aliases for each alias instance in findData.
     *
     * @param criteria - the criteria to update.
     * @param findData - the FindData with the aliases.
     * @return the updated criteria.
     */
    protected Criteria applyFindDataAlias(Criteria criteria, FindData findData) {
        if (findData != null) {
            if (findData.getCriterionAliases() != null) {
                for (String associationPath : findData.getCriterionAliases().keySet()) {
                    String alias = findData.getCriterionAliases().get(associationPath);
                    JoinType joinType = (JoinType) findData.getCriterionJoinTypes().get(associationPath);
                    Criterion withClause = (Criterion) findData.getCriterionWithClauses().get(associationPath);
                    if (joinType != null && withClause != null) {
                        criteria.createAlias(associationPath, alias, joinType, withClause);
                    }
                    else if (joinType != null) {
                        criteria.createAlias(associationPath, alias, joinType);
                    }
                    else {
                        criteria.createAlias(associationPath, alias);
                    }
                }
            }
        }
        return criteria;
    }

    /**
     * Apply settings from the findData object to the Hibernate criteria.
     *
     * @param criteria - the criteria to update.
     * @param findData - the FindData with the criteria settings.
     * @return the updated criteria.
     */
    protected Criteria applyFindData(Criteria criteria, FindData findData) {
        if (findData != null) {
            if (findData.getMaxResults() > 0) {
                criteria.setMaxResults(findData.getMaxResults());
            }
            if (findData.getFirstResult() > 0) {
                criteria.setFirstResult(findData.getFirstResult());
            }
            if (findData.getOrder() != null) {
                Iterator<FindOrder> iterator = findData.getOrder().iterator();
                while (iterator.hasNext()) {
                    FindOrder findOrder = iterator.next();
                    Order order = findOrder.getAscending() ? Order
                            .asc(findOrder.getPropertyName()) : Order
                            .desc(findOrder.getPropertyName());
                    if (findOrder.getIgnoreCase()) {
                        order = order.ignoreCase();
                    }
                    criteria.addOrder(order);
                }
            }
        }
        return criteria;
    }

    /**
     * Apply settings from the findData object to the Hibernate query.
     *
     * @param query - the query to update.
     * @param findData - the FindData with the query settings.
     * @return the updated query.
     */
    protected Query applyFindData(Query query, FindData findData) {
        if (findData != null) {
            if (findData.getMaxResults() > 0) {
                query.setMaxResults(findData.getMaxResults());
            }
            if (findData.getFirstResult() > 0) {
                query.setFirstResult(findData.getFirstResult());
            }
        }
        return query;
    }

    /**
     * Execute a simple get operation.
     *
     * @param findData - holds the class and key information for the get.
     * @return the found object.
     * @throws G9DataAccessException if an exception occurred.
     */
    protected Object simpleGet(FindData findData)
            throws G9DataAccessException {
        Object result = null;
        try {
            result = simpleGetImpl(findData);
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return result;
    }

    /**
     * Execute a Query-By-Example operation.
     *
     * @param findData - holds the input parameters for the QBE.
     * @param all - false if only the first result matching the query is returned.
     * @return the found object.
     * @throws G9DataAccessException if an exception occurred.
     */
    protected Object qbe(FindData findData, boolean all)
            throws G9DataAccessException {
        Object result = null;
        try {
            Criteria criteria = createCriteriaImpl(findData);
            criteria.add(Example.create(findData.getExample()));
            applyFindData(criteria, findData);
            if (all) {
                result = criteria.list();
            } else {
                result = criteria.setMaxResults(1).uniqueResult();
            }
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return result;
    }

    /**
     * Execute a Criteria based query operation.
     *
     * @param findData - holds the input parameters for the query.
     * @param all - false if only the first result matching the query is returned.
     * @return the found object.
     * @throws G9DataAccessException if an exception occurred.
     */
    protected Object qbc(FindData findData, boolean all)
            throws G9DataAccessException {
        Object result = null;
        try {
            Criteria criteria = createCriteriaImpl(findData);
            applyFindDataAlias(criteria, findData);
            applyFindDataProjection(criteria, findData);
            Iterator<Object> iter = findData.getCriterions().iterator();
            while (iter.hasNext()) {
                Object criterion = iter.next();
                if (criterion instanceof Criterion) {
                    criteria.add((Criterion) criterion);
                }
            }
            applyFindData(criteria, findData);
            if (all) {
                result = criteria.list();
            } else {
                result = criteria.setMaxResults(1).uniqueResult();
            }
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return result;
    }

    /**
     * Subclass specific implementation of simpleGet.
     *
     * @param findData - the class and key info for the get operation.
     * @return the found instance.
     */
    protected abstract Object simpleGetImpl(FindData findData);

    /**
     * Subclass specific implementation of createCriteria.
     *
     * @param findData - the FindData input for the criteria to be created.
     * @return the new criteria.
     */
    protected abstract Criteria createCriteriaImpl(FindData findData);

    /**
     * Subclass specific implementation of insert.
     *
     * @param object - the object to insert.
     * @return the key for the inserted object.
     */
    protected abstract Serializable insertImpl(Object object);

    /**
     * Subclass specific implementation of update.
     *
     * @param object - the object to update.
     */
    protected abstract void updateImpl(Object object);

    /**
     * Subclass specific implementation of merge.
     *
     * @param object - the object to merge.
     * @return the merged object.
     */
    protected abstract Object mergeImpl(Object object);

    /**
     * Subclass specific implementation of delete.
     *
     * @param object - the object to delete.
     */
    protected abstract void deleteImpl(Object object);

    /**
     * Subclass specific implementation of refresh.
     *
     * @param object - the object to refresh.
     */
    protected abstract void refreshImpl(Object object);

    /**
     * Subclass specific implementation of session close.
     */
    protected abstract void closeImpl();

    /**
     * Subclass specific implementation of session flush.
     */
    protected abstract void flushImpl();

    /**
     * Subclass specific implementation of session beginTransaction.
     *
     * @return the transaction object.
     */
    protected abstract Transaction beginTransactionImpl();

    /**
     * Subclass specific implementation of session hasActiveTransaction.
     *
     * @return true if the session has an active transaction.
     */
    protected abstract boolean hasActiveTransactionImpl();

    /**
     * Subclass specific implementation of buildQuery.
     *
     * @param queryString - the input for the query.
     * @return the new query.
     */
    protected abstract Query buildQueryImpl(String queryString);
}