/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.Configuration;
import org.hibernate.service.ServiceRegistry;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;

import no.esito.log.Logger;
import no.g9.exception.G9DataAccessException;
import no.g9.message.CRuntimeMsg;
import no.g9.message.Message;
import no.g9.message.MessageSystem;
import no.g9.service.G9Spring;
import no.g9.support.Registry;
import no.g9.support.RegistrySet;

/**
 * Hibernate version of SessionFactory implemented as singleton. Holds a map of
 * all Hibernate SessionFactories, indexed by a logical name. Also holds a map
 * of all ThreadLocal sessions indexed by the same name.
 */
public class HibernateSessionFactory implements SessionFactory {

    /** The singleton object */
    private static SessionFactory sessionFactory;

    /**
     * The default property suffix used for the database server key. Change this
     * if you want another default.
     */
    private static String databaseServerSuffix = "server";

    /**
     * The default property suffix used for the database properties key. Change
     * this if you want another default.
     */
    private static String databasePropertiesSuffix = "properties";

    /**
     * The default property suffix used for the database configuration key.
     * Change this if you want another default.
     */
    private static String databaseConfigurationSuffix = "configuration";

    /** The bean prefix used for the Spring beans */
    private static String beanPrefix = "sessionFactory";

    /** The default part of the bean ID used for the Spring beans */
    private static String beanDefaultPart = "default";

    /**
     * The name of the global hibernate properties file. This file will always
     * be loaded by Hibernate. The name is hard coded in Hibernate, there is no
     * way to ask Hibernate for the file name.
     */
    private static final String hibernatePropertiesFile = "hibernate.properties";

    /** The default context, used if lookup on a given context fails */
    private static final String DEFAULT_CONTEXT = "*";

    /** A map of all known Hibernate SessionFactories */
    private final Map<String, org.hibernate.SessionFactory> sessionFactories = new HashMap<String, org.hibernate.SessionFactory>();

    /** A map of all ThreadLocal sessions */
    private final Map<String, ThreadLocal<Object>> sessionHolders = new HashMap<String, ThreadLocal<Object>>();

    /** The logger */
    private Logger logger = Logger.getLogger(HibernateSessionFactory.class);

    /**
     * Private constructor.
     */
    private HibernateSessionFactory() {
        // Empty.
    }

    /**
     * @param contextName (missing javadoc)
     * @param mappingName (missing javadoc)
     * @return (missing javadoc)
     * @throws no.g9.exception.G9DataAccessException (missing javadoc)
     *
     * @see no.g9.dataaccess.SessionFactory#getSession(String, String)
     */
    @Override
    public synchronized Session getSession(final String contextName, final String mappingName)
            throws G9DataAccessException {
        ThreadLocal<Object> sessionHolder = getSessionHolder(contextName, mappingName);
        Session s = (Session) sessionHolder.get();
        // Open a new Session, if this Thread has none yet
        if (s == null) {
            s = openSession(getSessionFactory(contextName, mappingName), false);
            sessionHolder.set(s);
        }
        return s;
    }

    /**
     * @param contextName (missing javadoc)
     * @param mappingName (missing javadoc)
     * @return (missing javadoc)
     * @throws no.g9.exception.G9DataAccessException (missing javadoc)
     *
     * @see no.g9.dataaccess.SessionFactory#getStatelessSession(String, String)
     */
    @Override
    public synchronized Session getStatelessSession(final String contextName, final String mappingName)
            throws G9DataAccessException {
        return openSession(getSessionFactory(contextName, mappingName), true);
    }

    /**
     * @param session (missing javadoc)
     * @return (missing javadoc)
     * @throws IllegalArgumentException (missing javadoc)
     * @throws no.g9.exception.G9DataAccessException (missing javadoc)
     *
     * @see no.g9.dataaccess.SessionFactory#cloneSession(no.g9.dataaccess.Session)
     */
    @Override
    public synchronized Session cloneSession(final Session session)
            throws IllegalArgumentException, G9DataAccessException {
        if (!(session instanceof HibernateSession)) {
            throw new IllegalArgumentException(
                    "Not supported for this Session implementation");
        }
        return openSession(((HibernateSession) session).getSession().getSessionFactory(), false);
    }

    /**
     * @param s (missing javadoc)
     * @throws no.g9.exception.G9DataAccessException (missing javadoc)
     * @see no.g9.dataaccess.SessionFactory#closeSession(no.g9.dataaccess.Session)
     */
    @Override
    public synchronized void closeSession(final Session s) throws G9DataAccessException {
        if (s != null) {
            Iterator<ThreadLocal<Object>> iterator = sessionHolders.values().iterator();
            while (iterator.hasNext()) {
                ThreadLocal<Object> sessionHolder = iterator.next();
                Session itSession = (Session) sessionHolder.get();
                if (itSession == s) {
                    sessionHolder.set(null);
                    break;
                }
            }
            try {
                s.close();
            } catch (Exception e) {
                Object[] args = { e.getMessage() };
                Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9DataAccessException(e, msg);
            }
        }
    }

    /**
     * Switch the cached <code>Session</code> for the given
     * <code>contextName</code> and <code>mappingName</code> with a new one.
     * Intended for internal use only!
     *
     * @param contextName the database context name
     * @param mappingName the database mapping name
     * @param newSession the new <code>Session</code> to set
     * @return the previously used <code>Session</code> (or <code>null</code>)
     */
    public synchronized Session switchSession(final String contextName, final String mappingName, final Session newSession) {
        ThreadLocal<Object> sessionHolder = getSessionHolder(contextName, mappingName);
        Session oldSession = (Session) sessionHolder.get();
        sessionHolder.set(newSession);
        return oldSession;
    }

    /**
     * @return the singleton session factory.
     */
    public static synchronized SessionFactory getSessionfactory() {
        if (HibernateSessionFactory.sessionFactory == null) {
            HibernateSessionFactory.sessionFactory = new HibernateSessionFactory();
        }
        return HibernateSessionFactory.sessionFactory;
    }

    private org.hibernate.SessionFactory getSessionFactory(final String contextName, final String mappingName) {
        String keyName = getMapKey(contextName, mappingName);
        org.hibernate.SessionFactory sf = sessionFactories.get(keyName);
        if (sf == null) {
            sf = createSessionFactory(contextName, mappingName);
            sessionFactories.put(keyName, sf);
        }
        return sf;
    }

    private ThreadLocal<Object> getSessionHolder(final String contextName, final String mappingName) {
        String keyName = getMapKey(contextName, mappingName);
        ThreadLocal<Object> sessionHolder = sessionHolders.get(keyName);
        if (sessionHolder == null) {
            sessionHolder = new ThreadLocal<Object>();
            sessionHolders.put(keyName, sessionHolder);
        }
        return sessionHolder;
    }

    private String getMapKey(final String contextName, final String mappingName) {
        return (contextName == null || mappingName == null) ? "" : contextName + mappingName;
    }

    private Session openSession(org.hibernate.SessionFactory sf, boolean stateless)
            throws G9DataAccessException {
        Session s = null;
        try {
            if (stateless) {
                s = new HibernateStatelessSession(sf.openStatelessSession());
            }
            else {
                s = new HibernateSession(sf.openSession());
            }
        } catch (Exception e) {
            Object[] args = { e.getMessage() };
            Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
            MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
            throw new G9DataAccessException(e, msg);
        }
        return s;
    }

    private org.hibernate.SessionFactory createSessionFactory(
            final String contextName, final String mappingName)
            throws G9DataAccessException {

        org.hibernate.SessionFactory factory = null;
        Configuration cfg = null;
        ServiceRegistry sr = null;

        Registry reg = Registry.getRegistry();

        // Get the server name if not using the default session factory
        String serverName = null;
        if (contextName != null && mappingName != null) {
            final String propName= mappingName + "." + HibernateSessionFactory.databaseServerSuffix;
            serverName= reg.getDatabaseContextProperty(contextName, propName);
            if (serverName == null) {
                serverName= reg.getDatabaseContextProperty(DEFAULT_CONTEXT, propName);
            }
        }

        // Try using Spring beans to create a session factory
        String propertiesBeanId = HibernateSessionFactory
                .getPropertiesBeanId(serverName);
        String resourcesBeanId = HibernateSessionFactory
                .getResourcesBeanId(serverName != null ? mappingName : null);
        HibernateProperties hibernateProperties = HibernateSessionFactory
                .getHibernateProperties(propertiesBeanId);
        HibernateResources hibernateResources = HibernateSessionFactory
                .getHibernateResources(resourcesBeanId);
        if (hibernateProperties != null && hibernateResources != null) {
            try {
                logger.trace("Creating a new SessionFactory using Spring["
                        + serverName + ", " + mappingName + "]");
                cfg = HibernateSessionFactory.buildConfig(hibernateProperties, hibernateResources);
                sr = HibernateSessionFactory.buildServiceRegistry(cfg);
                factory = cfg.buildSessionFactory(sr);
            } catch (Exception e) {
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.DB_ORM_ERROR, e.getMessage());
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9DataAccessException(e, msg);
            }
        }

        // Use the Hibernate cfg.xml approach if not using Spring
        if (factory == null) {
            String propertyFileName = null;
            String configFileName = null;
            if (serverName != null) {
                propertyFileName = reg.getDatabaseServerProperty(serverName,
                        HibernateSessionFactory.databasePropertiesSuffix);
                configFileName = reg.getDatabaseMappingProperty(mappingName,
                        HibernateSessionFactory.databaseConfigurationSuffix);
            }
            try {
                logger.trace("Creating a new SessionFactory using cfg.xml["
                        + propertyFileName + ", " + configFileName + "]");
                cfg = HibernateSessionFactory.buildConfig(propertyFileName, configFileName);
                sr = HibernateSessionFactory.buildServiceRegistry(cfg);
                factory = cfg.buildSessionFactory(sr);
            } catch (Exception e) {
                Message msg = MessageSystem.getMessageFactory().getMessage(
                        CRuntimeMsg.DB_ORM_ERROR, e.getMessage());
                MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
                throw new G9DataAccessException(e, msg);
            }
        }

        return factory;
    }

    /**
     * This is a singleton, clone is not supported.
     *
     * @see java.lang.Object#clone()
     */
    @Override
    public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * @return the database configuration property suffix.
     */
    public static String getDatabaseConfigurationSuffix() {
        return HibernateSessionFactory.databaseConfigurationSuffix;
    }

    /**
     * Set the database configuration property suffix.
     *
     * @param databaseConfigurationSuffix (missing javadoc)
     */
    public static void setDatabaseConfigurationSuffix(
            final String databaseConfigurationSuffix) {
        HibernateSessionFactory.databaseConfigurationSuffix = databaseConfigurationSuffix;
    }

    /**
     * @return the database properties property suffix.
     */
    public static String getDatabasePropertiesSuffix() {
        return HibernateSessionFactory.databasePropertiesSuffix;
    }

    /**
     * Get the database properties property suffix.
     *
     * @param databasePropertiesSuffix (missing javadoc)
     */
    public static void setDatabasePropertiesSuffix(
            final String databasePropertiesSuffix) {
        HibernateSessionFactory.databasePropertiesSuffix = databasePropertiesSuffix;
    }

    /**
     * @return the database server property suffix.
     */
    public static String getDatabaseServerSuffix() {
        return HibernateSessionFactory.databaseServerSuffix;
    }

    /**
     * Get the database server property suffix.
     *
     * @param databaseServerSuffix (missing javadoc)
     */
    public static void setDatabaseServerSuffix(String databaseServerSuffix) {
        HibernateSessionFactory.databaseServerSuffix = databaseServerSuffix;
    }

    /**
     * Applies properties from a Hibernate Configuration to a ServiceRegistry, which holds
     * services which need access to the SessionFactory during initialization.
     *
     * @param cfg The Hibernate {@link Configuration}
     *
     * @return a new {@link ServiceRegistry} with properties from cfg applied.
     *
     */
    static ServiceRegistry buildServiceRegistry(Configuration cfg) {
        return new StandardServiceRegistryBuilder().applySettings(cfg.getProperties()).build();
    }

    /**
     * Build a new Hibernate configuration based on the properties and
     * resources given.
     *
     * @param hibernateProperties the set of properties to use
     * @param hibernateResources the resource files (mappings) to use
     *
     * @return a new Hibernate configuration
     */
    static Configuration buildConfig(HibernateProperties hibernateProperties,
            HibernateResources hibernateResources) {
        Properties props = hibernateProperties.buildProperties();
        Configuration cfg = new Configuration();
        cfg.addProperties(props);
        hibernateResources.addResourcesToConfiguration(cfg);
        return cfg;
    }

    /**
     * Build a new Hibernate configuration based on the property and
     * configuration file names given.
     *
     * @param propertyFileName the file containing Hibernate properties
     * @param configFileName the file with a Hibernate configuration
     *
     * @return a new Hibernate configuration
     */
    static Configuration buildConfig(String propertyFileName, String configFileName) {
        Configuration cfg = new Configuration();
        Properties properties = null;

        // No need to load the global Hibernate properties file once more.
        if (propertyFileName != null &&
            !"".equals(propertyFileName) &&
            !HibernateSessionFactory.hibernatePropertiesFile.equals(propertyFileName)) {
            properties = RegistrySet.loadProperties(propertyFileName, true);
        }

        if (properties != null) {
            cfg.setProperties(properties);
        }

        if (configFileName != null) {
            cfg.configure(configFileName);
        } else {
            // Default configuration, reading from hibernate.cfg.xml
            cfg.configure();
        }
        return cfg;
    }

    /**
     * Load the HibernateProperties bean with the given ID.
     *
     * @param beanID the Spring bean ID
     *
     * @return a HibernateProperties object, or null if bean not found
     */
    static HibernateProperties getHibernateProperties(String beanID) {
        HibernateProperties retVal = null;
        try {
            retVal = G9Spring.getBean(HibernateProperties.class, beanID);
        }
        catch (NoSuchBeanDefinitionException e) {
            // Bean not found, return null
        }
        return retVal;
    }

    /**
     * Load the HibernateResources bean with the given ID.
     *
     * @param beanID the Spring bean ID
     *
     * @return a HibernateResources object, or null if bean not found
     */
    static HibernateResources getHibernateResources(String beanID) {
        HibernateResources retVal = null;
        try {
            retVal = G9Spring.getBean(HibernateResources.class, beanID);
        }
        catch (NoSuchBeanDefinitionException e) {
         // Bean not found, return null
        }
        return retVal;
    }

    /**
     * Get the Spring bean ID for the Hibernate properties.
     *
     * @param serverName the server name, or null if using the default bean
     *
     * @return the Spring bean ID
     */
    static String getPropertiesBeanId(String serverName) {
        String propertyBeanId = null;
        if (serverName != null) {
            propertyBeanId = HibernateSessionFactory.beanPrefix + "_"
            + serverName + "_"
            + HibernateSessionFactory.databasePropertiesSuffix;
        }
        else {
            propertyBeanId = HibernateSessionFactory.beanPrefix + "_"
                    + HibernateSessionFactory.beanDefaultPart + "_"
                    + HibernateSessionFactory.databasePropertiesSuffix;
        }
        return propertyBeanId;
    }

    /**
     * Get the Spring bean ID for the Hibernate resources.
     *
     * @param mappingName the G9 database mapping name, or null if using
     *                    the default bean
     *
     * @return the Spring bean ID
     */
    static String getResourcesBeanId(String mappingName) {
        String resourceBeanId = null;
        if (mappingName != null) {
            resourceBeanId = HibernateSessionFactory.beanPrefix + "_"
                    + mappingName + "_"
                    + HibernateSessionFactory.databaseConfigurationSuffix;
        }
        else {
            resourceBeanId = HibernateSessionFactory.beanPrefix + "_"
                    + HibernateSessionFactory.beanDefaultPart + "_"
                    + HibernateSessionFactory.databaseConfigurationSuffix;
        }
        return resourceBeanId;
    }

}
