/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.io.Serializable;

import no.g9.exception.G9DataAccessException;
import no.g9.message.*;
import no.g9.support.FindData;

import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;

/**
 * A Hibernate implementation of the <code>Session</code> interface.
 */
public class HibernateStatelessSession extends AbstractHibernateSession {

    /**
     * The Hibernate session object.
     */
    protected org.hibernate.StatelessSession session;

    /**
     * Create a new session with the given Hibernate session object.
     *
     * @param session (missing javadoc)
     */
    public HibernateStatelessSession(org.hibernate.StatelessSession session) {
        this.session = session;
    }

    /**
     * Get access to the Hibernate Session object
     *
     * @return (missing javadoc)
     */
    public org.hibernate.StatelessSession getSession() {
        return this.session;
    }

    /* (non-Javadoc)
     * @see no.g9.dataaccess.Session#hasLockFlag(java.lang.Class)
     */
    @Override
    public boolean hasLockFlag(Class<?> domainClass) {
        Object[] args = { "hasLockFlag() not supported for stateless sessions" };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9DataAccessException(msg);
    }

    @Override
    protected Object simpleGetImpl(FindData findData) {
        return this.session.get(findData.getDomainClass(), findData.getKey());
    }

    @Override
    protected Criteria createCriteriaImpl(FindData findData) {
        return this.session.createCriteria(findData.getDomainClass());
    }

    @Override
    protected Serializable insertImpl(Object object) {
        return this.session.insert(object);
    }

    @Override
    protected void updateImpl(Object object) {
        this.session.update(object);
    }

    @Override
    protected Object mergeImpl(Object object) {
        Object[] args = { "merge() not supported for stateless sessions" };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9DataAccessException(msg);
    }

    @Override
    protected void deleteImpl(Object object) {
        this.session.delete(object);
    }

    @Override
    protected void refreshImpl(Object object) {
        this.session.refresh(object);
    }

    @Override
    protected void closeImpl() {
        this.session.close();
    }

    @Override
    protected void flushImpl() {
        Object[] args = { "flush() not supported for stateless sessions" };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9DataAccessException(msg);
    }

    @Override
    protected Transaction beginTransactionImpl() {
        return this.session.beginTransaction();
    }

    @Override
    protected boolean hasActiveTransactionImpl() {
        return (this.session.getTransaction().getStatus() == TransactionStatus.ACTIVE);
    }

    @Override
    protected Query buildQueryImpl(String queryString) {
        return this.session.createQuery(queryString);
    }

    @Override
    public void initialize(Object object) throws G9DataAccessException {
        Object[] args = { "initialize() not supported for stateless sessions" };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
        throw new G9DataAccessException(msg);
    }

}
