/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape.trigger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import no.esito.log.Logger;
import no.g9.jgrape.trigger.JGrapeTrigger.TriggerType;

import org.springframework.beans.factory.annotation.Autowired;

/**
 * The default implementation of the TriggerProvider interface.
 */
public class JGrapeTriggerProvider implements TriggerProvider {

    private static final Logger log = Logger
        .getLogger(JGrapeTriggerProvider.class);

    private Map<Class<?>, List<JGrapeTrigger<?>>> cachedTriggers = new HashMap<Class<?>, List<JGrapeTrigger<?>>>();

    @Autowired
    private RegistryManager registryManager;

    @Override
    public List<JGrapeTrigger<?>> getTrigger(Class<?> domainClass,
        TriggerType type) {

        log.info("Getting " + type + " triggers for " + domainClass);

        List<JGrapeTrigger<?>> triggerList = cachedTriggers.get(domainClass);
        if (triggerList == null) {
            triggerList = createTriggerList(domainClass);
            cachedTriggers.put(domainClass, triggerList);
        }

        List<JGrapeTrigger<?>> filteredList = TriggerFilter.filter(triggerList, type);

        log.info("Found triggers are: " + filteredList);

        return filteredList;
    }

    /**
     * Creates the complete list of triggers. The list starts with interface
     * triggers (from super class first), continues with triggers for super
     * classes and ends with triggers for the specified class.
     * 
     * @param domainClass the class to get triggers for.
     * @return the complete list of triggers.
     */
    private List<JGrapeTrigger<?>> createTriggerList(Class<?> domainClass) {

        log.debug("Creating trigger list for " + domainClass);

        List<JGrapeTrigger<?>> triggerList = new ArrayList<JGrapeTrigger<?>>();
        SuperClassIterator superClassTraverser = new SuperClassIterator(
            domainClass);

        if (log.isTraceEnabled()) {
            String msg = "Class list used for traversal is " + domainClass;
            log.trace(msg);
            log.trace("superclasses are: " + superClassTraverser);
        }


        // Super classes
        for (Class<?> clazz : superClassTraverser) {
            List<JGrapeTrigger<?>> interfaceList = buildInterfaceList(clazz);
            
            if (log.isTraceEnabled()) {
                log.trace("Triggers for " + clazz + " interfaces are " + interfaceList);
            }
        
            triggerList.addAll(interfaceList);
            List<JGrapeTrigger<?>> superClassTriggers = registryManager
                .getTriggerList(clazz);

            if (log.isTraceEnabled()) {
                log.trace("Triggers for superclass " + clazz + " are "
                    + superClassTriggers);
            }

            triggerList.addAll(superClassTriggers);
        }

        return removeDuplicates(triggerList);

    }

    /**
     * Get the list of all JGrape triggers registered to the interfaces of the
     * class.
     * 
     * @param clazz the class.
     * @return the list of JGrape triggers.
     */
    private List<JGrapeTrigger<?>> buildInterfaceList(Class<?> clazz) {
        List<JGrapeTrigger<?>> triggerList = new ArrayList<JGrapeTrigger<?>>();
        Class<?>[] localInterfaces = clazz.getInterfaces();
        List<Class<?>> allSuperInterfaces = new ArrayList<Class<?>>();
        for (Class<?> interfaze : localInterfaces) {
            allSuperInterfaces.addAll(getAllSuperInterfaces(interfaze));
        }

        if (log.isTraceEnabled()) {
            log.trace("Found interfaces for " + clazz + " are "
                + allSuperInterfaces);
        }
        
        for (Class<?> interfaze : allSuperInterfaces) {
            
            List<JGrapeTrigger<?>> interfaceTriggers = registryManager
            .getTriggerList(interfaze);
            triggerList.addAll(interfaceTriggers);
        }

        return triggerList;
    }

    /**
     * Returns a new list where all duplicates are removed, ordering is
     * preserved.
     * 
     * @param list the list containing possible duplicates
     * @return a list without duplicates.
     */
    private List<JGrapeTrigger<?>> removeDuplicates(List<JGrapeTrigger<?>> list) {
        log.trace("Removing possible duplicates");
        Set<JGrapeTrigger<?>> foundTriggers = new HashSet<JGrapeTrigger<?>>();
        List<JGrapeTrigger<?>> noDuplicateList = new ArrayList<JGrapeTrigger<?>>();
        for (JGrapeTrigger<?> jGrapeTrigger : list) {
            if (!foundTriggers.contains(jGrapeTrigger)) {
                noDuplicateList.add(jGrapeTrigger);
            } else {
                log.trace("Trigger " + jGrapeTrigger + " is already found.");
            }
            foundTriggers.add(jGrapeTrigger);
        }

        return noDuplicateList;
    }

    /**
     * Creates a list of all interfaces including super interfaces for the
     * specified interface.
     * 
     * @param interfaze the interface whose list to create.
     * @return a list of all interfaces.
     */
    private List<Class<?>> getAllSuperInterfaces(Class<?> interfaze) {
        assert (interfaze.isInterface());

        List<Class<?>> interfaceList = new ArrayList<Class<?>>();

        Class<?>[] interfaces = interfaze.getInterfaces();

        if (interfaces != null) {
            for (Class<?> parentInterfaze : interfaces) {
                interfaceList.addAll(getAllSuperInterfaces(parentInterfaze));
            }
        }
        interfaceList.add(interfaze);
        return interfaceList;
    }

    /**
     * An iterator from super to sub class.
     */
    private static class SuperClassIterator implements Iterable<Class<?>> {

        private List<Class<?>> superClassList;

        /**
         * Create a new Super class iterator.
         * 
         * @param clazz the class where the iterator ends.
         */
        SuperClassIterator(Class<?> clazz) {
            superClassList = buildList(clazz);
        }

        @Override
        public Iterator<Class<?>> iterator() {
            return superClassList.iterator();
        }

        @Override
        public String toString() {
            return "SuperClassIterator [superClassList=" + superClassList + "]";
        }

        /**
         * Recursively builds a list the supplied class and all super classes.
         * The list starts with the top-most superclass (Object.class), and ends
         * with the supplied class.
         * 
         * @param clazz the class to build the list from
         * @return a list containing all super classes of the clazz, and the
         *         clazz it self.
         */
        private List<Class<?>> buildList(Class<?> clazz) {
            List<Class<?>> classList;
            log.trace("Building class list for " + clazz);

            // Base case
            if (Object.class.equals(clazz)) {
                classList = new ArrayList<Class<?>>();
                classList.add(clazz);
                return classList;
            }

            Class<?> superclass = clazz.getSuperclass();
            if (log.isTraceEnabled()) {
                log.trace("Superclass of " + clazz + " is " + superclass);
            }
            if (superclass != null) {
                classList = buildList(superclass);
            } else {
                classList = new ArrayList<Class<?>>();
            }
            classList.add(clazz);
            return classList;
        }

    }

}
