/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.io.Serializable;

import no.g9.support.FindData;

import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Transaction;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.resource.transaction.spi.TransactionStatus;

/**
 * A Hibernate implementation of the <code>Session</code> interface.
 */
public class HibernateSession extends AbstractHibernateSession {

    /**
     * The Hibernate session object.
     */
    protected org.hibernate.Session session;

    /**
     * Create a new session with the given Hibernate session object.
     *
     * @param session (missing javadoc)
     */
    public HibernateSession(org.hibernate.Session session) {
        this.session = session;
    }

    /**
     * Get access to the Hibernate Session object
     *
     * @return (missing javadoc)
     */
    public org.hibernate.Session getSession() {
        return this.session;
    }

    /* (non-Javadoc)
     * @see no.g9.dataaccess.Session#hasLockFlag(java.lang.Class)
     */
    @Override
    public boolean hasLockFlag(Class<?> domainClass) {
        ClassMetadata cm = getSession().getSessionFactory().getClassMetadata(domainClass);
        if (cm == null) {
        	throw new IllegalArgumentException("Class " + domainClass
					+ " is unknown to Hibernate");
        }
        if (cm.isVersioned()) {
        	return true;
        }
        return false;
    }

    @Override
    protected Object simpleGetImpl(FindData findData) {
        return this.session.get(findData.getDomainClass(), findData.getKey());
    }

    @Override
    protected Criteria createCriteriaImpl(FindData findData) {
        return this.session.createCriteria(findData.getDomainClass());
    }

    @Override
    protected Serializable insertImpl(Object object) {
        return this.session.save(object);
    }

    @Override
    protected void updateImpl(Object object) {
        this.session.update(object);
    }

    @Override
    protected Object mergeImpl(Object object) {
        return this.session.merge(object);
    }

    @Override
    protected void deleteImpl(Object object) {
        this.session.delete(object);
    }

    @Override
    protected void refreshImpl(Object object) {
        this.session.refresh(object);
    }

    @Override
    protected void closeImpl() {
        if (this.session.isOpen()) {
            this.session.close();
        }
    }

    @Override
    protected void flushImpl() {
        this.session.flush();
    }

    @Override
    protected Transaction beginTransactionImpl() {
        return this.session.beginTransaction();
    }

    @Override
    protected boolean hasActiveTransactionImpl() {
        return (this.session.getTransaction().getStatus() == TransactionStatus.ACTIVE);
    }

    @Override
    protected Query buildQueryImpl(String queryString) {
        return this.session.createQuery(queryString);
    }

}
