/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess.usertype;

import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Properties;

import no.g9.support.Numeric;

import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.usertype.ParameterizedType;
import org.hibernate.usertype.UserType;

/**
 * Based on org.hibernate.test.typeparameters.DefaultValueIntegerType
 */
@SuppressWarnings("rawtypes")
public class BcdType implements UserType, ParameterizedType, Serializable {

    /** BigDecimal-version of Long.MIN_VALUE. */
    public static final BigDecimal BIG_MIN_LONG=  BigDecimal.valueOf(Long.MIN_VALUE);

	/** BigDecimal-version of Long.MAX_VALUE. */
    public static final BigDecimal BIG_MAX_LONG= BigDecimal.valueOf(Long.MAX_VALUE);

    private Object defaultValue = null;

    private int scale = 0;

    private boolean isG9Numeric = false;

    /**
     * @return int[]
     */
    @Override
    public int[] sqlTypes() {
        return new int[] { Types.NUMERIC };
    }

    /**
     * @return java.lang.Class
     */
    @Override
    public Class returnedClass() {
        return this.isG9Numeric ? Numeric.class : BigDecimal.class;
    }

    /**
     * @param x (missing javadoc)
     * @param y (missing javadoc)
     * @return (missing javadoc)
     * @throws HibernateException (missing javadoc)
     */
    @Override
    public boolean equals(Object x, Object y) throws HibernateException {
        if (x == y)
            return true;
        if (x == null || y == null)
            return false;
        return x.equals(y);
    }

    /**
     * @param rs (missing javadoc)
     * @param names (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws HibernateException (missing javadoc)
     * @throws SQLException (missing javadoc)
     */
    @Override
    public Object nullSafeGet(ResultSet rs, String[] names, SessionImplementor session, Object owner)
            throws HibernateException, SQLException {
    	BigDecimal result = rs.getBigDecimal(names[0]);
        return result == null ? defaultValue : (this.isG9Numeric ? fromBase(result) : result);
    }

    /**
     * @param st (missing javadoc)
     * @param value (missing javadoc)
     * @param index (missing javadoc)
     * @throws HibernateException (missing javadoc)
     * @throws SQLException (missing javadoc)
     */
    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index, SessionImplementor session)
            throws HibernateException, SQLException {
        if (value == null) {
            st.setNull(index, Types.NUMERIC);
        } else {
        	BigDecimal big = null;
        	if (this.isG9Numeric) {
        		final Numeric numericValue = (Numeric) value;
        		big = fromJava(numericValue);
        	}
        	else {
        		big = (BigDecimal) value;
        	}
            if ( this.scale == 0 &&
                 big.compareTo(BIG_MAX_LONG) <= 0 &&
                 big.compareTo(BIG_MIN_LONG) >= 0 ) {
                st.setLong(index, big.longValue());
            }
            else {
                st.setBigDecimal(index, big);
            }
        }
    }

    /**
     * @param value (missing javadoc)
     * @return  (missing javadoc)
     * @throws HibernateException (missing javadoc)
     */
    @Override
    public Object deepCopy(Object value) throws HibernateException {
        if (value==null) {
           return null;
        }
        if (this.isG9Numeric) {
        	return ((Numeric)value).clone();
        }
        BigDecimal big = (BigDecimal) value;
        return new BigDecimal(big.unscaledValue(), big.scale());
    }

    /**
     * @return (missing javadoc)
     */
    @Override
    public boolean isMutable() {
        return true;
    }

    /**
     * @param x (missing javadoc)
     * @return (missing javadoc)
     * @throws HibernateException (missing javadoc)
     */
    @Override
    public int hashCode(Object x) throws HibernateException {
        return x.hashCode();
    }

    /**
     * @param cached (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws HibernateException (missing javadoc)
     */
    @Override
    public Object assemble(Serializable cached, Object owner)
            throws HibernateException {
        return cached;
    }

    /**
     * @param value (missing javadoc)
     * @return (missing javadoc)
     * @throws HibernateException (missing javadoc)
     */
    @Override
    public Serializable disassemble(Object value) throws HibernateException {
        return (Serializable) value;
    }

    /**
     * @param original (missing javadoc)
     * @param target (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public Object replace(Object original, Object target, Object owner)
            throws HibernateException {
        return original;
    }

    /**
     * @param parameters (missing javadoc)
     */
    @Override
    public void setParameterValues(Properties parameters) {

        if (parameters != null) {
            if (parameters.get("scale") != null) {
                this.scale = Integer.parseInt((String) parameters.get("scale"));
            }
            if (parameters.get("g9numeric") != null) {
            	this.isG9Numeric = Boolean.parseBoolean((String) parameters.get("g9numeric"));
            }
            if (parameters.get("default") != null) {
                String defaultVal = (String) parameters.get("default");
                if (!defaultVal.equals("null")) {
                	if (this.isG9Numeric) {
                		this.defaultValue = new Numeric(defaultVal, this.scale);
                	}
                	else {
                		this.defaultValue = new BigDecimal(defaultVal).setScale(this.scale, BigDecimal.ROUND_HALF_UP);
                	}
                }
            }
        }
    }

    /**
     * @param value (missing javadoc)
     * @return (missing javadoc)
     */
    private Numeric fromBase(BigDecimal value) {
        if (value != null) {
            return new Numeric(value, this.scale);
        }
        return null;
    }

    /**
     * @param value (missing javadoc)
     * @return (missing javadoc)
     */
    private BigDecimal fromJava(Numeric value) {
        if (value != null) {
            return value.getValue();
        }
        return null;
    }

}
