/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess.usertype;

import java.io.Serializable;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.usertype.UserType;

/**
 * A UserType with support for reading database blobs into a byte array.
 */
public class BinaryBlobType implements UserType {

    @Override
    public int[] sqlTypes() {
        return new int[] { Types.BLOB };
    }

    @Override
    public Class<byte[]> returnedClass() {
        return byte[].class;
    }

    @Override
    public boolean equals(final Object x, final Object y) {
        return (x == y)
                || (x != null && y != null && java.util.Arrays.equals(
                        (byte[]) x, (byte[]) y));
    }

    @Override
    public Object nullSafeGet(final ResultSet rs, final String[] names,
    		SessionImplementor session, final Object owner) throws HibernateException, SQLException {
        final byte[] bytes = rs.getBytes(names[0]);
        return rs.wasNull() ? null : bytes;
    }

    @Override
    public void nullSafeSet(final PreparedStatement st, final Object value,
            final int index, SessionImplementor session) throws HibernateException, SQLException {
        if (value == null) {
            st.setNull(index, Types.BINARY);
        } else {
            final byte[] bytes = (byte[]) value;
            st.setBytes(index, bytes);
        }
    }

    @Override
    public Object deepCopy(final Object value) {
        if (value == null) {
            return null;
        }

        final byte[] bytes = (byte[]) value;
        final byte[] result = new byte[bytes.length];
        System.arraycopy(bytes, 0, result, 0, bytes.length);

        return result;
    }

    @Override
    public boolean isMutable() {
        return true;
    }

    @Override
    public Object assemble(final Serializable pCached, final Object pOwner)
            throws HibernateException {
        return pCached;
    }

    @Override
    public Serializable disassemble(final Object pValue)
            throws HibernateException {
        return (Serializable) pValue;
    }

    @Override
    public int hashCode(final Object pX) throws HibernateException {
        return java.util.Arrays.hashCode((byte[]) pX);
    }

    @Override
    public Object replace(final Object pOriginal, final Object pTarget,
            final Object pOwner) throws HibernateException {
        return pOriginal;
    }
}
