/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess.usertype;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Properties;

import no.g9.support.G9Enumerator;
import no.g9.support.ObjectFactory;

import org.hibernate.HibernateException;
import org.hibernate.engine.spi.SessionImplementor;
import org.hibernate.usertype.ParameterizedType;
import org.hibernate.usertype.UserType;

/**
 * Based on org.hibernate.test.typeparameters.DefaultValueIntegerType
 */
public class EnumeratorType implements UserType, ParameterizedType, Serializable {

	private static final String GET_VALUE_METHOD = "getValue";
	private static final String GET_METHOD = "get";

	private String defaultValue = null;
    private String enumClassName = null;
	private Class<?> enumClass;

    /**
     * @return (missing javadoc)
     */
    @Override
    public int[] sqlTypes() {
        return new int[] { Types.INTEGER };
    }

    /**
     * @return (missing javadoc)
     */
    @Override
    public Class<?> returnedClass() {
        return enumClass;
    }

    /**
     * @param x (missing javadoc)
     * @param y (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public boolean equals(Object x, Object y) throws HibernateException {
        if (x == y) return true;
        if (x == null || y == null) return false;
        return x.equals(y);
    }

    /**
     * @param rs (missing javadoc)
     * @param names (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     * @throws java.sql.SQLException (missing javadoc)
     */
    @Override
    public Object nullSafeGet(ResultSet rs, String[] names, SessionImplementor session, Object owner) throws HibernateException, SQLException {
    	Object result = null;
    	Number tmpResult = (Number) rs.getObject(names[0]);
    	if (tmpResult != null) {
    		result = getObjectValue(tmpResult.intValue());
        }
    	else if (defaultValue != null) {
    		result = getObjectValue(Integer.parseInt(defaultValue));
    	}
        return result;
    }

    /**
     * @param st (missing javadoc)
     * @param value (missing javadoc)
     * @param index (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     * @throws java.sql.SQLException (missing javadoc)
     */
    @Override
    public void nullSafeSet(PreparedStatement st, Object value, int index, SessionImplementor session) throws HibernateException, SQLException {
        if (value == null) {
            st.setNull(index, Types.INTEGER);
        } else {
            st.setInt(index, getIntValue(value));
        }
    }

    /**
     * @param value (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public Object deepCopy(Object value) throws HibernateException {
    	if (isJavaEnumClass()) {
    		return value;
    	}
        G9Enumerator result = null;
        if ( value != null ){
            result = (G9Enumerator) ObjectFactory.newObject(enumClassName);
            result.setCurrentValue(((G9Enumerator) value).getCurrentValue());
        }
        return result;
    }

    /**
     * @return (missing javadoc)
     */
    @Override
    public boolean isMutable() {
        return false;
    }

    /**
     * @param x (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public int hashCode(Object x) throws HibernateException {
        return x.hashCode();
    }

    /**
     * @param cached (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public Object assemble(Serializable cached, Object owner) throws HibernateException {
        return cached;
    }

    /**
     * @param value (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public Serializable disassemble(Object value) throws HibernateException {
        return (Serializable) value;
    }

    /**
     * @param original (missing javadoc)
     * @param target (missing javadoc)
     * @param owner (missing javadoc)
     * @return (missing javadoc)
     * @throws org.hibernate.HibernateException (missing javadoc)
     */
    @Override
    public Object replace(Object original, Object target, Object owner) throws HibernateException {
        return original;
    }

    /**
     * @param parameters (missing javadoc)
     */
    @Override
    public void setParameterValues(Properties parameters) {
        if (parameters != null) {
            if (parameters.get("class") != null) {
                this.enumClassName = (String) parameters.get("class");
            	try {
					this.enumClass = Class.forName(enumClassName);
				} catch (ClassNotFoundException e) {
					throw new HibernateException(e);
				}
            }
            if (parameters.get("default") != null) {
                this.defaultValue = (String) parameters.get("default");
                if (this.defaultValue.equals("null")) {
                	this.defaultValue = null;
                }
            }
        }
    }

    private boolean isJavaEnumClass() {
    	if (enumClass.isEnum()) {
    		return true;
    	}
    	return false;
    }

    private Enum<?> getEnumObject(int intValue) {
    	try {
			Method get = enumClass.getMethod(GET_METHOD, int.class);
			return (Enum<?>) get.invoke(null, intValue);
		} catch (Exception e) {
			throw new HibernateException(e);
		}
    }

    private Integer getEnumValue(Object enumObject) {
    	try {
			Method getValue = enumObject.getClass().getMethod(GET_VALUE_METHOD, (Class<?>[])null);
			return (Integer) getValue.invoke(enumObject, (Object[])null);
		} catch (Exception e) {
			throw new HibernateException(e);
		}
    }

    private Object getObjectValue(int intValue) {
    	Object value = null;
		if (isJavaEnumClass()) {
			value = getEnumObject(intValue);
		}
		else {
			G9Enumerator geVal = (G9Enumerator)ObjectFactory.newObject(enumClassName);
            geVal.setCurrentValue(intValue);
            value = geVal;
		}
		return value;
    }

    private int getIntValue(Object objectValue) {
    	int value;
    	if (isJavaEnumClass()) {
    		value = getEnumValue(objectValue).intValue();
    	}
    	else {
    		value = ((G9Enumerator) objectValue).getCurrentValue();
    	}
    	return value;
    }

}
