/*
 * Copyright 2013-2017 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.text.SimpleDateFormat;
import java.util.Calendar;

import no.g9.dataaccess.HibernateSessionFactory;
import no.g9.dataaccess.Session;
import no.g9.dataaccess.SessionFactory;
import no.g9.exception.G9BaseException;
import no.g9.service.JGrapeParamWrapper;
import no.g9.support.ClientContext;
import no.g9.support.ObjectSelection;
import no.g9.support.Registry;

/**
 * Common Session management methods used by the <code>JGrapeService</code>
 * implementors.
 */
public abstract class SessionManager {

    private static final SessionFactory factory = HibernateSessionFactory
            .getSessionfactory();

    private static final String OS_MAPPING = ".osmapping";

    private static final String LOG_FILENAME = "JGrapeService.logfile";

    private static final String LOG_NUMFILES = "JGrapeService.numLogfiles";

    private static final String LOG_DATEFORMAT = "JGrapeService.logfileDateFormat";

    private static String logfileDateFormat = "yyyyMMdd_HHmm";

    /** Client context parameter for log control */
    public static final String LOG_CONTROL = "LogJGrapeService";

    private static ObjectOutputStream[] logStreams;

    private static String logFile = null;

    private static int numLogFiles = 0;
    
    /** Set to true if logging is on */
    private static boolean logOn = false;

    /**
     * @return the session factory (singleton).
     */
    public static SessionFactory getSessionFactory() {
        return factory;
    }

    /**
     * Convenience method, returns a session for the given object selection and
     * client context. The session will be shared within the current thread.
     * 
     * @param objectSelection - the object selection.
     * @param ctx - the client context.
     * @return session.
     */
    public static Session getSession(final ObjectSelection objectSelection,
            final ClientContext ctx) {
    	final String dbContext = getDatabaseContextName(ctx, objectSelection.getApplicationName());
    	final String dbMapping = getDatabaseMappingName(objectSelection);
        return SessionManager.getSessionFactory().getSession(dbContext, dbMapping);
    }

    /**
     * Convenience method, returns a session for the given client context and
     * database mapping. The session will be shared within the current thread.
     * 
     * NB! This method will not use the default client context from
     * g9.config.properties, as the application name is unknown.
     * 
     * @param ctx - the client context.
     * @param dbMapping - the database mapping.
     * @return session.
     */
    public static Session getSession(final ClientContext ctx,
            final String dbMapping) {
    	final String dbContext = getDatabaseContextName(ctx, null);
        return SessionManager.getSessionFactory().getSession(dbContext, dbMapping);
    }

    /**
     * Returns the proper database context name according to the given
     * client context. Will try to load the default database context from
     * g9.config.properties if none is given in the client context. 
     * 
     * @param ctx - the client context
     * @param appName - the name of the service application
     * @return the database context name, <code>null</code> if no database context is found. 
     */
    public static String getDatabaseContextName(final ClientContext ctx, final String appName) {
    	
    	String contextName = null;
    	
    	if (ctx != null) {
    		contextName = ctx.getDatabaseContext();
    	}
        if (contextName == null && appName != null) {
            Registry reg= Registry.getRegistry();
            String propName= Registry.SRV_APP_PRE + "." + appName + "." + Registry.DEFAULT_DBC;
            contextName= reg.getG9Property(propName);
        }
    	return contextName;
    }

    /**
     * Returns the proper database mapping name according to the given
     * objectSelection
     * 
     * @param objectSelection - the object selection in question.
     * @return the database mapping name.
     */
    public static String getDatabaseMappingName(final ObjectSelection objectSelection) {

        String mappingName = null;

        final String propFileName = objectSelection.getApplicationName() + SessionManager.OS_MAPPING;
        Registry reg = Registry.getRegistry();
        String propName= null;
        propName= objectSelection.getObjectSelectionName() + "."
                + objectSelection.getCurrentRootName();
        mappingName= reg.getApplicationProperty(propFileName, propName);
        if (mappingName == null) {
            propName= objectSelection.getObjectSelectionName();
            mappingName= reg.getApplicationProperty(propFileName, propName);
            if (mappingName == null) {
                propName= Registry.SRV_APP_PRE + "." + objectSelection.getApplicationName() + "."
                        + Registry.DEFAULT_DBM;
                mappingName= reg.getG9Property(propName);
            }
        }
        return mappingName;
    }

    /**
     * Log the JGrapeService parameters.
     * 
     * The log file to use is derived from the user id in the supplied client
     * context. Which log file to use is calculated from the hashCode of the
     * user id.
     *  
     * If there is no client context, or the user id is not set, the first log
     * file is used.
     * 
     * @param logObject - the JGrapeService parameters to be logged.
     */
    protected void logParams(JGrapeParamWrapper logObject) {
        try {
            synchronized (SessionManager.class) {
                if (logOn) {
                    int logfile = 0;
                    if (logObject.getCtx() != null
                            && logObject.getCtx().getUserId() != null) {
                        logfile = logObject.getCtx().getUserId().hashCode()
                                % SessionManager.logStreams.length;
                    }
                    SessionManager.logStreams[logfile].writeObject(logObject);
                    SessionManager.logStreams[logfile].reset();
                }
            }
        } catch (IOException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * Turn service logging on or off.
     * When turned on, the log files are opened. When turned off,
     * the log files are closed.
     * 
     * @param on - true=on, false=off
     */
    protected void logControl(boolean on) {
        synchronized (SessionManager.class) {
            if (on && !SessionManager.logOn) {
                SessionManager.openLog();
                SessionManager.logOn = true;
            }
            else if (SessionManager.logOn) {
                SessionManager.logOn = false;
                SessionManager.closeLog();
            }
        }
    }

    /**
     * @return true if service logging is on, false if off.
     */
    protected boolean doLog() {
        return SessionManager.logOn;
    }
	
    /**
     * Initialize the JGrapeService logging services.
     */
    private static void initLogSession() {
        logFile= Registry.getRegistry().getG9Property(SessionManager.LOG_FILENAME);
        numLogFiles= Integer.parseInt(Registry.getRegistry().getG9Property(
                SessionManager.LOG_NUMFILES));
        logfileDateFormat= Registry.getRegistry().getG9Property(SessionManager.LOG_DATEFORMAT);
        if (logFile != null) {
            if (numLogFiles == 0) {
                numLogFiles = 1;
            }
        }
    }

    private static void openLog() {
        try {
			SessionManager.initLogSession();
            String ts = "";
            if (getLogfileDateFormat().length() > 0) {
                SimpleDateFormat df = new SimpleDateFormat(getLogfileDateFormat());
                ts = "-" + df.format(Calendar.getInstance().getTime());
            }
            logStreams = new ObjectOutputStream[numLogFiles];
            for (int i = 1; i <= numLogFiles; i++) {
                String fileName = logFile + ts + "." + i;
                FileOutputStream fos = new FileOutputStream(fileName);
                logStreams[i-1] = new ObjectOutputStream(fos);
            }
        } catch (FileNotFoundException e) {
            throw new G9BaseException(e);
        } catch (IOException e) {
            throw new G9BaseException(e);
        }
    }

    private static void closeLog() {
        try {
            for (int i = 1; i <= numLogFiles; i++) {
                SessionManager.logStreams[i-1].close();
            }
        } catch (IOException e) {
            throw new G9BaseException(e);
        }
    }

    /**
     * @return the format currently used for timestamping log files.
     */
    public static String getLogfileDateFormat() {
        return SessionManager.logfileDateFormat;
    }

    /**
     * Set the format currently used for timestamping log files.
     * @param logfileDateFormat - the new format to use.
     */
    public static void setLogfileDateFormat(String logfileDateFormat) {
        SessionManager.logfileDateFormat = logfileDateFormat;
    }

}
