/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.io.Serializable;
import java.util.List;

import no.g9.support.FindData;

/**
 * General session interface for the dataaccess package. All data access from
 * the application is handled using this interface.
 */
public interface Session {

    /**
     * Get one object from the database according to the given findData.
     * 
     * @param findData (missing javadoc)
     * @return (missing javadoc)
     */
    public Object get(FindData findData);

    /**
     * Get a list of objects from the database according to the given findData.
     * 
     * @param findData (missing javadoc)
     * @return (missing javadoc)
     */
    public List<?> getAll(FindData findData);

    /**
     * Insert a new object in the database.
     * 
     * @param object (missing javadoc)
     * @return (missing javadoc)
     */
    public Serializable insert(Object object);

    /**
     * Update the given object.
     * 
     * @param object (missing javadoc)
     */
    public void update(Object object);

    /**
     * Copy the state of the detached instance to a persistent instance with the
     * same identifier. The persistent instance will be loaded if necessary.
     * 
     * @param object the detached instance
     * @return the merged persistent instance
     */
    public Object merge(Object object);

    /**
     * Delete the given object in the database.
     * 
     * @param object (missing javadoc)
     */
    public void delete(Object object);

    /**
     * Check if the given proxy object or collection is initialized (loaded) from the database.
     * 
     * @param object the proxy object or collection to check
     */
    boolean isInitialized(Object object);

    /**
     * Initialize (load) a proxy object or collection from the database.
     * 
     * @param object (missing javadoc)
     */
    public void initialize(Object object);

    /**
     * Refresh the (cached) object from the database. Updates the object state
     * after database changes (triggers etc).
     * 
     * @param object (missing javadoc)
     */
    public void refresh(Object object);

    /**
     * Close this database session.
     */
    public void close();

    /**
     * Flush the session cache changes to the database.
     */
    public void flush();

    /**
     * Start a new transaction.
     */
    public void beginTransaction();

    /**
     * Commit the current transaction.
     */
    public void commitTransaction();

    /**
     * Rollback the current transaction.
     */
    public void rollbackTransaction();

    /**
     * Return true if the session has a transaction that can be rolled back.
     * 
     * @return (missing javadoc)
     */
    public boolean canRollbackTransaction();

    /**
     * Return true if the session has an active transaction.
     * 
     * @return (missing javadoc)
     */
    public boolean hasActiveTransaction();

    /**
     * Return true if the domain class uses lock flags (optimistic
     * locking) in the database.
     * 
     *  
     * @param domainClass the domain class to check.
     * @return true if lock flag is used, false otherwise.
     */
    public boolean hasLockFlag(Class<?> domainClass);

    /**
     * Build a query string for the given class and list of attribute names.
     * 
     * @param clazz (missing javadoc)
     * @param keyAttributeNames (missing javadoc)
     * @return (missing javadoc)
     */
    public String buildQueryString(Class<?> clazz, List<String> keyAttributeNames);

    /**
     * Build a list of parameter values for the given key object and list of
     * attribute names.
     * 
     * @param key (missing javadoc)
     * @param keyAttributeNames (missing javadoc)
     * @return (missing javadoc)
     */
    public List<Object> buildQueryParameterValueList(Object key, List<String> keyAttributeNames);
    
    /**
     * Build a list of "equal" criterions for the given key object and list of
     * attribute names.
     * 
     * @param key (missing javadoc)
     * @param keyAttributeNames (missing javadoc)
     * @return the list of criterions.
     */
    public List<Object> buildCriterionList(Object key, List<String> keyAttributeNames);
    
}
