/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess.usertype;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Properties;

/**
 * A type supporting date as integer in the database, and LocalDate in the domain code.
 */
public class IntLocalDate implements ParameterizedType, Serializable {

    private String dateFormat = "yyyyMMdd";

    private LocalDate defaultValue = null;

    @Override
    public synchronized void setParameterValues(Properties parameters) {
        if (parameters != null) {
            if (parameters.get("default") != null) {
                String defaultVal = (String) parameters.get("default");
                if (!defaultVal.equals("null")) {
                    this.defaultValue = fromInt(Integer.parseInt(defaultVal));
                }
            }
            if (parameters.get("format") != null) {
                this.dateFormat = (String) parameters.get("format");
            }
        }
    }

    /**
     * Conversion from int to LocalDate.
     *
     * @param value the integer value to convert
     * @return the LocalDate value
     */
    public synchronized LocalDate fromInt(int value) {
    	LocalDate result = null;
        if (value != 0) {
            try {
                result = LocalDate.parse(Integer.toString(value), DateTimeFormatter.ofPattern(dateFormat));
            } catch (DateTimeParseException pe) {
                throw new RuntimeException(pe);
            }
        }
        return result;
    }

    /**
     * Conversion from LocalDate to int.
     *
     * @param value the date value to convert
     * @return the integer value
     */
    public synchronized Integer fromLocalDate(LocalDate value) {
        if (value != null) {
            return new Integer(value.format(DateTimeFormatter.ofPattern(dateFormat)));
        }
        return null;
    }

    /**
     * @return the defaultValue
     */
    public LocalDate getDefaultValue() {
        return (defaultValue != null) ? LocalDate.from(defaultValue) : null;
    }

}
