/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape;

import java.util.List;

import no.g9.dataaccess.Session;
import no.g9.service.enumerator.ENavigation;
import no.g9.support.ClientContext;
import no.g9.support.ObjectFactory;
import no.g9.support.ObjectSelection;
import no.g9.support.Registry;

/**
 * This class is the common superclass for the classes representing the object
 * selection in the service layer.
 * <p>
 * The implementaion of each of the methods in the Manager iterface follow the same pattern
 * <ul>
 *     <li>Create a new role manager for current root in the object selection. </li>
 *     <li>Locate the role manager for the target role.</li>
 *     <li>Call this role managers method to do the wanted action.</li>
 * </ul>
 */
public abstract class OsManager implements Manager {

    private static final String OS_MANAGER = ".osmanager";
    private static final String OS_MANAGER_SUFFIX = "OsManager";

    /** (missing javadoc) */
    Role targetRole = null;

    /**
     * Find a single object.
     *
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return Object found. This could be a list containing the real object.
     */
    @Override
    public Object find(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.find(objectSelection,ctx,session);
           }
        }
        return null;
    }
   
    /**
     * Find multiple objects.
     *
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public List<?> findAll(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                List<?>list = targetRole.findAll(objectSelection,ctx,session);
                targetRole.setParent(null);
                return list;
            }
        }
        return null;
    }
   
    /**
     * Save an object. A save is either an update or an insert.
     * 
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object save(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                 return targetRole.save(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * Insert a new object.
     * 
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object insert(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.insert(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * Update an existing object.
     * 
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object update(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.update(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * delete an existing object and the objects depending on it.
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object delete(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.delete(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * Connect an object to it's parent.
     * 
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object connect(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.connect(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * Disconnect an object from it's parent.
     * 
     * @param objectSelection The object selection with data from the client.
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object disconnect(ObjectSelection objectSelection, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.disconnect(objectSelection,ctx,session);
            }
        }
        return null;
    }
   
    /**
     * @param objectSelection The object selection with data from the client.
     * @param navigation (missing javadoc)
     * @param ctx The <code>ClientContext</code> desciption.
     * @param session The <code>DataAccess</code> <code>Session</code>.
     * @return (missing javadoc)
     */
    @Override
    public Object get(ObjectSelection objectSelection, ENavigation navigation, ClientContext ctx, Session session) {
        Role rootRole = newManager(objectSelection.getApplicationName(),objectSelection.getCurrentRootName());
        if (rootRole != null) {
            rootRole.setCurrentObjectSelection(objectSelection);
            targetRole = rootRole.findRole(objectSelection.getTargetRoleName());
            if (targetRole != null) {
                return targetRole.get(objectSelection,navigation,ctx,session);
            }
        }
        return null;
    }

    @Override
    @Deprecated
    public void cleanupResult(Object rootResult) {
        if (targetRole != null) {
            targetRole.cleanupResult(rootResult);
        }
    }

    @Override
    @Deprecated
    public void cleanupResult(Object rootResult, ClientContext ctx) {
        if (targetRole != null) {
            targetRole.cleanupResult(rootResult, ctx);
        }
    }

    @Override
    public void cleanupResult(Object rootResult, ClientContext ctx, Session session) {
        if (targetRole != null) {
            targetRole.cleanupResult(rootResult, ctx, session);
        }
    }

    @Override
    public void cleanupResultAssociations(Object rootResult, ClientContext ctx, Session session) {
        if (targetRole != null) {
            targetRole.cleanupResultAssociations(rootResult, ctx, session);
        }
    }

    /**
     * Returns the proper manager accordning to objectSelection
     * 
     * @param objectSelection (missing javadoc)
     * @return (missing javadoc)
     */
    public static Manager getManager(ObjectSelection objectSelection) {
        String managerName = getManagerName(objectSelection);
        return (Manager) ObjectFactory.newObject(managerName);
    }

    private static String getManagerName(ObjectSelection objectSelection) {

        String managerName = null;

        final String propFileName = objectSelection.getApplicationName() + OS_MANAGER;
        Registry reg = Registry.getRegistry();
        String propName = objectSelection.getObjectSelectionName();

        managerName= reg.getApplicationProperty(propFileName, propName);
        if (managerName == null) {
            String objectSelectionName= objectSelection.getObjectSelectionName().substring(0, 1)
                    .toUpperCase()
                    + objectSelection.getObjectSelectionName().substring(1);
            propName= Registry.SRV_APP_PRE + "." + objectSelection.getApplicationName() + "."
                    + Registry.DEFAULT_SRV;
            managerName= reg.getG9Property(propName) + "." + objectSelectionName.toLowerCase()
                    + "." + objectSelectionName + OS_MANAGER_SUFFIX;
        }
        return managerName;
    }

    /**
     * Generate fully qualified class name from role- and package name.
     *
     * @param roleName OS node name
     * @param pkgName Package
     * @return "package.RoleManager"
     */
    public static String createManagerClassName(String roleName, String pkgName) {
        return pkgName + "."
                       +(roleName.substring(0,1).toUpperCase() 
                       + roleName.substring(1)) 
                       + "Manager";
    }

/**
     * Create a new manager for an object selection role
     * 
     * @param applicationName (missing javadoc)
     * @param roleName (missing javadoc)
     * @return The create RoleManager
     */
    public abstract Role newManager(String applicationName, String roleName);
}
