/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape;

import java.util.LinkedList;
import java.util.Iterator;
import no.g9.support.ObjectSelection;
import no.g9.support.ClientContext;
import no.g9.dataaccess.Session;
import no.g9.service.enumerator.ENavigation;
import java.util.List;

/**
 * This is the common superclass to all object selection role managers. An
 * object selection role manager represent one role node in the object
 * selection.
 */
public abstract class Role implements Manager {

    /**
     * Name of the object selection role
     */
    private String name;

    /**
     * Names of all attributes excluded from the object selection role.
     */
    private List<String> excludedAttributes;

    /**
     * Names of all attributes included in the object selection role.
     */
    private List<String> includedAttributes;

    /**
     * Parent role manager Name of the object selection role
     */
    private Role parent;

    /**
     * List of down related role managers, i.e roles connected to this role via
     * an association where this role is the owner.
     */
    private List<Role> downRelated;

    /**
     * List of up related role managers, i.e roles connected to this role via an
     * association where this role is the member.
     */
    private List<Role> upRelated;

    /**
     * The Key class holds information for one unique key. A key is either an
     * attribute or a group.
     */
    protected static class Key {

        /** Name of key */
        String keyName;

        /** Is all attributes in this key included in the object selection? */
        boolean inOs;

        /** Is this the primary key for the class? */
        boolean isPrimaryKey;

        /** List of all attributes contained in this key. */
        List<String> attributes;

        /**
         * Add a new attribute to this key.
         * 
         * @param attribute Name of attribute to add.
         */
        public void addAttribute(String attribute) {
            attributes.add(attribute);
        }
    }

    /**
     * List of all unique keys for this object selection role
     */
    private List<Key> keys;

    /**
     * Current ObjectSelection
     */
    private ObjectSelection currentObjectSelection = null;

    /**
     * Add a new unique key to this role managers List of keys.
     * 
     * @param keyName the name of the key
     * @param inOs the inOs property
     * @param isPrimaryKey the isPrimaryKey property
     * @return the new unique key
     */
    protected Key addKey(String keyName, boolean inOs, boolean isPrimaryKey) {
        Key key = new Key();
        key.keyName = keyName;
        key.inOs = inOs;
        key.isPrimaryKey = isPrimaryKey;
        key.attributes = new LinkedList<String>();
        keys.add(key);
        return key;
    }

    /**
     * Get list of attribute names for a unique key
     * 
     * @param keyName is the name of the key
     * @return The list of attribute names. Returns null if keyName is unknown
     */
    protected List<String> getKeyAttributes(String keyName) {
        List<String> result = null;
        if (keyName != null && keys != null) {
            Key key;
            Iterator<Key> iter = keys.iterator();
            while (iter.hasNext()) {
                key = iter.next();
                if (keyName.equals(key.keyName)) {
                    result = key.attributes;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Get the name of primary key if the object selection role has a primary
     * key and all attributes in that key are included in the object selection.
     * 
     * @return Either Name of primary key or null if not all attributes are
     *         included in the object selection.
     */
    protected String getPrimaryKey() {
        String result = null;
        if (keys != null) {
            Key key;
            Iterator<Key> iter = keys.iterator();
            while (iter.hasNext()) {
                key = iter.next();
                if (key.inOs && key.isPrimaryKey) {
                    result = key.keyName;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Get the name of main key if this object selection role has a main key. A
     * main key is the first unique key where all attributes are included in the
     * object selection.
     * 
     * @return Name of main key
     */
    protected String getMainKey() {
        String result = null;
        if (keys != null) {
            Key key;
            Iterator<Key> iter = keys.iterator();
            while (iter.hasNext()) {
                key = iter.next();
                if (key.inOs) {
                    result = key.keyName;
                    break;
                }
            }
        }
        return result;
    }

    /**
     * Default constructor.
     */
    public Role() {
        super();
    }

    /**
     * Access method for the name property.
     * 
     * @return The current value of the name property
     */
    protected String getName() {
        return name;
    }

    /**
     * Sets the value of the name property.
     * 
     * @param aName the new value of the name property
     */
    protected void setName(String aName) {
        name = aName;
    }

    /**
     * Access method for the excludedAttributes property.
     * 
     * @return the current value of the excludedAttributes property
     */
    protected List<String> getExcludedAttributes() {
        return excludedAttributes;
    }

    /**
     * Sets the value of the excludedAttributes property.
     * 
     * @param aExcludedAttributes the new value of the excludedAttributes
     *            property
     */
    protected void setExcludedAttributes(List<String> aExcludedAttributes) {
        excludedAttributes = aExcludedAttributes;
    }

    /**
     * Access method for the includedAttributes property.
     * 
     * @return the current value of the includedAttributes property
     */
    protected List<String> getIncludedAttributes() {
        return includedAttributes;
    }

    /**
     * Sets the value of the includedAttributes property.
     * 
     * @param aIncludedAttributes the new value of the includedAttributes
     *            property
     */
    protected void setIncludedAttributes(List<String> aIncludedAttributes) {
        includedAttributes = aIncludedAttributes;
    }

    /**
     * Access method for the downRelated property.
     * 
     * @return the current value of the downRelated property
     */
    protected List<Role> getDownRelated() {
        return downRelated;
    }

    /**
     * Sets the value of the downRelated property.
     * 
     * @param aDownRelated the new value of the downRelated property
     */
    protected void setDownRelated(List<Role> aDownRelated) {
        downRelated = aDownRelated;
    }

    /**
     * Access method for the upRelated property.
     * 
     * @return the current value of the upRelated property
     */
    protected List<Role> getUpRelated() {
        return upRelated;
    }

    /**
     * Sets the value of the upRelated property manager.
     * 
     * @param aUpRelated the new value of the upRelated property
     */
    protected void setUpRelated(List<Role> aUpRelated) {
        upRelated = aUpRelated;
    }


    /**
     * Access method for the currentObjectSelection property.
     * 
     * @return the current value of the currentObjectSelection property
     */
    public ObjectSelection getCurrentObjectSelection() {
        return currentObjectSelection;
    }

    /**
     * Sets the value of the currentObjectSelection property.
     * 
     * @param anObjectSelection the new value of the currentObjectSelection
     *            property
     */
    public void setCurrentObjectSelection(ObjectSelection anObjectSelection) {
        currentObjectSelection = anObjectSelection;
    }

    /**
     * Access method for the parent property.
     * 
     * @return the current value of the parent property
     */
    protected Role getParent() {
        return parent;
    }

    /**
     * Sets the value of the parent property.
     * 
     * @param aParent the new value of the parent property
     */
    protected void setParent(Role aParent) {
        parent = aParent;
    }

    /**
     * Initialize an object selection role manager.
     * 
     * @param applicationName Name of the application
     * @param roleName Name of the role
     * @param parentRole Parent role to this role in the object selection
     */
    public void initializeRole(String applicationName, String roleName, Role parentRole) {
        this.name = roleName;
        this.parent = parentRole;
        downRelated = new LinkedList<Role>();
        upRelated = new LinkedList<Role>();
        excludedAttributes = new LinkedList<String>();
        includedAttributes = new LinkedList<String>();
        keys = new LinkedList<Key>();
    }

    /**
     * Find an object selection role manager given name.
     * 
     * @param roleName the name of the object selection role.
     * @return no.g9.service.Role
     */
    public Role findRole(String roleName) {
        Role role = null;
        if (roleName.equals(this.name)) {
            role = this;
        }
        if (downRelated != null) {
            Iterator<Role> drIt = downRelated.listIterator();
            while (role == null && drIt.hasNext()) {
                role = drIt.next().findRole(roleName);
            }
        }
        if (upRelated != null) {
            Iterator<Role> urIt = upRelated.listIterator();
            while (role == null && urIt.hasNext()) {
                role = urIt.next().findRole(roleName);
            }
        }
        return role;
    }

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object find(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract List<?> findAll(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object save(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object insert(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object update(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object delete(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object connect(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object disconnect(ObjectSelection objectSelection,
            ClientContext ctx, Session session);

    /**
     * Part of the Manager interface.
     * 
     * @param objectSelection (missing javadoc)
     * @param navigation (missing javadoc)
     * @param ctx (missing javadoc)
     * @param session (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public abstract Object get(ObjectSelection objectSelection,
            ENavigation navigation, ClientContext ctx, Session session);

    @Deprecated
    @Override
    public void cleanupResult(Object rootResult) {
	    // Default implementaion is empty. 
    }

    @Deprecated
    @Override
    public void cleanupResult(Object rootResult, ClientContext ctx) {
	    // If called, i.e. if not implemented in the generated sub-class, the real role implements the deprecated method
	    cleanupResult(rootResult);
    }

    @Override
    public void cleanupResult(Object rootResult, ClientContext ctx, Session session) {
	    // If called, i.e. if not implemented in the generated sub-class, the real role implements the deprecated method
	    cleanupResult(rootResult, ctx);
    }

    @Override
    public void cleanupResultAssociations(Object rootResult, ClientContext ctx, Session session) {
	    // Default implementation is empty. Generating this cleanup is optional
    }

    /**
     * Get the root object selection role manager.
     * 
     * @return (missing javadoc)
     */
    protected Role getRootRole() {
        if (parent == null) {
            return this;
        }
        return parent.getRootRole();
    }

}
