/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape.trigger;

import java.util.ArrayList;
import java.util.List;

import no.esito.log.Logger;

import no.g9.jgrape.trigger.JGrapeTrigger.TriggerType;
import static no.g9.jgrape.trigger.JGrapeTrigger.TriggerType.*;

/**
 * Utility class used for filtering list of JGrapeTriggers.
 */
final class TriggerFilter {

    private static Logger log = Logger.getLogger(TriggerFilter.class);
    
    /** No instances of this class. */
    private TriggerFilter() {
        super();
    }

    /**
     * Filters the list based on the specified filter.
     * @param list the list to filter
     * @param filter the filter to use
     * @return a new list containing only values that passed the filter.
     */
    static List<JGrapeTrigger<?>> filter(List<JGrapeTrigger<?>> list,
        TriggerType filter) {
        log.trace("Filter: " + filter + " list " + list); 
        List<JGrapeTrigger<?>> filteredList = new ArrayList<JGrapeTrigger<?>>(
            list.size());

        for (JGrapeTrigger<?> jGrapeTrigger : list) {
            TriggerType actual = jGrapeTrigger.getTriggerType();
            if (passFilter(actual, filter)) {
                filteredList.add(jGrapeTrigger);
            }
        }

        log.trace("Done filtering : " + filter + " result " + filteredList); 
        
        return filteredList;

    }

    /**
     * Test the actual value against filter value and returns <code>true</code>
     * if actual value passes filter.
     * 
     * @param actual the actual value
     * @param filter the filter value
     * @return <code>true</code> if actual value mathces filter.
     */
    private static boolean passFilter(TriggerType actual, TriggerType filter) {

        switch (filter) {
            case DELETE:
                return actual == DELETE || actual == ALL;
            case INSERT:
                return actual == INSERT || actual == ALL;
            case UPDATE:
                return actual == UPDATE || actual == ALL;
            case SAVE:
                return actual == SAVE || actual == INSERT || actual == ALL || actual == UPDATE;
            default: // ALL
                return true;

        }

    }

}
