/*
 * Copyright 2013-2018 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape.tx;

import no.g9.dataaccess.HibernateSessionFactory;
import no.g9.dataaccess.Session;
import no.g9.dataaccess.SessionFactory;
import no.g9.jgrape.SessionManager;
import no.g9.message.*;
import no.g9.support.ClientContext;

/**
 * Common Session management for TxService implementors.
 */
public class TxSessionManager extends SessionManager {

    private Session oldSession;

    /**
     * Get a <code>Session</code> for the given <code>contextName</code>
     * and <code>mappingName</code>.
     * Keep the previously cached <code>Session</code> if using Hibernate.
     *
     * @param contextName the database context name
     * @param mappingName the database mapping name
     * @return the session
     */
    protected Session getSession(final String contextName, final String mappingName) {
        final SessionFactory sf = SessionManager.getSessionFactory();
        if (sf instanceof HibernateSessionFactory) {
            this.oldSession = ((HibernateSessionFactory) sf).switchSession(contextName, mappingName, null);
        }
        return sf.getSession(contextName, mappingName);
    }

    /**
     * Reset the cached <code>Session</code> if using Hibernate.
     *
     * @param contextName the database context name
     * @param mappingName the database mapping name
     */
    protected void resetSession(final String contextName, final String mappingName) {
        final SessionFactory sf = SessionManager.getSessionFactory();
        if (sf instanceof HibernateSessionFactory) {
            ((HibernateSessionFactory) sf).switchSession(contextName, mappingName, this.oldSession);
        }
    }

    /**
     * Method to execute the <code>command</code>.
     *
     * @param session Session context
     * @param clientContext Client Context
     * @param command The command to run
     */
    protected void runTx(final Session session,
                         final ClientContext clientContext,
                         final TxCommand command) {
        try {
            session.beginTransaction();
            command.execute(clientContext, session);
            session.commitTransaction();
        }
        catch (final RuntimeException e) {
            // No handling of Error - they are just propagated
            if (session.canRollbackTransaction()) {
                try {
                    session.rollbackTransaction();
                }
                catch (final RuntimeException bad) {
                    // Must be able to close session anyway
                    logExceptionQuietly(bad);
                }
            }
            throw e;
        }
        finally {
            try {
                SessionManager.getSessionFactory().closeSession(session);
            }
            catch (final Exception bad) {
                logExceptionQuietly(bad);
            }
        }
    }

    /**
     * Log errors occurring during command execution.
     *
     * @param bad Exception
     */
    protected void logExceptionQuietly(final Exception bad) {
        final Object[] args= { bad.getMessage() };
        Message msg = MessageSystem.getMessageFactory().getMessage(CRuntimeMsg.DB_ORM_ERROR, args);
        msg.setException(bad);
        MessageSystem.getMessageDispatcher(MessageSystem.NO_INTERACTION).dispatch(msg);
    }

}
