/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.dataaccess;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

/**
 * This class holds a list of Hibernate properties and property files.
 * Properties may be added one by one, or a new set of properties may be set.
 * Files may be added one by one, or a new list of files may be set.
 *
 * The list of properties and property files can be built into a new
 * set of properties.
 */
public class HibernateProperties {

    private List<String> propertyFiles = new LinkedList<String>();

    private Properties properties = new Properties();

    /**
     * @return the list of property files
     */
    public List<String> getPropertyFiles() {
        return propertyFiles;
    }

    /**
     * Replace the list of property files.
     *
     * @param propertyFiles the new list of property files to use
     */
    public void setPropertyFiles(List<String> propertyFiles) {
        this.propertyFiles = propertyFiles;
    }

    /**
     * Add a new property file to the list of property files.
     *
     * @param file the new property file.
     */
    public void addPropertyFile(String file) {
        propertyFiles.add(file);
    }

    /**
     * @return the set of properties
     */
    public Properties getProperties() {
        return properties;
    }

    /**
     * Replace the set of properties.
     *
     * @param properties the new set of properties to use
     */
    public void setProperties(Properties properties) {
        this.properties = properties;
    }

    /**
     * Add a new property.
     *
     * @param key the property key
     * @param value the property value
     */
    public void addProperty(String key, String value) {
        properties.put(key, value);
    }

    /**
     * Build a set of properties from the local properties and
     * property files.
     *
     * @return a new properties object with all the known properties
     */
    public Properties buildProperties() {
        Properties props = new Properties();
        addPropertiesFromFiles(props);
        props.putAll(getProperties());
        return props;
    }

    private void addPropertiesFromFiles(Properties props) {
        for (String file : propertyFiles) {
            Properties nProps = new Properties();
            try {
                InputStream stream = getClass().getResourceAsStream(file);
                if (stream == null) {
                    throw new IllegalArgumentException(file);
                }
                nProps.load(stream);
            }
            catch (IOException e) {
                throw new IllegalArgumentException(e);
            }
            props.putAll(nProps);
        }
    }

}
