/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape;

import java.util.List;

import no.esito.log.Logger;

import no.g9.dataaccess.Session;
import no.g9.service.JGrapeService;
import no.g9.service.JGrapeParamWrapper;
import no.g9.service.enumerator.ENavigation;
import no.g9.support.ClientContext;
import no.g9.support.FindData;
import no.g9.support.ObjectSelection;

/**
 * An implementation of the <code>JGrapeService</code> interface using service
 * managers in the same (local) JVM.
 */
public class JGrapeLocal extends SessionManager implements JGrapeService {

    private Logger logger = no.esito.log.Logger.getLogger(JGrapeService.class);

    /**
     * Default empty constructor.
     */
    public JGrapeLocal() {
        super();
        logger.debug("Finished constructing JGrapeLocal.");
    }
    
    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object find(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("FIND", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            result = manager.find(objectSelection, ctx,
                    session);
        } catch (RuntimeException e) {
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }

        manager.cleanupResultAssociations(result, ctx, session);
        return result;
    }

    /**
     * @param findData (missing javadoc)
     * @param ctx (missing javadoc)
     * @param dbMapping (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object find(FindData findData, ClientContext ctx, String dbMapping) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("FIND", ctx, findData, dbMapping);
        logger.debug(logObject);
        Object result = null;
        Session session = getSession(ctx, dbMapping);
        try {
            DataService dataService = new DataServiceImpl();
            result = dataService.find(findData, session);
        } catch (RuntimeException e) {
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public List<?> findAll(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("FINDALL", ctx, objectSelection);
        logger.debug(logObject);
        List<?> result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            result = manager.findAll(
                    objectSelection, ctx, session);
        } catch (RuntimeException e) {
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResultAssociations(result, ctx, session);
        return result;
    }

    /**
     * @param findData (missing javadoc)
     * @param ctx (missing javadoc)
     * @param dbMapping (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public List<?> findAll(FindData findData, ClientContext ctx, String dbMapping) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("FINDALL", ctx, findData, dbMapping);
        logger.debug(logObject);
        List<?> result = null;
        Session session = getSession(ctx, dbMapping);
        try {
            DataService dataService = new DataServiceImpl();
            result = dataService.findAll(findData, session);
        } catch (RuntimeException e) {
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object save(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("SAVE", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.save(objectSelection, ctx, session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object insert(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("INSERT", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.insert(objectSelection, ctx, session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object update(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("UPDATE", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.update(objectSelection, ctx, session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object delete(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("DELETE", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.delete(objectSelection, ctx,
                    session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object connect(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("CONNECT", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.connect(objectSelection, ctx,
                    session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object disconnect(ObjectSelection objectSelection, ClientContext ctx) {
        JGrapeParamWrapper logObject = new JGrapeParamWrapper("DISCONNECT", ctx, objectSelection);
        logger.debug(logObject);
        Object result = null;
        Manager manager = OsManager.getManager(objectSelection);
        Session session = getSession(objectSelection, ctx);
        try {
            session.beginTransaction();
            result = manager.disconnect(objectSelection,
                    ctx, session);
            session.commitTransaction();
        } catch (RuntimeException e) {
            if (session.canRollbackTransaction()) {
                session.rollbackTransaction();
            }
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        manager.cleanupResult(result, ctx, session);
        return result;
    }

    /**
     * @param objectSelection (missing javadoc)
     * @param navigation (missing javadoc)
     * @param ctx (missing javadoc)
     * @return (missing javadoc)
     */
    @Override
    public Object get(ObjectSelection objectSelection, ENavigation navigation,
            ClientContext ctx) {
        Object result = null;
        Session session = getSession(objectSelection, ctx);
        try {
            // Not Impl'd result =
            // getManager(objectSelection).get(objectSelection, navigation, ctx,
            // session);
        } catch (RuntimeException e) {
            throw e;
        } finally {
            getSessionFactory().closeSession(session);
        }
        return result;
    }

}
