/*
 * Copyright 2013-2020 Esito AS
 * Licensed under the g9 Runtime License Agreement (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://download.esito.no/licenses/g9runtimelicense.html
 */
package no.g9.jgrape.trigger;

import java.util.List;

import no.esito.log.Logger;
import no.esito.util.ServiceLoader;
import no.g9.dataaccess.Session;
import no.g9.jgrape.trigger.JGrapeTrigger.TriggerType;
import no.g9.support.ClientContext;

/**
 * Utility class used to invoke the triggers for the specified domain object.
 */
public final class TriggerInvoker {

    private static Logger log = Logger.getLogger(TriggerInvoker.class);

    /** Don't create any instances of this class. */
    private TriggerInvoker() {
        // empty.
    }

    /**
     * Invoke the triggers for specified domain object.
     * 
     * @param <T> The domain object type.
     * @param domainObject the domain object to invoke triggers for.
     * @param type the trigger type to invoke.
     * @param isTargetNode <code>true</code> if the domain object's role this is
     *            the target node of the ongoing action.
     * @param ctx the current client context.
     * @param session the current session.
     * @throws IllegalArgumentException if either <code>domainObject</code> or
     *             <code>type</code> is <code>null</code>.
     */
    @SuppressWarnings("unchecked")
    public final static <T> void invoke(T domainObject, TriggerType type,
        boolean isTargetNode, ClientContext ctx, Session session) {
        log.debug("Invoking " + type + "triggers for " + domainObject);

        // Sanity checks...
        nullCheck(domainObject, type);
        
        TriggerProvider provider = ServiceLoader.getService(TriggerProvider.class);
        List<JGrapeTrigger<?>> trigger = provider.getTrigger(domainObject
            .getClass(), type);
        for (JGrapeTrigger<?> unTyped : trigger) {
            JGrapeTrigger<T> jGrapeTrigger = (JGrapeTrigger<T>) unTyped;
            jGrapeTrigger.trigger(domainObject, type, isTargetNode, ctx,
                session);
        }

    }

    private static void nullCheck(Object domainObject, TriggerType type)
        throws IllegalArgumentException {
        String errorMsg = null;
        if (domainObject == null) {
            errorMsg = "Domain object cannot be <code>null</code>";
        } else if (type == null) {
            errorMsg = "Trigger type cannot be <code>null</code>";
        }
        if (domainObject == null || type == null) {
            throw new IllegalArgumentException(errorMsg);
        }
    }

}
